/***************************************************************************
    file	         : kb_copyexec.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<qapplication.h>


#include	"kb_dom.h"

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_dblink.h"

#include	"kb_attrdict.h"
#include	"kb_paramsetdlg.h"

#include	"kb_copyexec.h"
#include	"kb_copyfile.h"
#include	"kb_copytable.h"
#include	"kb_copysql.h"
#include	"kb_copyxml.h"
#include	"kb_copyquery.h"

#if		! __KB_RUNTIME
#include	"kb_paramdlg.h"
#endif


/*  KBCopyExec								*/
/*  KBCopyExec	: Constructor for copy executor				*/
/*  srce	: KBCopyBase *	: Source copier				*/
/*  dest	: KBCopyBase *	: Destination copier			*/
/*  (returns)	: KBCopyExec	:					*/

KBCopyExec::KBCopyExec
	(	KBCopyBase	*srce,
		KBCopyBase	*dest
	)
	:
	m_srce	(srce),
	m_dest	(dest)
{
	m_paramDict	= 0 ;
	m_progress	= 0 ;
}

/*  KBCopyExec								*/
/*  KBCopyExec	: Destructor for copy executor				*/
/*  (returns)	:		:					*/

KBCopyExec::~KBCopyExec ()
{
	DELOBJ	(m_paramDict) ;
	DELOBJ	(m_progress ) ;
}

/*  KBCopyExec								*/
/*  execute	: Execute copy						*/
/*  report	: QString &	: Success completion report		*/
/*  pError	: KBError &	: Error return				*/
/*  nRows	: int &		: Number of rows copied			*/
/*  _paramDict	:		: Caller parameters			*/
/*  _paramSet	:		: Copied parameter set			*/
/*  showProg	: bool		: Show progress dialog			*/
/*  (returns)	: bool		: Success				*/

bool	KBCopyExec::execute
	(	QString			&report,
		KBError			&pError,
		int			&nRows,
		const QDict<QString>	&paramDict,
		QDict<KBParamSet>	&paramSet,
		bool			showProg
	)
{
	int		sCols	;
	int		dCols	;
	int		nCols	;

	bool		rc	= false ;
	KBValue		*values	= 0	;

	DELOBJ	(m_paramDict) ;
	m_paramDict = new KBAttrDict (paramDict) ;

	if (paramSet.count() > 0)
	{
		bool	ok	;
		KBParamSetDlg	pDlg (TR("Set Parameters"), paramSet, 0, pError, ok) ;

		if (!ok) return false ;

		if (!pDlg.exec ())
		{
			pError	= KBError
				  (	KBError::Warning,
					TR("User cancelled parameter dialog"),
					QString::null,
					__ERRLOCN
				  )	;
			return	false	;
		}

		QDictIterator<KBParamSet> iter (paramSet) ;
		KBParamSet		  *pSet		   ;

		while ((pSet = iter.current()) != 0)
		{
			m_paramDict->replace (iter.currentKey(), new QString(pSet->m_value)) ;
			iter += 1 ;
		}
	}

	if (!m_srce->prepare(*m_paramDict, m_dest))
	{	pError	= m_srce->lastError () ;
		goto	failed	;
	}
	if (!m_dest->prepare(*m_paramDict, m_srce))
	{	pError	= m_dest->lastError () ;
		goto	failed	;
	}

	/* Get the numbers of source and destination columns. The	*/
	/* returned values are zero if unknown (ie., delimited files)	*/
	/* and negative on error.					*/
	sCols	= m_srce->getNumCols () ;
	dCols	= m_dest->getNumCols () ;

	if (sCols < 0)
	{	pError	= m_srce->lastError () ;
		goto	failed	;
	}
	if (dCols < 0)
	{	pError	= m_srce->lastError () ;
		goto	failed	;
	}
	if ((sCols != 0) && (dCols != 0) && (sCols != dCols))
	{	pError	= KBError
			  (	KBError::Error,
				TR("Mismatched number of rows in copy"),
				QString(TR("Source: %1, Destination %2"))
					.arg(sCols)
					.arg(dCols),
				__ERRLOCN
			  )	;
		goto	failed	;
	}


	/* Note the number of columns to be copied and allocate space.	*/
	/* Note that we allocate one extra value to allow the		*/
	/* destination to possibly add a primary key value.		*/
	/* NOTE: The only case where we don't know is copying delimited	*/
	/*	 file to delimited file, in which case allocate a nice	*/
	/*	 large amount.						*/
	nCols	= sCols > 0 ? sCols : dCols > 0 ? dCols : 500 ;
	values	= new KBValue[nCols + 1]  ;


	/* Initial dummy put, used to do stuff like write file headers	*/
	/* or delete extant table data.					*/
	if (!m_dest->putRow (0, 0))
	{	pError	= m_dest->lastError () ;
		goto	failed ;
	}


	if (showProg)
	{
		m_progress = new TKProgress
			     (		TR("Copying ...."),
					TR("Copied"),
					TR("records"),
					false
			     )		;
		m_progress->show()	;
	}

	/* This is is. The actual copy loop runs inside the source	*/
	/* copier, so that it can handle "difficult" cases like a SAX-	*/
	/* based XML source parser.					*/
	if ((nRows = m_srce->execute (m_dest, values, nCols, this)) < 0)
	{
		pError	= m_srce->lastError() ;
		goto	failed	;

	}

	rc	= true	;

	failed	:

		DELOBJ	(m_progress) ;

		if (!m_srce->finish (report))
		{	if (rc) pError = m_srce->lastError() ;
			rc = false ;
		}
		if (!m_dest->finish (report))
		{	if (rc) pError = m_srce->lastError() ;
			rc = false ;
		}

		if (values != 0) delete []values ;
		return	rc ;
}

/*  KBCopyExec								*/
/*  loadSpecification							*/
/*		: Load a copier source or destination specification	*/
/*  location	: KBLocation &	: Document location			*/
/*  root	: QDomElement &	: DOM element at root of specification	*/
/*  srce	: bool		: True for source, else destination	*/
/*  pError	: KBError &	: Error return				*/
/*  (returns)	: KBCopyBase *	: Copied or null on error		*/

KBCopyBase
	*loadSpecification
	(	KBLocation	&location,
		QDomElement	&root,
		bool		srce,
		KBError		&pError
	)
{
	/* Locate the DOM element appropriate to this being either the	*/
	/* source or the destination.					*/
	QDomElement	copy	= root.namedItem(srce ? "srce" : "dest").toElement () ;

	if (copy.isNull ())
	{
		pError	= KBError
			  (	KBError::Error,
				QString(TR("Document lacks %1 part")).arg(srce ? "source" : "destination"),
				QString::null,
				__ERRLOCN
			  )	;
		return	false	;
	}

	/* Get the tag, which specifies which of the possible sources	*/
	/* (or destinations) is to be used, then instantiate the	*/
	/* appropriate copier. Report an error if the tag does not	*/
	/* match.							*/
	QString	   tag	 = copy.attribute("tag") ;
	KBCopyBase *base = 0 ;

	if	(tag == "file" ) base = new KBCopyFile  (srce, location) ;
	else if (tag == "table") base = new KBCopyTable (srce, location) ;
	else if (tag == "sql"  ) base = new KBCopySQL   (srce, location) ;
	else if (tag == "xml"  ) base = new KBCopyXML   (srce, location) ;
	else if (tag == "query") base = new KBCopyQuery (srce, location) ;

	if (base == 0)
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Unrecognised tag in copied document"),
			  	QString(TR("%1: tag %2"))
			  		.arg(srce ? TR("Source") : TR("Destination"))
			  		.arg(tag),
			  	__ERRLOCN
			  )	;
		return	0 ;
	}

	/* Set the source (or destination) details, checking for errors	*/
	/* perchance.							*/
	if (!base->set (copy, pError))
	{	DELOBJ	(base)	;
		return	false	;
	}

	return	base	;
}

/*  KBCopyExec								*/
/*  execDocument: Execute copy document					*/
/*  location	: KBLocation &	: Document location			*/
/*  report	: QString &	: Success completion report		*/
/*  error	: KBError &	: Error return				*/
/*  paramDict	:		: Caller parameters			*/
/*  showProg	: bool		: Show progress dialog			*/
/*  (returns)	: int		: Rows copied or negative on error	*/

int	KBCopyExec::execDocument
	(	KBLocation		&location,
		QString			&report,
		KBError			&pError,
		const QDict<QString>	&paramDict,
		bool			showProg
	)
{
	/* Get the document text. Report an error if it is not		*/
	/* available or if the document is empty.			*/
	QString	text	= location.contents (pError) ;

	if (text.isNull())
		return	-1 ;

	if (text.isEmpty ())
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Copier document is empty"),
			  	location.title(),
			  	__ERRLOCN
			  )	;
		return	-1	;
	}

	/* Got a document. Verify that is parses as a DOM document and	*/
	/* that it has a root document element.				*/
	QDomDocument	doc	;
	doc.setContent	(text)	;

	QDomElement	root	= doc.documentElement() ;
	if (root.isNull())
	{
		pError	= KBError
			  (	KBError::Error,
			  	TR("Copier document has no root element"),
			  	location.title(),
			  	__ERRLOCN
			  )	;
		return	-1	;
	}

	/* Load the source and destination specifications. This will	*/
	/* load just the appropriate bits.				*/
	KBCopyBase *srce = loadSpecification (location, root, true,  pError) ;
	if (srce == 0)
		return -1 ;

	KBCopyBase *dest = loadSpecification (location, root, false, pError) ;
	if (dest == 0)
	{	DELOBJ	(srce) ;
		return	-1 ;
	}

	QDict<KBParamSet>	params	;
	params.setAutoDelete	(true)	;

	for (QDomNode pnode = root.firstChild() ; !pnode.isNull() ; pnode = pnode.nextSibling())
	{
		QDomElement pelem = pnode.toElement() ;
		if (pelem.isNull() || (pelem.tagName() != "param"))
			continue ;

		params.insert
		(	pelem.attribute ("name"),
			new KBParamSet
			(	pelem.attribute ("legend") ,
				pelem.attribute ("defval"),
				QString::null,
				QString::null,
				false
		)	)	;
	}

	KBCopyExec exec	 (srce, dest) ;
	int	   nRows ;

	/* OK, loaded, now go ahead and execute the copy. Return the	*/
	/* number of rows copied unless there was an error, in which	*/
	/* case the result is negative.					*/
	if (!exec.execute (report, pError, nRows, paramDict, params, showProg))
	{	DELOBJ	(srce)	;
		DELOBJ	(dest)	;
		return	-1	;
	}

	DELOBJ	(srce)	;
	DELOBJ	(dest)	;
	return	nRows	;
}

bool	KBCopyExec::showProgress
	(	int		nRows
	)
{
	if (m_progress != 0)
	{
		qApp->processEvents() ;
		m_progress->setDone (nRows) ;
		return	m_progress->cancelled() ;
	}

	return	false	;
}
