/***************************************************************************
    file	         : kb_ctrl.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#ifndef 	_WIN32
#include	"kb_ctrl.moc"
#else
#include	"kb_ctrl.h"
#endif

#include	"kb_block.h"
#include	"kb_display.h"
#include	"kb_writer.h"
#include	"kb_nodemonitor.h"


/*  KBEventFilter							*/
/*  KBEventFilter: Event helper class constructor			*/
/*  control	 : KBControl *	 : Associated control			*/
/*  widget	 : QWidget *	 : Widget to be filtered		*/
/*  (returns)	 : KBEventFilter :					*/

KBEventFilter::KBEventFilter
	(	KBControl	*control,
		QWidget		*widget
	)
	:
	m_control	(control),
	m_widget	(widget)
{

	/* Install an event filter, so that we can trap various widget	*/
	/* events that the database framework needs to grab.		*/
	if (m_widget != 0)
		m_widget->installEventFilter (this) ;
}

/*  KBEventFilter							*/
/*  eventFilter	: Widget event filter					*/
/*  obj		: QObject *	: Object originating event		*/
/*  e		: QEvent *	: Event in question			*/
/*  (returns)	: bool		: False to continue processing		*/

bool	KBEventFilter::eventFilter
	(	QObject	*o,
		QEvent	*e
	)
{
	return	m_control->eventFilter (o, e)	;
}


/*  ------------------------------------------------------------------  */

/*  KBControl								*/
/*  KBControl	: Constructor for KBase QT widget base class		*/
/*  widget	: QWidget *	: Actual Qt widget			*/
/*  display	: KBDisplay *	: Owning display			*/
/*  object	: KBObject *	: Owning object				*/
/*  (returns)	: KBCtrlHelper	:					*/

KBControl::KBControl
	(	QWidget		*widget,
		KBDisplay	*display,
		KBObject	*object
	)
	:
	m_filter	(this, widget),
	m_widget	(widget),
	m_display	(display),
	m_object	(object),
	m_item		(0),
	m_drow		(0)
{
	setupWidget	() ;
}

/*  KBControl								*/
/*  KBControl	: Constructor for KBase QT widget base class		*/
/*  widget	: QWidget *	: Actual Qt widget			*/
/*  display	: KBDisplay *	: Owning display			*/
/*  item	: KBItem *	: Owning item				*/
/*  drow	: uint		: Display row				*/
/*  (returns)	: KBCtrlHelper	:					*/

KBControl::KBControl
	(	QWidget		*widget,
		KBDisplay	*display,
		KBItem		*item,
		uint		drow
	)
	:
	m_filter	(this, widget),
	m_widget	(widget),
	m_display	(display),
	m_object	(item),
	m_item		(item),
	m_drow		(drow)
{
	setupWidget	()	;
}

/*  KBControl								*/
/*  ~KBControl	: Destructor for KBase QT widget base class		*/
/*  (returns)	: void		:					*/

KBControl::~KBControl ()
{
	/* The control may have been morphed, and is being destroyed	*/
	/* as the display is shrunk, so we need to clear the area that	*/
	/* it occupied.							*/
	KBDisplay *ds = getDisplay() ;
	QWidget	  *bg = ds->getDisplayWidget() ;
	if (bg != 0)
	{
		QPainter p (bg) ;
		QRect	 rRect  = m_rect ;

		ds->cvtCtrlToView(rRect) ;
		p.eraseRect      (rRect) ;
	}

	if (m_object != 0) m_object->ctrlGone (this) ;
	DELOBJ	(m_monitor) ;
}

/*  KBControl								*/
/*  setupWidget	: Set up the Qt widget being used			*/
/*  (returns)	: void		:					*/

void	KBControl::setupWidget ()
{
	m_showing	= KB::ShowAsUnknown ;
	m_monitor	= 0	;
	m_lastState	= 0	;
	m_enabled	= true	;
	m_visible	= true	;
	m_morphed	= false	;
	
	/* Add the control to the display. This is needed for topmost	*/
	/* controls, where the parent is a QScrollView.			*/
	m_display->addChild (m_widget, 0, 0) ;

	/* All controls get tab and click focus. This is important,	*/
	/* not just for user convenience, but to make update work!	*/
	m_widget ->setFocusPolicy (QWidget::StrongFocus) ;
}

/*  KBControl								*/
/*  getDisplay	: Get underlying display				*/
/*  (returns)	: KBDisplay *	: Display				*/

KBDisplay *KBControl::getDisplay ()
{
	return	m_display ;
}

/*  KBControl								*/
/*  topWidget	: Get top-level widget					*/
/*  (returns)	: QWidget *	: Widget				*/

QWidget	*KBControl::topWidget ()
{
	/* This returns the top-level widget for the control, ie., the	*/
	/* one which gets resized and such, and to which the sizer	*/
	/* attaches the control blobs.					*/
	return	m_widget	;
}

/*  KBControl								*/
/*  mainWidget	: Get main widget					*/
/*  (returns)	: QWidget *	: Widget				*/

QWidget	*KBControl::mainWidget ()
{
	/* This returns the main widget for the control, ie., the one	*/
	/* which occupies most of the area. The sizer will install an	*/
	/* event filter to get the design popup menu.			*/
	return	m_widget	;
}

/*  KBControl								*/
/*  getIniValue	: Get initial value					*/
/*  (returns)	: KBValue	: Initial value				*/

KBValue	KBControl::getIniValue ()
{
	return	m_item == 0 ?
			KBValue() :
			m_item->getIniValue (m_item->getBlock()->getCurDRow() + m_drow) ;
}

/*  KBControl								*/
/*  eventFilter	: Widget event filter					*/
/*  obj		: QObject *	: Object originating event		*/
/*  e		: QEvent *	: Event in question			*/
/*  (returns)	: bool		: False to continue processing		*/

bool	KBControl::eventFilter
	(	QObject	*,
		QEvent	*e
	)
{
#if	! __KB_RUNTIME
	/* Do nothing if in design mode; the sizer will install an	*/
	/* event filter to do the work.					*/
	if (m_showing == KB::ShowAsDesign)
		return	false	;
#endif
	/* Focus in event. Pass this to the object, so that is needed	*/
	/* the owning block can keep track of the current row. The	*/
	/* event is also processed normally.				*/
	if (e->type() == QEvent::FocusIn)
	{
		m_object->focusInEvent (m_drow, QFocusEvent::reason()) ;
		return	false ;
	}

	/* A right-button press may be used to bring up a context menu	*/
	/* for the associated object.					*/
	if (e->type() == QEvent::MouseButtonPress)
	{
		if (((QMouseEvent *)e)->stateAfter() & QMouseEvent::RightButton)
		{	m_object->contextMenu ((QMouseEvent *)e, m_drow) ;
			return	true ;
		}
		return	false	;
	}

	/* Last case is a key press. These are trapped to handle such	*/
	/* stuff primarily as form navigation.				*/
	if (e->type() == QEvent::KeyPress)
		return	m_object->keyStroke ((QKeyEvent *)e) ;

	/* Anything else is handled by the underlying widget ...	*/
	return	false	;
}

/*  KBControl								*/
/*  updateMorph	: Update morphed control				*/
/*  (returns)	: void		:					*/

void	KBControl::updateMorph ()
{
	getDisplay()->updateMorph
	(	m_item,
		QRect
		(       m_rect.x     () - 1,
			m_rect.y     () - 1,
			m_rect.width () + 2,
			m_rect.height() + 2
		)
	)	;
}

/*  KBControl								*/
/*  setGeometry	: Set control geometry					*/
/*  x, y, ..	: int, ...	: Position and size			*/
/*  (returns)	: void		:					*/

void	KBControl::setGeometry
	(	int	x,
		int	y,
		int	w,
		int	h
	)
{
	m_rect	= QRect (x, y, w, h) ;

	m_widget  ->resize    (w, h) ;
	m_display ->moveChild (m_widget, x, y) ;

	if (m_morphed) updateMorph () ;
}

/*  KBControl								*/
/*  setGeometry	: Set control position					*/
/*  rect	: const QRect &	: Position and size			*/
/*  (returns)	: void		:					*/

void	KBControl::setGeometry
	(	const QRect	&rect
	)
{
	setGeometry (rect.x(), rect.y(), rect.width(), rect.height()) ;
}


/*  KBControl								*/
/*  setPalette	: Set palette on widget					*/
/*  pal		: const QPalette *: Palette in question			*/
/*  (returns)	: void		  :					*/

void	KBControl::setPalette
	(	const QPalette *pal
	)
{
	m_palette = *pal ;
	m_widget->setPalette (m_palette) ;
	if (m_morphed) updateMorph () ;
}

/*  KBControl								*/
/*  setFont	: Set font on widget					*/
/*  font	: const QFont *	: Font in question			*/
/*  (returns)	: void		:					*/

void	KBControl::setFont
	(	const QFont *font
	)
{
	m_font	= *font ;
	m_widget->setFont (m_font) ;
	if (m_morphed) updateMorph () ;
}

/*  KBControl								*/
/*  redraw	: Redraw control after design change			*/
/*  (returns)	: void		:					*/

void	KBControl::redraw ()
{
	/* Default does nothing, interested control reimplement		*/
}

/*  KBControl								*/
/*  showAs	: Set/unset design mode					*/
/*  mode	: KB::ShowAs	: Design mode				*/
/*  (returns)	: void		:					*/

void	KBControl::showAs
	(	KB::ShowAs	mode
	)
{
	m_showing = mode ;
	m_enabled = true ;
	m_visible = true ;

	m_widget->setEnabled (true) ;

	if ((mode == KB::ShowAsData) && m_morphed)
		m_widget->hide () ;
	else	m_widget->show () ;
}

/*  KBControl								*/
/*  setFocus	: Set focus to widget					*/
/*  (returns)	: void		:					*/

void	KBControl::setFocus ()
{
	if (!m_widget->hasFocus())
		if (m_object->moveFocusOK (m_drow))
			giveFocus () ;
}

/*  KBControl								*/
/*  geometry	: Get control geometry					*/
/*  (returns)	: QRect		: Geometry				*/

QRect	KBControl::geometry ()
{
	return	m_rect	;
}

/*  KBControl								*/
/*  setData	: Control-specific data function			*/
/*  data	: void *	: Data pointer				*/
/*  (returns)	: void		:					*/

void	KBControl::setData
	(	void	*
	)
{
	/* This method is used to set control-specific information, and	*/
	/* avoids problems with interactions between setting such	*/
	/* information and control implementations.			*/
}

/*  KBControl								*/
/*  showName	: Show control name					*/
/*  (returns)	: void		:					*/

void	KBControl::showName ()
{
	/* Implemented in the derived classes ...			*/
}

/*  KBControl								*/
/*  setMonitor	: Set monitor display value				*/
/*  value	: const KBValue&: Value					*/
/*  (returns)	: void		:					*/

void	KBControl::setMonitor
	(	const KBValue &v
	)
{
	if (m_monitor != 0)
	{
		QString	t = v.getRawText() ;
		if (t.length() > 80)
		{	t.truncate  (77)    ;
			t.append    ("...") ;
		}

		m_monitor->setText (2, t) ;
	}

}

/*  KBControl								*/
/*  startUpdate	: Check that record update can start			*/
/*  (returns)	: bool		: Update started			*/

bool	KBControl::startUpdate ()
{
	if ((m_showing == KB::ShowAsData) && (m_item != 0))
	{
		/* If an update cannot be started then restore the	*/
		/* current value. This must be the original value.	*/
		if (!m_item->startUpdate (m_item->getBlock()->getCurDRow() + m_drow))
		{
			setValue (m_curVal) ;
			return	 false	    ;
		}

		return	true	;
	}

	return	false	;
}

/*  KBControl								*/
/*  setValue	: Set value						*/
/*  value	: const KBValue&: Value					*/
/*  (returns)	: void		:					*/

void	KBControl::setValue
	(	const KBValue &v
	)
{
	setMonitor (m_curVal = v) ;
  
	// fprintf (stderr, "KBControl::setValue: [%s][%s:%d]<-[%s]\n",
	//		 (cchar *)object->getElement(),
	//		 (cchar *)object->getName   (),
	//		 drow,
	//		 (cchar *)v.getRawText()) ;
}

/*  KBControl								*/
/*  getValue	: Get control value					*/
/*  (returns)	: KBValue	: Value					*/

KBValue	KBControl::getValue ()
{
	/* This is a stub. It will be implemented in the derived	*/
	/* classes for the various types of controls.			*/
	return	KBValue () ;
}

/*  KBControl								*/
/*  clearValue	: Clear value						*/
/*  query	: bool		: Clearing for query			*/
/*  (returns)	: void		:					*/

void	KBControl::clearValue
	(	bool
	)
{
	if (m_monitor != 0) m_monitor->setText (2, "") ;
}

/*  KBControl								*/
/*  changed	: Get value changed status				*/
/*  (returns)	: bool		: Value changed				*/

bool	KBControl::changed ()
{
	/* This is a stub. It will be implemented in the derived	*/
	/* classes for the various types of controls.			*/
	return	false ;
}

/*  KBControl								*/
/*  isEmpty	: See if control is empty				*/
/*  (returns)	: bool		: Empty					*/

bool	KBControl::isEmpty ()
{
	/* This is a stub. It will be implemented in the derived	*/
	/* classes for the various types of controls.			*/
	return	false ;
}

/*  KBControl								*/
/*  isValid	: See if control is valid				*/
/*  allowNull	: bool		: Ignore not-null checks		*/
/*  (returns)	: bool		: Valid					*/

bool	KBControl::isValid
	(	bool
	)
{
	/* This is a stub. It will be implemented in the derived	*/
	/* classes for the various types of controls which might have	*/
	/* invalid contents.						*/
	return	true ;
}

/*  KBControl								*/
/*  setEnabled	: Set control enabled/disabled				*/
/*  enabled	: bool		: Enabled/disabled			*/
/*  (returns)	: void		:					*/

void	KBControl::setEnabled
	(	bool	enabled
	)
{
	m_enabled = enabled ;

	if (m_morphed)
	{
		getDisplay()->updateMorph (m_item, m_drow) ;
		return	;
	}

	m_widget->setEnabled (enabled) ;
}

/*  KBControl								*/
/*  setVisible	: Set visibility					*/
/*  visible	: bool		: Set visible				*/
/*  (returns)	: void		:					*/

void	KBControl::setVisible
	(	bool	visible
	)
{
	m_visible = visible ;

	if (m_morphed)
	{
		getDisplay()->updateMorph (m_item, m_drow) ;
		return	;
	}

	if (visible)
		m_widget->show () ;
	else	m_widget->hide () ;
}

/*  KBControl								*/
/*  isEnabled	: Get control enabled/disabled				*/
/*  (returns)	: bool		: Enabled/disabled			*/

bool	KBControl::isEnabled ()
{
	return	m_enabled ;
}

/*  KBControl								*/
/*  isVisible	: Get control visibility				*/
/*  (returns)	: bool		: Visibility				*/

bool	KBControl::isVisible ()
{
	return	m_visible ;
}

/*  KBControl								*/
/*  sizeHint	: Get size hint for label				*/
/*  (returns)	: QSize		: Size hint				*/

QSize	KBControl::sizeHint () const
{
	/* Don't understand this, maybe a Qt bug? If we don't have this	*/
	/* and the label is the first control on the page, it gets	*/
	/* auto-resized.						*/
	return	m_object->geometry().size() ;
}

/*  KBControl								*/
/*  write	: Dummy write routine					*/
/*  writer	: KBWriter *	: Writer				*/
/*  rect	: QRect		: Area elative to writer		*/
/*  value	: KBValue &	: Value for reports			*/
/*  fSubs	: bool		: Substitution flag			*/
/*  extra	: init &	: Return extra space			*/
/*  (returns)	: bool		: Success				*/

bool	KBControl::write
	(	KBWriter	*writer,
		QRect		rect,
		const KBValue	&value,
		bool		fSubs,
		int		&extra
	)
{
	extra	= 0	;

	/* We should not get called, unless the actual control has not	*/
	/* implemented this method. The default report action is to	*/
	/* write some appropriate text, otherwise grap the widget.	*/
	if (writer->asReport())
	{
		new KBWriterText
		(	writer,
			rect,
			m_item->getPalette(true),
			m_item->getFont	  (true),
			value.getRawText  (),
			Qt::AlignLeft|Qt::AlignVCenter,
			fSubs
		)	;
	
		return	true	;
	}

	new KBWriterPixmap
	(	writer,
		rect,
		QPixmap::grabWidget (m_widget)
	)	;

	return	true	;
}

/*  KBControl								*/
/*  showMonitor	: Show or clear monitor for this node			*/
/*  parent	: QListViewItem * : Parent monitor			*/
/*  (returns)	: void		  :					*/

void	KBControl::showMonitor
	(	QListViewItem	*parent
	)
{
	if (parent == 0)
	{	m_monitor = 0 ;
		return	;
	}

	if (m_item != 0)
	{
		QString	t = getValue().getRawText() ;
		if (t.length() > 80)
		{	t.truncate (77)	   ;
			t.append   ("...") ;
		}

		m_monitor = new KBNodeMonitor (0, parent) ;

		m_monitor->setText (0, "Control") ;
		m_monitor->setText (1, QString("Row %1").arg(m_drow)) ;
		m_monitor->setText (2, t) ;
	}
}

#if	! __KB_EMBEDDED

/*  KBControl								*/
/*  property	: Get QT widget property				*/
/*  name	: cchar *	: Property name				*/
/*  (returns)	: QVariant	: Property value			*/

QVariant
	KBControl::property
	(	cchar		*name
	)
{
	return	m_widget->property (name) ;
}

/*  KBControl								*/
/*  setProperty	: Set QT widget property				*/
/*  name	: cchar *	   : Property name			*/
/*  value	: const QVariant & : Value				*/
/*  (returns)	: bool		   : Success				*/

bool	KBControl::setProperty
	(	cchar		*name,
		const QVariant	&value
	)
{
	return	m_widget->setProperty (name, value) ;
}

#endif

/*  KBControl								*/
/*  ctrlSetFrame: Set frame style on frame object			*/
/*  frame	: QFrame *	: QFrame frame object in question	*/
/*  (returns)	: uint		: Frame width				*/

uint	KBControl::ctrlSetFrame
	(	QFrame		*frame
	)
{
	if (m_object)
	{
		QString	spec	= m_object->getAttrVal("frame") ;

		int	coff	= spec.find (',') ;
		int	fs	= 0 ;
		int	lw	= 0 ;

		if (coff >= 0)
		{	fs = spec.left(coff    ).toInt() ;
			lw = spec.mid (coff + 1).toInt() ;
		}

		frame->setFrameStyle (fs) ;
		frame->setLineWidth  (lw) ;

		return	lw ;
	}

	return	0 ;
}

/*  KBControl								*/
/*  setMorphed	: Set morphed state					*/
/*  morphed	: bool		: True to morph				*/
/*  (returns)	: void		:					*/

void	KBControl::setMorphed
	(	bool		morphed
	)
{
//	fprintf
//	(	stderr,
//		"KBControl::setMorphed: [%s][%d] [%d]->[%d]\n",
//		(cchar *)m_object->getName(),
//		m_drow,
//		m_morphed,
//		morphed
//	)	;

	if (morphed == m_morphed) return ;

	m_morphed = morphed ;

	if (m_morphed)
	{
		m_widget->hide () ;
		getDisplay()->updateMorph (m_item, m_drow) ;
		return	;
	}

	m_widget->setEnabled (m_enabled) ;

	if (m_visible)
		m_widget->show() ;
	else	m_widget->hide() ;

	m_widget->update () ;
}

/*  KBControl								*/
/*  paintMorph	: Paint value when morphed				*/
/*  p		: QPainter *	  : Painter				*/
/*  text	: const QString & : Value				*/
/*  (returns)	: void		  :					*/

void	KBControl::paintMorph
	(	QPainter	*p,
		const QString	&text
	)
{
	QRect	bRect
		(	m_rect.x     () + 3,
			m_rect.y     () + 1,
			m_rect.width () - 4,
			m_rect.height() - 2
		)	;

//	static	uint	count = 0 ;
//	fprintf
//	(	stderr,
//		"paintMorph[%6d] [%s/%d] %s\n",
//		count,
//		(cchar *)item->getName(),
//		drow,
//		_TEXT(mRect)
//	)	;
//	count	+= 1 ;

	if (m_visible)
	{
		p->setPen	(m_palette.active().text()) ;
		p->setBrush	(m_palette.active().base()) ;
		p->setFont	(m_font) ;
		p->drawRect	(m_rect) ;
		p->drawText	(bRect, Qt::AlignVCenter|Qt::AlignLeft, text) ;
	}
	else	p->eraseRect	(m_rect) ;
}

/*  KBControl								*/
/*  repaintMorph: Paint value when morphed				*/
/*  p		: QPainter *	  : Painter				*/
/*  (returns)	: void		  :					*/

void	KBControl::repaintMorph
	(	QPainter	*p
	)
{
	paintMorph (p, morphText()) ;
}

/*  KBControl								*/
/*  morphText	: Get text for morphed control				*/
/*  (returns)	: QString	: Text					*/

QString	KBControl::morphText ()
{
	return	m_curVal.getRawText() ;
}
