/***************************************************************************
    file	         : kb_blockdlg.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	"kb_classes.h"
#include	"kb_type.h"
#include	"kb_value.h"
#include	"kb_location.h"

#ifndef		_WIN32
#include	"kb_blockdlg.moc"
#else
#include	"kb_blockdlg.h"
#endif

#include	"kb_block.h"
#include	"kb_hidden.h"
#include	"kb_qrybase.h"

static	IntChoice	choicePThrow[] =
{
	{	0,		"None"			},
	{	1,		"Group"			},
	{	2,		"Record"		},
	{	-1,		0			}
}	;

static	IntChoice	choiceLocking[] =
{
	{	0,		"None"			},
	{	1,		"Lock row on update"	},
	{	-1,		0			}
}	;

static	cchar		*attrExclude[] =
{
	"taborder",
	"rdonly",
	"default",
	"onenter",
	"onleave",
	"onset",
	"stretch",
	0
}	;

static	HelpMap	helpMap	[] =
{
	{	"dx",		"spacing"	},
	{	"dy",		"spacing"	},
	{	"master",	"linkage"	},
	{	"child",	"linkage"	},
	{	0,		0		}
}	;

/*  ------------------------------------------------------------------  */

/*  KBPropBlockDlg							*/
/*  KBPropBlockDlg: Constructor for block properties dialog		*/
/*  block	  : KBBlock * 	    : Associated block			*/
/*  caption	  : cchar *    	    : Dialog box caption		*/
/*  attribs	  : QList<KBAttr>&  : List of attributes		*/
/*  iniattr	  : cchar *	    : Initial attribute			*/
/*  (returns)	  : KBBlockPropDlg  :					*/

KBBlockPropDlg::KBBlockPropDlg
	(	KBBlock		*block,
		cchar		*caption,
		QList<KBAttr>	&attribs,
		cchar		*iniattr
	)
	:
	KBItemPropDlg	(block, caption, attribs, iniattr),
	m_hiddenDlg	(&topWidget, block),
	m_bQuery	(this),
	m_block		(block)
{
	m_hiddenDlg.hide() ;

	switch (m_block->getBlkType())
	{
		case KBBlock::BTTable	:
		case KBBlock::BTQuery	:
		case KBBlock::BTSQL	:
			m_bQuery  .setText   (TR("Query")) ;
			m_bQuery  .show	     () ;
			butLayout .addWidget (&m_bQuery, 1, 4) ;
			connect (&m_bQuery, SIGNAL(clicked()), this, SLOT(clickQuery ())) ;
			break	;

		default	:
			m_bQuery.hide () ;
			break	;
	}
}

/*  KBPropBlockDlg							*/
/*  ~KBPropBlockDlg: Destructor for block properties dialog		*/
/*  (returns)	   : void	:					*/

KBBlockPropDlg::~KBBlockPropDlg ()
{
}

/*  KBBlockPropDlg							*/
/*  findHelpMapping							*/
/*		: Find possible help mapping for name			*/
/*  name	: const QString & : Name				*/
/*  (returns)	: cchar *	  : Mapping or null if none		*/

cchar	*KBBlockPropDlg::findHelpMapping
	(	const QString	&name
	)
{
	for (HelpMap *ptr = &helpMap[0] ; ptr->name != 0 ; ptr += 1)
		if (ptr->name == name)
			return	ptr->help ;

	return	KBItemPropDlg::findHelpMapping (name) ;
}

/*  KBBlockPropDlg								*/
/*  resizeEvent	: Handle dialog resizing				*/
/*  e		: QResizeEvent * : Resize event				*/
/*  (returns)	: void		 :					*/

void	KBBlockPropDlg::resizeEvent
	(	QResizeEvent	*e
	)
{
	KBItemPropDlg::resizeEvent (e) ;
}

/*  KBBlockPropDlg							*/
/*  getAttrItem	: Get attribute item for attribute			*/
/*  attr	: KBAttr *	: Attribute				*/
/*  (returns)	: KBAttrItem *	: Associated attribute item		*/

KBAttrItem *KBBlockPropDlg::getAttrItem
	(	KBAttr	*attr
	)
{
	if (attr->getName() == "pthrow" )
		return	new KBAttrIntChoice (attr, choicePThrow ) ;

	if (attr->getName() == "locking")
		return	new KBAttrIntChoice (attr, choiceLocking) ;

	return	KBItemPropDlg::getAttrItem (attr) ;
}

/*  KBBlockPropDlg							*/
/*  dropProperty: Drop property after user clicks ignore button		*/
/*  (returns)	: void		:					*/

void	KBBlockPropDlg::dropProperty ()
{
	KBItemPropDlg::dropProperty () ;
}

/*  KBBlockPropDlg							*/
/*  showProperty: Show property						*/
/*  item	: KBAttrItem *	: Associated item			*/
/*  (returns)	: bool		: Property on show			*/

bool	KBBlockPropDlg::showProperty
	(	KBAttrItem	*item
	)
{
	const QString	&aName = item->attr()->getName () ;

	/* Hidden field. We just show the hidden field widget and	*/
	/* enable/disable to remove button appropriately.		*/
	if (aName == "hidden")
	{
		setUserWidget (&m_hiddenDlg) ;
		return	true  ;
	}

	/* Child							*/
	/* This should display a list of all columns available from	*/
	/* the associated query, although the user can type in an	*/
	/* arbitrary expression.					*/
	if (aName == "child")
	{
		KBQryBase    *query  ;

		if ((query = m_block->getQuery()) == 0)
			return	warning (TR("Block lacks a query")) ;

		return	pickQueryField
			(	query,
				m_block->getQryLevel(),
				item->value(),
				m_block->getParent() == 0
			)	;
	}

	/* PThrow							*/
	/* This is the pagethrow property (none, per record, per group)	*/
	/* which is a choice.						*/
	if (aName == "pthrow" )
	{
		showChoices (item, choicePThrow,  item->value()) ;
		return	true ;
	}

	if (aName == "locking")
	{
		showChoices (item, choiceLocking, item->value()) ;
		return	true ;
	}

	return	KBItemPropDlg::showProperty (item) ;
}

/*  KBBlockPropDlg							*/
/*  saveProperty: Check and save attribute				*/
/*  item	: KBAttrItem *	: Associated item			*/
/*  (returns)	: bool		: Attribute OK				*/

bool	KBBlockPropDlg::saveProperty
	(	KBAttrItem	*item
	)
{
	const QString	&aName = item->attr()->getName () ;

	/* Hidden fields						*/
	/* The actual information is in the list view, but stitch	*/
	/* together the appropriate string.				*/
	if (aName == "hidden")
	{
		setProperty (aName, m_hiddenDlg.getText()) ;
		return	true ;
	}

	/* Child							*/
	/* In this case, copy pback the current selection from the	*/
	/* combo box.							*/
	if (aName == "child")
	{
		setProperty (aName, comboBox.text(comboBox.currentItem())) ;
		return	true ;
	}

	/* PThrow							*/
	/* This is the pagethrow property (none, per record, per group)	*/
	/* which is a choice.						*/
	if (aName == "pthrow" )
	{
		saveChoices (item, choicePThrow ) ;
		return	true	;
	}

	if (aName == "locking")
	{
		saveChoices (item, choiceLocking) ;
		return	true	;
	}

	return	KBItemPropDlg::saveProperty (item) ;
}

/*  KBBlockPropDlg							*/
/*  hideProperty: Hide property						*/
/*  attr	: KBAttr *	: Attribute in question			*/
/*  (returns)	: void		:					*/

bool	KBBlockPropDlg::hideProperty
	(	KBAttr	*attr
	)
{
	const QString	&aName	= attr->getName() ;
	bool  masterChild	= (aName == "master") || (aName == "child") ;

	if (aName == "hidden") return false ;

	/* Check for the properties which are actually in a form or	*/
	/* report, or are otherwise not relevant here, and ignore	*/
	/* them, and also the read-only property.			*/
	if (qstrcmp (attr->getOwnerName(), "KBForm"  ) == 0)
		return	true	;
	if (qstrcmp (attr->getOwnerName(), "KBReport") == 0)
		return	true	;

	for (cchar **exp = &attrExclude[0] ; *exp != 0 ; exp += 1)
		if (*exp == aName)
			return	true ;

	/* If the block is a menu then there is no master-child		*/
	/* relationship.						*/
	if (m_block->getQuery()->isQryNull())
		if (masterChild)
			return	true ;

	/* Ditto the above if the block has a non-zero query level, in	*/
	/* which case it is a nested subblock.				*/
	if (m_block->getQryLevel() != 0)
		if (masterChild)
			return	true ;

	/* Unless we have a parent block, there is no point in showing	*/
	/* the master attribute, nor the X and Y values which should	*/
	/* always be zero at this level). This also hides the frame	*/
	/* and title properties. Also skip the name since it has no	*/
	/* use at the top level.					*/
	KBBlock	*parentBlk = m_block->getBlock () ;

	if (parentBlk == 0)
		if ( (aName == "master"  ) ||
		     (aName == "name"    ) ||
		     (aName == "x"       ) ||
		     (aName == "y"       ) ||
		     (aName == "xmode"   ) ||
		     (aName == "ymode"   ) ||
		     (aName == "title"   ) ||
		     (aName == "frame"   ) ) return true ;

	/* If the parent block has a null query then again there is no	*/
	/* master-child relationship.					*/
	if ((parentBlk != 0) && parentBlk->getQuery()->isQryNull())
		if (masterChild)
			return	true	;

	return	false	;
}

/*  KBBlockPropDlg:							*/
/*  propertyOK	: Check if property is OK				*/
/*  item	: KBAttrItem *	: Associated item			*/
/*  (returns)	: bool		: Property is OK			*/

bool	KBBlockPropDlg::propertyOK
	(	KBAttrItem	*item
	)
{
	if (item->attr()->getName() == "child")
		if (m_block->getParent() == 0)
			return	true ;

	return	KBItemPropDlg::propertyOK (item) ;
}

/*  KBBlockPropDlg:							*/
/*  clockQuery	  : Handle query button click				*/
/*  (returns)	  : void	:					*/

void	KBBlockPropDlg::clickQuery ()
{
	m_block->getQuery()->propertyDlg () ;
}

/*  KBBlockPropDlg:							*/
/*  clickOK	  : Handle OK button click				*/
/*  (returns)	  : void	:					*/

void	KBBlockPropDlg::clickOK ()
{
	m_hiddenDlg   .clickOK () ;
	KBItemPropDlg::clickOK () ;
}

/*  KBBlockPropDlg:							*/
/*  clickCancel	  : Handle cancel button click				*/
/*  (returns)	  : void	:					*/

void	KBBlockPropDlg::clickCancel ()
{
	m_hiddenDlg   .clickCancel () ;
	KBItemPropDlg::clickCancel () ;
}

/*  KBBlockPropDlg:							*/
/*  preExec	  : Called immediately before QDialog::exec		*/
/*  (returns)	  : void	:					*/

void	KBBlockPropDlg::preExec ()
{
	/* Trap this and load the notional hidden attribute. We have to	*/
	/* do this here since it is not until "KBPropDlg::exec" has	*/
	/* been called that "setProperty" will work.			*/
	setProperty ("hidden", m_hiddenDlg.getText()) ;
}

/*  blockPropDlg: Run block properties dialog				*/
/*  block	: KBBlock *	  : Block				*/
/*  caption	: cchar *  	  : Dialog box caption			*/
/*  attribs	: QList<KBAttr>&  : List of attributes			*/
/*  iniattr	: cchar *	  : Initial attribute			*/
/*  (returns)	: bool		  : Success				*/

bool	blockPropDlg
	(	KBBlock		*block,
		cchar		*caption,
		QList<KBAttr>	&attribs,
		cchar		*iniattr
	)
{
	KBBlockPropDlg bDlg (block, caption, attribs, iniattr) ;
	return	bDlg.exec () ;
}
