/***************************************************************************
    file	         : eld.cpp
    copyright            : (C) 1999,2000,2001,2002,2003 by Mike Richardson
			   (C) 2000,2001,2002,2003 by theKompany.com
			   (C) 2001,2002,2003 by John Dean
    license              : This file is released under the terms of
                           the GNU General Public License, version 2. The
                           copyright holders retain the right to release
                           this code under diffenent non-exclusive licences.
    email                : mike@quaking.demon.co.uk                                     
 ***************************************************************************/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<fcntl.h>
#include	<ctype.h>
#include	<stdarg.h>
#include	<setjmp.h>
#include	<std.h>

#ifndef _WIN32
#include	<unistd.h>
#else
#include <io.h>
#endif

#include	"eli.h"
#include	"interp.h"
#include	"syn.h"
#include	"code.h"
#include	"optab.h"

#define	USAGE	"usage: elc file\n"

typedef	struct
{	char	*inst	;
	int	atype	;
}	INST	;

#define	A_INT	0
#define	A_DBL	1
#define	A_STR	2
#define	A_VAR	3
#define	A_FUNC	4
#define	A_OPER	5
#define	A_AT	6
#define	A_NONE	7

LVAR	INST	itable[] =
{	{	"oper",		A_OPER	},		/* 0		*/
	{	"bra",		A_INT	},		/* 1		*/
	{	"brat",		A_INT	},		/* 2		*/
	{	"braf",		A_INT	},		/* 3		*/
	{	NULL,		0	},		/* 4		*/
	{	"lout",		A_VAR	},		/* 5		*/
	{	"sout",		A_VAR	},		/* 6		*/
	{	"lblk",		A_INT	},		/* 7		*/
	{	"sblk",		A_INT	},		/* 8		*/
	{	"lnum",		A_INT	},		/* 9		*/
	{	"lstr",		A_STR	},		/* 10		*/
	{	"pop",		A_INT	},		/* 11		*/
	{	"call",		A_INT	},		/* 12		*/
	{	"ret",		A_NONE	},		/* 13		*/
	{	"argc",		A_INT	},		/* 14		*/
	{	"blks",		A_INT	},		/* 15		*/
	{	"sind",		A_NONE	},		/* 16		*/
	{	"lind",		A_NONE	},		/* 17		*/
	{	"dup",		A_NONE	},		/* 18		*/
	{	"swap",		A_NONE	},		/* 19		*/
	{	"callv",	A_INT	},		/* 20		*/
	{	"soutp",	A_INT	},		/* 21		*/
	{	"sblkp",	A_INT	},		/* 22		*/
	{	"meth",		A_INT	},		/* 23		*/
	{	"retn",		A_INT	},		/* 24		*/
	{	"incr",		A_INT	},		/* 25		*/
	{	"decr",		A_INT	},		/* 26		*/
	{	"sblks",	A_INT	},		/* 27		*/
	{	"sblkn",	A_INT	},		/* 28		*/
	{	"lblkn",	A_INT	},		/* 29		*/
	{	"lblks",	A_INT	},		/* 30		*/
	{	"at",		A_AT	},		/* 31		*/
	{	"ldbl",		A_DBL	},		/* 32		*/
	{	"retd",		A_DBL	},		/* 33		*/
	{	"lblkd",	A_INT	},		/* 34		*/
	{	"sblkd",	A_INT	},		/* 35		*/
	{	NULL,		0	}		/* 36		*/
}	;


LVAR	int	opcode	;
LVAR	int	oparg	;

LVAR	int	nnames	;
LVAR	int	nstrs	;
LVAR	char	**names	;
LVAR	char	**strs	;

LVAR	int	*code	;
LVAR	int	coff	;
LVAR	int	clen	;

	jmp_buf	cgErrEnv ;

/*L allocate	: Allocate space checking for errors			*/
/*  size	: int		: Space required in bytes		*/
/*  (returns)	: void *	: Pointer at allocated space		*/

LFUNC	void	*allocate
	(	int	size
	)
{
	void	*p	= NULL ;

	if (size > 0)
		if ((p = (void *) calloc (1, size)) == NULL)
			errorE ("el: out of memory\n") ;
	return	p ;

}

/*L errorD	: Report disassembly error and exit			*/
/*  msg		: char *	: Error message				*/
/*  ...		: ...		: Possible arguments			*/
/*  (returns)	: void		: (Never returns)			*/

LFUNC	void	errorD
	(	char	*msg,
		...
	)
{
	va_list  aptr	;

	va_start (aptr, msg) ;
	fprintf  (stderr, "eld: ") ;
	vfprintf (stderr, msg, aptr) ;
	fprintf	 (stderr, "\n") ;
	va_end   (aptr) ;
	exit	 (1) ;
}

/*L rdinst	: Read instruction from input stream			*/
/*  fid		: int		: Input stream				*/
/*  (returns)	: int		: Non-zero on success			*/

LFUNC	int	rdinst
	(	int	fid
	)
{
	unsigned int	word	;

	/* Read in a int. This contains the opcode and, normally, the	*/
	/* argument.							*/
	if (read (fid, &word, sizeof(int)) != sizeof(int)) return 0 ;

	/* Extract the opcode and argument. If the argument is not	*/
	/* escaped then we can return.					*/
	opcode	= word >> OPSHIFT ;
	oparg	= word &  ARBITS  ;
	if (oparg != ARBITS) return 1 ;

	/* Escaped argument so the next word contains the actual value.	*/
	if (read (fid, &word, sizeof(int)) != sizeof(int)) return 0 ;
	oparg	= word ;

	return	1 ;
}

/*L getinst	: Get instruction from buffer				*/
/*  (returns)	: int		: Non-zero on success			*/

LFUNC	int	getinst ()
{
	unsigned int	word	;

	/* Get the next instruction if there is one. This contains	*/
	/* the opcode and, normally, the argument.			*/
	if (coff >= clen) return 0 ;
	word	= code[coff++] ;

	/* Extract the opcode and argument. If the argument is not	*/
	/* escaped then we can return.					*/
	opcode	= word >> OPSHIFT ;
	oparg	= word &  ARBITS  ;
	if (oparg != ARBITS) return 1 ;

	/* Escaped argument so the next word contains the actual value.	*/
	if (coff >= clen) return 0 ;
	oparg	= code[coff++] ;

	return	1 ;
}

LFUNC	void	prstr
	(	char	*str
	)
{
	while (*str)
	{	if	(*str ==  '\n') printf ("\\n") ;
		else if (*str ==  '\r') printf ("\\r") ;
		else if (*str ==  '\b') printf ("\\b") ;
		else if (*str ==  '\t') printf ("\\t") ;
		else if (*str ==  0033) printf ("\\e") ;
		else if (*str >=  0177) printf ("\\0%03o", *str) ;
		else if	(iscntrl(*str)) printf ("\\^%c", *str - 1 + 'a') ;
		else			printf ("%c", *str) ;
		str += 1 ;
	}
}

/*L rdntab	: Read name and string tables				*/
/*  fid		: int		: Input stream				*/
/*  (returns)	: void		:					*/

LFUNC	void	rdntab
	(	int	fid
	)
{
	int	idx	;

	/* First check that the name table and string table lengths	*/
	/* are correctly specified. If all is OK then allocate them.	*/
	if (!rdinst (fid))    errorD ("name table length missing") ;
	if (opcode != C_NOUT) errorD ("expected C_NOUT") ;
	nnames	= oparg	;
	if (!rdinst (fid))    errorD ("string table length missing") ;
	if (opcode != C_NSTR) errorD ("expected C_NSTR") ;
	nstrs	= oparg	;

	names	= (char **)allocate(nnames * sizeof(char *)) ;
	strs	= (char **)allocate(nstrs  * sizeof(char *)) ;

	/* Read in the outer name table, checking everyting in sight	*/
	/* and printing the names. These should be marked as either	*/
	/* local or global, and as variables, functions or references.	*/
	printf	("Names:\n") ;
	for (idx = 0 ; idx < nnames ; idx += 1)
	{
		int	len	;
		char	*buf	;

		if (!rdinst (fid))    errorD ("name marker missing") ;
		if (opcode != C_NDEF) errorD ("expected C_NDEF for name") ;
		if ((oparg & N_SCOPE) != N_GLOBAL &&
		    (oparg & N_SCOPE) != N_LOCAL  &&
		    (oparg & N_SCOPE) != N_PUBLIC )
				      errorD ("expected N_GLOBAL, N_LOCAL or N_PUBLIC") ;
		if ((oparg & N_TYPE ) != N_VAR    &&
		    (oparg & N_TYPE ) != N_FUNC   )
				      errorD ("expected N_VAR or N_FUNC") ;
		if (read (fid, &len, sizeof(int)) != sizeof(int))
			errorD ("error reading name length") ;
		names[idx] = buf = (char *)allocate(len) ;
		if (read (fid, buf, len) != len)
			errorD ("error reading name text") ;

		printf	("%4d %c%c %s\n", idx,
			 (oparg & N_SCOPE) == N_LOCAL  ? 'L' :
			 (oparg & N_SCOPE) == N_PUBLIC ? 'P' : 'G',
			 (oparg & N_TYPE ) == N_VAR    ? 'V' :
			 (oparg & N_TYPE ) == N_FUNC   ? 'F' : 'R',
			 buf) ;
	}

	/* Similarly for the strings. These should be marked simply	*/
	/* as strings.							*/
	printf	("Strings:\n") ;
	for (idx = 0 ; idx < nstrs ; idx += 1)
	{
		int	len	;
		char	*buf	;

		if (!rdinst (fid))     errorD ("string marker missing") ;
		if (opcode != C_NDEF)  errorD ("expected C_NDEF for string") ;
		if (oparg != N_STRING) errorD ("expected N_STRING") ;
		if (read (fid, &len, sizeof(int)) != sizeof(int))
			errorD ("error reading string length") ;
		strs[idx] = buf = (char *)allocate(len) ;
		if (read (fid, buf, len) != len)
			errorD ("error reading string text") ;

		printf	("%4d    ", idx) ;
		prstr   (buf ) ;
		printf	("\n") ;
	}
}

/*L disasm	: Disassemble code in buffer				*/
/*  fbuf	: int *		: Function code buffer			*/
/*  flen	: int		: Function length			*/
/*  (returns)	: void		:					*/

LFUNC	void	disasm
	(	int	*fbuf,
		int	flen
	)
{
	int	idx	;

	code	= fbuf	;
	coff	= 0	;
	clen	= flen	;

	while (idx = coff, getinst ())
	{
		INST	*inst	= &itable[opcode] ;

		if (inst->inst == NULL)
			errorD ("Bad opcode %d", opcode) ;

		printf	("%5d   %-6s", idx, inst->inst) ;

		switch (inst->atype)
		{
			case A_AT   :
			{
				int	stno	= oparg >> 12	 ;
				int	fidx	= oparg & 0x0fff ;

				prstr	(strs[fidx]) ;
				printf	(",") ;
				printf	("%d", stno) ;
				break	;
			}

			case A_INT  :
				printf	("%d", oparg) ;
				break	;

			case A_DBL  :
			{
				double	dbl	;
				int	*dp	= (int *)&dbl ;
				int	idx	= sizeof(double)/sizeof(int) ;

				while	(idx > 0)
				{	*dp	= code[coff++] ;
					dp     += 1 ;
					idx    -= 1 ;
				}

				printf	("%e", dbl) ;
				break	;
			}

			case A_STR  :
				if (oparg >= nstrs)
					errorD	("string index %d", oparg) ;
				printf	("\"") ;
				prstr	(strs[oparg]) ;
				printf	("\"") ;
				break	;

			case A_VAR  :
				if (oparg >= nnames)
					errorD	("variable index %d", oparg) ;
				printf	("%s", names[oparg]) ;
				break	;

			case A_FUNC :
				if	(oparg == -1)
					printf	("<indirect>") ;
				else if ((oparg >= nnames) || (oparg < -1))
					errorD	("function index %d", oparg) ;
				else	printf	("%s", names[oparg]) ;
				break	;

			case A_OPER :
				if ((oparg > O_MAX) || (oparg < 1))
					errorD	("operator %d", oparg) ;
				printf	("%s", optab[oparg]) ;
				break	;
				
			default	    :
				break	;
		}

		printf ("\n") ;
	}
}

/*L rdfuncs	: Read functions from file				*/
/*  fd		: int		: Input stream				*/
/*  (returns)	: void		:					*/

LFUNC	void	rdfuncs
	(	int	fid
	)
{
	while (rdinst (fid))
	{
		int	*fbuf ;

		if (opcode != C_FDEF &&
		    opcode != C_PDEF) errorD ("expected C_FDEF/C_PDEF") ;
		if (oparg >=  nnames) errorD ("function index %d", oparg) ;
		printf	("%s function %s:\n",
				opcode == C_FDEF ? "Global/Local" :
				opcode == C_PDEF ? "Public"       :
						   "Unknown",
				names[oparg]) ;
		if (!rdinst (fid))    errorD ("function length missing") ;
		if (opcode !=  C_LEN) errorD ("expected C_LEN") ;

		fbuf = (int *)allocate (oparg * sizeof(int)) ;
		if (read (fid, fbuf, oparg * sizeof(int)) !=
				     (int)(oparg * sizeof(int)))
				      errorD ("end-of-file in function") ;

		disasm (fbuf, oparg) ;
		free   (fbuf) ;
	}
}

/*G main	: Standard entry routine				*/
/*  argc	: int		: Number of arguments			*/
/*  argv	: char *[]	: Vector of arguments			*/
/*  (returns)	: int		: Exit code				*/

GFUNC	int	main
	(	int	argc,
		char	*argv[]
	)
{
	char		fnam[256] ;
	int		fid	  ;
	unsigned long	magic	  ;

	if (argc != 2) errorE (USAGE) ;

	strcpy (fnam, argv[1]) ;

	if (strcmp (&fnam[strlen(fnam) - 4], ".elc") != 0)
		strcat (fnam, ".elc") ;

	if ((fid = open (fnam, O_RDONLY)) < 0)
		errorE ("eld: unable to open \"%s\": %e\n", fnam) ;

	if (read (fid, &magic, sizeof(long)) != sizeof(long))
		errorE ("eld: unable to read magic code: %e\n") ;

	if (magic != MAGIC)
		errorD ("magic code wrong: %08x", MAGIC) ;

	rdntab	(fid) ;
	rdfuncs	(fid) ;
	close	(fid) ;
	return	0     ;
}

