/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSUTILS_H
#define RDSUTILS_H

#include <QString>
#include <ReturnValue>
#include "rdsdefines.h"
#include <rdsglobal.h>

class RdsSid;
class RdsLdapSession;
class RdsNtAcl;

/**
	@author Brendan Powers <brendan@resara.com>
*/
class RDS_SHARED_EXPORT RdsUtils
{
public:
	~RdsUtils();

	enum PathType
	{
		System,
		Volume,
		Share
	};

	/*
	* PATH FUNCTIONS
	*/

	/**
	 * Return the file system path for a given RDS path
	 * @param path An RDS path
	 * @return Returns a file system path
	 */
	static ReturnValue getSystemPath(const QString &path);

	/**
	 * Return the share path for a given RDS path
	 * @param path An RDS path
	 * @return Returns a QStringList of share paths. One file can belong to more than one share
	 */
	static ReturnValue getSharePath(const QString &path);
	
	/**
	 * Return the volume path for a given RDS path
	 * @param path An RDS path
	 * @return Returns a volume
	 */
	static ReturnValue getVolumePath(const QString &path);
	
	/**
	 *  Determines the type of the given path
	 * @param path An RDS path
	 * @return The type of the given path
	 */
	static PathType pathType(const QString &path);

	/**
	 * Returns the relative part of an RDS path. For example, a path of @SHARE/MyShare/Users/TestUser would return
	*  /Users/TestUser
	 * @param path An RDS path
	 * @return the relative portion of an RDS path, or the full path if a system path was given
	 */
	static ReturnValue getRelativePath(const QString &path);

	/**
	 * Get the share name from an RDS path. For example, a path of @SHARE/MyShare/Users/TestUser would return
	 *  MyShare
	 * @param path an RDS path
	 * @return the share portion of the RDS path, or an error if a share path was not specified
	 */
	static ReturnValue getShareName(const QString &path);

	/**
	 * Get the volume name from an RDS path. For example, a path of @VOLUME/MyVolume/Users/TestUser would return
	 *  MyVolume
	 * @param path an RDS path
	 * @return the volume portion of the RDS path, or an error if a volume path was not specified
	 */
	static ReturnValue getVolumeName(const QString &path);

	/**
	 *  Removes duplicate slashes from a path. It also makes sure there is no slash at the end of the path
	 * @param path An RDS or file system path
	 * @return A normalized path
	 */
	static QString normalizePath(const QString path);




	/*
	* SID FUNCTIONS
	*/

	/**
	 * Returns the LDAP object that matches the given SID
	 * @param sid The SID of the object
	 * @return The DN of the object
	 */
	static ReturnValue getObjectBySid(RdsSid sid);

	/**
	 * Returns the SID for a given LDAP object
	 * @param dn The ldap DN of the object
	 * @return The SID of the object
	 */
	static ReturnValue getSidOfObject(QString dn);



	/*
	* DOMAIN FUNCTIONS
	*/

	/**
	 * Retrives the base DN for the domain.
	 * @return The base DN for the domain
	 */
	static QString baseDn();

	/**
	 *  Returns the kerberos realm for the domain.
	 * @return The kerberos realm
	 */
	static QString realm();

	/**
	 *  Returns the NT4 domain name of the domain
	 * @return The NT4 domain name
	 */
	static QString domain();

#ifdef __RDS_SERVER
	/**
	 *  Binds to an AD LDAP server as an administrator
	 * @return returns true on success, an error on failure
	 */
	static ReturnValue adminBind(RdsLdapSession &session);
#endif

#ifdef __RDS_CLIENT
	/**
	 *  Fetches the domain data from the server. This must be called afer the client connection is set up.
	 * @return returns true on success, an error on failure
	 */
	static ReturnValue init();
#endif

	/*
	* FILESYSTEM FUNCTIONS
	*/

	static ReturnValue followSymLink(const QString &path);
	static ReturnValue createDirectory(const QString &path, bool recursive = false);
	static ReturnValue remove(const QString &path, bool recursive = false);
	static ReturnValue move(const QString &path, const QString &newpath);
	static ReturnValue rename(const QString &path, const QString &newname);
	static ReturnValue fileContents(const QString &path);
	static ReturnValue setFileContents(const QString &path, const QByteArray &data);
	static ReturnValue unixPermissions(const QString &path);
	static ReturnValue setUnixPermissions(const QString &path, int mode);
	static ReturnValue owner(const QString &path);
	static ReturnValue setOwner(const QString &path, const QString &owner);
	static ReturnValue setOwner(const QString &path, int owner);
	static ReturnValue group(const QString &path);
	static ReturnValue setGroup(const QString &path, const QString &group);
	static ReturnValue setGroup(const QString &path,  int group);
	static ReturnValue ntPermissions(const QString &path);
	static ReturnValue setNtPermissions(const QString &path, const RdsNtAcl &permissions);

	/*
	* Misc functions
	*/
	static ReturnValue runCommand(const QString &command, const QStringList &args);

protected:
	RdsUtils();

};

Q_DECLARE_METATYPE(RdsUtils::PathType);

#endif
