/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsstoragedevice.h"
#include "rdsstoragedevice_p.h"

#include "RdsVolume"
#include "RdsVolumeManager"

#include "rdsvolume_p.h"

#include <QFile>
#include <QFileInfo>
#include <QTemporaryFile>
#include <QProcess>
#include <QDebug>

#include <sys/types.h>
#include <attr/xattr.h>
#include <sys/statfs.h>

using namespace QtRpc;

QTRPC_SERVICEPROXY_PIMPL_IMPLEMENT(RdsStorageDevice);

#define CREATE_DBUS_OBJECT RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), qxt_d().path, QDBusConnection::systemBus());

bool RdsStorageDevicePrivate::isStorageDevice(RdsUDisksDeviceInterface* device)
{
	return (device->deviceIsDrive() && !device->deviceIsLinuxLoop() && device->deviceFile() != "/dev/fd0");
}

bool RdsStorageDevicePrivate::isVolume(RdsUDisksDeviceInterface* device)
{
	return ((device->deviceIsPartition() || !device->idType().isEmpty()) && !device->deviceIsLinuxLoop() && device->partitionType() != "0x05");
}

RdsStorageDevice::RdsStorageDevice(const char * device)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsStorageDevice);
	qxt_d().path = device;
}

RdsStorageDevice::RdsStorageDevice(const QString &device)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsStorageDevice);
	qxt_d().path = device;
}

RdsStorageDevice::RdsStorageDevice(const RdsStorageDevice &other)
		: ServiceProxy(NULL)
{
	QXT_INIT_PRIVATE(RdsStorageDevice);
	qxt_d().path = other.qxt_d().path;
}

RdsStorageDevice::~RdsStorageDevice()
{
}

ReturnValue RdsStorageDevice::path() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	return device.path();
}

ReturnValue RdsStorageDevice::listVolumes() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	if (RdsVolumePrivate::isVolume(device))
		return device.idUuid();
	QStringList list;
	RdsUDisksInterface interface(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());
	foreach(QString deviceFile, qxt_d().getPartitions(device.deviceFile()))
	{
		QDBusPendingReply<QDBusObjectPath> ret = interface.FindDeviceByDeviceFile(deviceFile);
		ret.waitForFinished();
		if (!ret.isError())
		{
			RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), ret.value().path(), QDBusConnection::systemBus());
			if (RdsStorageDevicePrivate::isVolume(&device))
				list << device.path();
		}
	}
	return list;
}

ReturnValue RdsStorageDevice::initialize()
{
	return initialize(true, "ext3");
}

ReturnValue RdsStorageDevice::initialize(bool partitionTable, const QString &fstype)
{
	foreach(QString uuid, listVolumes().toStringList())
	{
		RdsVolume v(uuid);
		ReturnValue ret = v.isMounted();
		if (ret.isError())
			return ret;
		if (ret.toBool())
			return ReturnValue(1, "You cannot initialize a device while any of it's volumes are mounted.");

		RdsVolumeManager().removeEntity(uuid);

		ret = v.isAdded();
		if (ret.isError())
			return ret;
		if (!ret.toBool())
			continue;

		ret = v.name();
		if (ret.isError())
			return ret;

		RdsVolumeManager().removeVolume(ret.toString());
	}
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	QString devfile = device.deviceFile();
	if (partitionTable)
	{
		QSharedPointer<PedDevice> ped(qxt_d().getDevice(device.deviceFile()));
		if (!ped)
			return ReturnValue(1, "Failed to find device in parted");
		ReturnValue ret = RdsStorageDevicePrivate::createSinglePartition(ped.data());
		if (ret.isError())
		{
			return ret;
		}
		devfile = ret.toString();
	}
	ReturnValue ret = RdsStorageDevicePrivate::createFilesystem(devfile, fstype);
	if (ret.isError())
	{
		return ret;
	}
	QString newId;
	for (int i = 0; i < 50; ++i)
	{
		newId = RdsVolume(listVolumes().toStringList().first()).id().toString();
		if (!newId.isEmpty())
			break;
		usleep(100000); // :<
	}

	if (newId.isEmpty())
		return ReturnValue(1, "Failed to detect the newly initialized device.");
	RdsVolume(newId).mount();
	RdsVolumeManager().addEntity(newId);
	return newId;
}

ReturnValue RdsStorageDevice::isEjectable() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	return device.driveIsMediaEjectable();
}

ReturnValue RdsStorageDevice::eject()
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	QDBusPendingReply<> ret = device.DriveEject(QStringList());
	ret.waitForFinished();
	if (ret.isError())
	{
		QDBusError err = ret.error();
		return ReturnValue(err.type(), err.message());
	}
	return true;
}

ReturnValue RdsStorageDevice::size() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	return device.deviceSize();
}

ReturnValue RdsStorageDevice::type() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	QString type = device.driveConnectionInterface();
	if (device.deviceIsOpticalDisc())
		type = "CDROM";
	else if (type == "usb")
		type = "USB Disk";
	else if (type == "virtual")
		type = "Virtual Disk";
	else if (type == "")
		type = "Virtual Disk";
	else
		type = "Hard Disk";
	/*
	if (type == "linux_raid_member")
	type = "Raid Member";
	else if (type == "LVM2_member")
	type = "LVM2 Member";
	else if (type == "swap")
	type = "Swap";
	else if (type == "ext3")
	type = "Hard Disk";
	*/
	return type;
}

ReturnValue RdsStorageDevice::name() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	return device.deviceFile();
}

ReturnValue RdsStorageDevice::serialNumber() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	return device.driveSerial();
}

ReturnValue RdsStorageDevice::model() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;
	return device.driveModel();
}

ReturnValue RdsStorageDevice::inUse() const
{
	if (qxt_d().path.isEmpty())
		return ReturnValue(1, "Invalid storage device");
	CREATE_DBUS_OBJECT;

	//Device is part of an LVM volume
	if (device.deviceIsLinuxLvm2PV())
		return(true);
	//Device is part of a software raid volume
	else if (device.deviceIsLinuxMd())
		return(true);
	else if (device.deviceIsLinuxMdComponent())
		return(true);
	//Device is part of a device mapper
	else if (device.deviceIsLinuxDmmpComponent())
		return(true);
	else if (device.deviceIsLinuxDmmp())
		return(true);
	if (device.deviceSize() == 0) return(true);


	RdsVolumeManager mgr;

	foreach(QString vid, listVolumes().toStringList())
	{
		//This is intentional
		RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), vid, QDBusConnection::systemBus());
		//Device is part of an LVM volume
		if (device.deviceIsLinuxLvm2PV())
			return(true);
		//Device is part of a software raid volume
		else if (device.deviceIsLinuxMdComponent())
			return(true);
		//Device is part of a device mapper
		else if (device.deviceIsLinuxDmmpComponent())
			return(true);
		else if (device.deviceIsLinuxDmmp())
			return(true);

		ReturnValue ret = mgr.volume(vid);
		if (ret.isError()) continue;

		RdsVolume volume;
		volume = ret;

		if (volume.isMounted().toBool()) return(true);
	}

// 	qDebug() << "None of these!" << qxt_d().path;

	//Not in use
	return(false);
}

ReturnValue RdsStorageDevice::list()
{
	QStringList list;
	RdsUDisksInterface interface(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());
	QDBusPendingReply<QList<QDBusObjectPath> > ret = interface.EnumerateDevices();
	ret.waitForFinished();
	foreach(QDBusObjectPath path, ret.value())
	{
		RdsUDisksDeviceInterface device(RdsUDisksInterface::staticInterfaceName(), path.path(), QDBusConnection::systemBus());
		if (RdsStorageDevicePrivate::isStorageDevice(&device))
			list << device.path();
	}
	return list;
}

RdsStorageDevice& RdsStorageDevice::operator=(const RdsStorageDevice & other)
{
	Q_UNUSED(other);
	return *this;
}

//Get the partimage device for a given device file(ie /dev/sda)
QSharedPointer<PedDevice> RdsStorageDevicePrivate::getDevice(const QString &path)
{
	//probe devices
	return QSharedPointer<PedDevice>(ped_device_get(qPrintable(path)), ped_device_destroy);
}

QStringList RdsStorageDevicePrivate::getPartitions(const QString &drive)
{
	return getPartitions(getDevice(drive).data());
}

QStringList RdsStorageDevicePrivate::getPartitions(PedDevice *dev)
{
	QStringList list;

	if (!dev)
		return list;

	QSharedPointer<PedDisk> disk(ped_disk_new(dev), ped_disk_destroy);
	if (disk.isNull())
		return list; //skip invalid disks

	PedPartition* partition(0);
	while ((partition = ped_disk_next_partition(disk.data(), partition)))
	{
		//skip partitions that would not contain a file system
		if ((partition->type != PED_PARTITION_LOGICAL) && (partition->type != PED_PARTITION_NORMAL))
			continue;

		list << ped_partition_get_path(partition);
	}

	return(list);
}

ReturnValue RdsStorageDevicePrivate::removePartitionTable(PedDevice *dev)
{
	if (!dev)
		return ReturnValue(1, "Invalid parted device object");
//	const PedDiskType* type = getGoodType();
	QSharedPointer<PedDisk> disk(ped_disk_new_fresh(dev, getGoodType()), ped_disk_destroy);
	if (ped_disk_commit_to_dev(disk.data()) == 0)
	{
		return ReturnValue(1, "Failed to commit changes to disk!");
	}
	return true;
}

ReturnValue RdsStorageDevicePrivate::createSinglePartition(PedDevice *dev)
{
	if (!dev)
		return ReturnValue(1, "Invalid parted device object");
	QProcess::execute("dd", QStringList() << "if=/dev/zero" << QString("of=%1").arg(dev->path) << "bs=1M" << "count=5");

	QSharedPointer<PedDisk> disk(ped_disk_new_fresh(dev, getGoodType()), ped_disk_destroy);
	if (disk.isNull())
		return ReturnValue(1, "Failed to create partition table");

	//create partition
	PedPartition* part(ped_partition_new(disk.data(), PED_PARTITION_NORMAL, ped_file_system_type_get("ext2"), 1, dev->length - 1));
	if (!part)
	{
		return ReturnValue(1, "Failed to create partition");
	}

	//add the partition to the disk
	if (ped_disk_add_partition(disk.data(), part, ped_constraint_new_from_max(&part->geom)) == 0)
	{
		return ReturnValue(1, "Failed to add partition to disk");
	}

	//commit changes to disk
	if (ped_disk_commit_to_dev(disk.data()) == 0)
	{
		return ReturnValue(1, "Failed to save changes");
	}

	//notify the OS to reload partition table
	if (ped_disk_commit_to_os(disk.data()) == 0)
	{
		return ReturnValue(1, "Failed to update OS");
	}
	return (dev->path + QString::number(part->num));
}

ReturnValue RdsStorageDevicePrivate::setPartitionType(const QString &dev, const QString &fstype)
{
	QSharedPointer<PedDevice> device(getDevice(dev.left(dev.lastIndexOf(QRegExp("[^0-9]")) + 1)));
	if (device.isNull())
		return ReturnValue(1, "Failed to fetch device");
	PedDisk* disk = ped_disk_new(device.data());
	PedPartition* part = ped_disk_get_partition(disk, dev.mid(dev.lastIndexOf(QRegExp("[^0-9]")) + 1).toInt());
	QString found = QString("%1%2").arg(device->path).arg(part->num);
	if (found != dev)
	{
		qCritical() << "Looking for" << dev << "found" << found;
		return ReturnValue(1, "Failed to find partition");
	}
	if (!ped_partition_set_system(part, ped_file_system_type_get(qPrintable(fstype))))
		return ReturnValue(1, "Failed to change partition type");

	//commit changes to disk
	if (ped_disk_commit_to_dev(disk) == 0)
	{
		return ReturnValue(1, "Failed to save changes");
	}

	//notify the OS to reload partition table
	if (ped_disk_commit_to_os(disk) == 0)
	{
		return ReturnValue(1, "Failed to update OS");
	}

	return true;
}

ReturnValue RdsStorageDevicePrivate::createFilesystem(const QString &dev, const QString &fstype)
{
	QProcess::execute("dd", QStringList() << "if=/dev/zero" << QString("of=%1").arg(dev) << "bs=1M" << "count=5");
	RdsUDisksInterface interface(RdsUDisksInterface::staticInterfaceName(), "/org/freedesktop/UDisks", QDBusConnection::systemBus());

	QDBusPendingReply<QDBusObjectPath> ret = interface.FindDeviceByDeviceFile(dev);
	ret.waitForFinished();
	if (ret.isError())
		return ReturnValue(1, "Failed to find device");

	QStringList options;
	if (fstype.startsWith("ext"))
	{
		options << "-F";
	}

	if (QProcess::execute("mkfs", QStringList() << "-t" << fstype << options << dev) != 0)
		return ReturnValue(1, "Failed to create filesystem");

	return true;
}

const PedDiskType* RdsStorageDevicePrivate::getGoodType()
{
	static const PedDiskType* type = ped_disk_type_get("msdos");
	return type;
}

