/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsentitymanager.h"
#include "rdsentitymanager_p.h"
#include <QMetaObject>
#include <QDebug>
#include <QStringList>

using namespace QtRpc;

QMutex& RdsEntityManagerPrivate::mutex()
{
	static QMutex mutex;
	return mutex;
}

QHash<QString, EntityType *>& RdsEntityManagerPrivate::types()
{
	static QHash<QString, EntityType *> types;
	return types;
}

RdsEntityManager::RdsEntityManager(QObject *parent)
		: ServiceProxy(parent)
{
	QXT_INIT_PRIVATE(RdsEntityManager);
}

void RdsEntityManager::createInternalObject()
{
	QMutexLocker lock(&qxt_d().mutex());
	if ((!qxt_d().types().contains(metaObject()->className())) || (qxt_d().types()[metaObject()->className()] == NULL))
	{
		qxt_d().types()[metaObject()->className()] = new EntityType(NULL);
	}

	EntityType *type =  qxt_d().types()[metaObject()->className()];
	QObject::connect(type, SIGNAL(entityAdded(QString)), this, SIGNAL(entityAdded(QString)));
	QObject::connect(type, SIGNAL(entityRemoved(QString)), this, SIGNAL(entityRemoved(QString)));
	QObject::connect(type, SIGNAL(entityRenamed(QString, QString)), this, SIGNAL(entityRenamed(QString, QString)));
	QObject::connect(type, SIGNAL(entityUpdated(QString)), this, SIGNAL(entityUpdated(QString)));
	QObject::connect(type, SIGNAL(entityMetadataUpdated(QString,QVariantMap)), this, SIGNAL(entityMetadataUpdated(QString,QVariantMap)));
	QObject::connect(type, SIGNAL(editingStarted(QString, RdsEntityManager*)), this, SLOT(editingStarted(QString, RdsEntityManager*)));
	QObject::connect(type, SIGNAL(editingStopped(QString, RdsEntityManager*)), this, SLOT(editingStopped(QString, RdsEntityManager*)));
	QObject::connect(this, SIGNAL(destroyed(QObject*)), type, SLOT(managerDestroyed(QObject *)), Qt::QueuedConnection);
}

RdsEntityManager::RdsEntityManager(const RdsEntityManager &object)
		: ServiceProxy(object.parent())
{
	qxt_d() = object.qxt_d();
}

RdsEntityManager::~RdsEntityManager()
{
}

RdsEntityManager &RdsEntityManager::operator = (const RdsEntityManager & object)
{
	qxt_d() = object.qxt_d();
	return(*this);
}

ReturnValue RdsEntityManager::addEntity(const QString &entity)
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(false);
	qxt_d().types()[metaObject()->className()]->addEntity(entity);
	return(true);
}

ReturnValue RdsEntityManager::removeEntity(const QString &entity)
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(false);
	qxt_d().types()[metaObject()->className()]->removeEntity(entity);
	return(true);
}

ReturnValue RdsEntityManager::renameEntity(const QString &oldentity, const QString &newentity)
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(false);
	qxt_d().types()[metaObject()->className()]->renameEntity(oldentity, newentity);
	return(true);
}

ReturnValue RdsEntityManager::updateEntity(const QString &entity)
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(false);
	qxt_d().types()[metaObject()->className()]->updateEntity(entity);
	return(true);
}

ReturnValue RdsEntityManager::updateEntityMetadata(const QString& entity, const QVariantMap& metadata)
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(false);
	qxt_d().types()[metaObject()->className()]->updateEntityMetadata(entity, metadata);
	return(true);
}

QtRpc::ReturnValue RdsEntityManager::startEditing(const QString &entity)
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(false);
	qxt_d().types()[metaObject()->className()]->startEditing(entity, this);
	return(true);
}

QtRpc::ReturnValue RdsEntityManager::stopEditing(const QString &entity)
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(false);
	qxt_d().types()[metaObject()->className()]->stopEditing(entity, this);
	return(true);
}

void RdsEntityManager::editingStarted(const QString &entity, RdsEntityManager *object)
{
	emit(editingStarted(entity));
}

ReturnValue RdsEntityManager::listEntities(const QString &base) const
{
	return(listEntities(base, false));
}

void RdsEntityManager::editingStopped(const QString &entity, RdsEntityManager *object)
{
	emit(editingStopped(entity));
}

ReturnValue RdsEntityManager::init()
{
	createInternalObject();
	return(true);
}

ReturnValue RdsEntityManager::editingList()
{
	if (qxt_d().types()[metaObject()->className()] == NULL) return(QStringList());
	return(QStringList(qxt_d().types()[metaObject()->className()]->_editing.values()));
}


void EntityType::addEntity(const QString &entity)
{
	emit(entityAdded(entity));
}

void EntityType::removeEntity(const QString &entity)
{
	emit(entityRemoved(entity));
}

void EntityType::renameEntity(const QString &oldentity, const QString &newentity)
{
	emit(entityRenamed(oldentity, newentity));
}

void EntityType::updateEntity(const QString &entity)
{
	emit(entityUpdated(entity));
}

void EntityType::updateEntityMetadata(const QString& entity, const QVariantMap& metadata)
{
	emit(entityMetadataUpdated(entity, metadata));
}

void EntityType::startEditing(const QString &entity, RdsEntityManager *object)
{
	_editing.insert(object, entity);
	emit(editingStarted(entity, object));
}

void EntityType::stopEditing(const QString &entity, RdsEntityManager *object)
{
	_editing.remove(object, entity);
	if (!_editing.values().contains(entity)) emit(editingStopped(entity, object));
}

void EntityType::managerDestroyed(QObject *mgr)
{
	if (mgr == NULL) return;
	foreach(QString entity, _editing.values((RdsEntityManager *)mgr))
	{
		emit(editingStopped(entity, (RdsEntityManager *)mgr));
	}
	_editing.remove((RdsEntityManager *)mgr);
}
