/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#ifndef RDSDHCPVALUES_H
#define RDSDHCPVALUES_H

#include <ServiceProxy>
#include <QxtPimpl>

class RdsDhcpValuesPrivate;

/**
	@author Chris Vickery <chris@resara.com>
*/
class RdsDhcpValues : public QtRpc::ServiceProxy
{
	friend class RdsDhcpManager;
	Q_OBJECT;
	QXT_DECLARE_PRIVATE(RdsDhcpValues);
	QTRPC_SERVICEPROXY_PIMPL(RdsDhcpValues);
public:
	RdsDhcpValues(const RdsDhcpValues &other);
	~RdsDhcpValues();
	RdsDhcpValues& operator=(const RdsDhcpValues &other);
	virtual void setData(void* data);

protected:
	RdsDhcpValues();

public slots:
	/**
	 *        Returns the config values
	 * @return QVariantMap with values
	 */
	virtual ReturnValue values() const;
	/**
	 *        Sets configuration values overwriting existing values, and merging options
	 * @param map the map containing the values
	 * @return true
	 */
	virtual ReturnValue setValues(const QVariantMap &map);
	/**
	 *        Gets the value associated with the key
	 * @param key the key to get the value of
	 * @return QString (it's really a QVariant).  Empty if key is not found.
	 */
	virtual ReturnValue value(const QString &key) const;
	/**
	 *        Sets a configuration value
	 * @param key value key
	 * @param value the value
	 * @return true
	 */
	virtual ReturnValue setValue(const QString &key, const QString &value);
	/**
	*        Returns the existence of a value (can NOT remove options)
	* @param key the key of the value to remove
	* @return true if the value exists, false if not
	*/
	virtual ReturnValue valueExists(const QString &key);
	/**
	 *        Removes a value from the configuration (can NOT remove options)
	 * @param key the key of the value to remove
	 * @return true on success, error on failure
	 */
	virtual ReturnValue removeValue(const QString &key);
	/**
	 *        Gets a the map of options for the configuration
	 * @return QVariantMap
	 */
	virtual ReturnValue options() const;
	/**
	 *        Sets the optoins for the configuration overwriting existing values
	 * @param map the map of options
	 * @return true
	 */
	virtual ReturnValue setOptions(const QVariantMap &map);
	/**
	 *        Gets an option
	 * @param key the key of the option to get
	 * @return QString (it's really a QVariant)
	 */
	virtual ReturnValue option(const QString &key) const;
	/**
	 *        Sets an option value
	 * @param key the key of the option
	 * @param value the value of the option
	 * @return true
	 */
	virtual ReturnValue setOption(const QString &key, const QString &value);
	/**
	*        Returns the existence of an option
	* @param key the key of the option to remove
	* @return true if the option exists, false otherwise
	*/
	virtual ReturnValue optionExists(const QString &key);
	/**
	 *        Removes an option from the configuration
	 * @param key the key of the option to remove
	 * @return true
	 */
	virtual ReturnValue removeOption(const QString &key);
	/**
	 *        Gets a list of the groups
	 * @return QStringList of group names
	 */
	virtual ReturnValue groups() const;
	/**
	 *        Gets a list of names of all the subnets
	 * @return QStringList of subnets
	 */
	virtual ReturnValue subnets() const;
	/**
	 *        Gets a list of shared networks
	 * @return QStringList of shared networks
	 */
	virtual ReturnValue sharedNetworks() const;
	/**
	 *        Gets a list of hosts
	 * @return QStringList of hosts
	 */
	virtual ReturnValue hosts() const;
	/**
	 *        Gets RdsDhcpSharedNetwork object for sharedNetwork
	 * @param sharedNetwork the name of the shared network to get
	 * @return RdsDhcpSharedNetwork on success, error on failure
	 */
	virtual ReturnValue sharedNetwork(const QString &sharedNetwork) const;
	/**
	 *        Gets RdsDhcpHost object with hostname
	 * @param hostname the name of the host to get
	 * @return RdsDhcpHost on success, error on failure
	 */
	virtual ReturnValue host(const QString &hostname) const;
	/**
	 *        Gets RdsDhcpGroup object with group name an information
	 * @param group the name of the group to get
	 * @return RdsDhcpGroup on success, error on failure
	 */
	virtual ReturnValue group(const QString &group) const;
	/**
	 *        Gets RdsDhcpSubnet object for subnet
	 * @param subnet the name of the subnet to get
	 * @return RdsDhcpSubnet on success, error on failure
	 */
	virtual ReturnValue subnet(const QString &subnet) const;
	/**
	 *        Adds a shared network to the configuration
	 * @param sharedNetwork the name of the shared network
	 * @return RdsDhcpSharedNetwork object with the shared network name
	 */
	virtual ReturnValue addSharedNetwork(const QString &sharedNetwork);
	/**
	 *        Adds a host to the configuration
	 * @param hostname the name of the host to add
	 * @return RdsDhcpHost object with the host name
	 */
	virtual ReturnValue addHost(const QString &hostname);
	/**
	 *        Adds a group to the configuration
	 * @param group the name of the group to add
	 * @return RdsDhcpGroup object with the group name 
	 */
	virtual ReturnValue addGroup(const QString &group);
	/**
	 *        Adds a subnet to the configuration
	 * @param subnet the name of the subnet to add
	 * @return RdsDhcpSubnet object with the subnet name
	 */
	virtual ReturnValue addSubnet(const QString &subnet);
	/**
	 *        Removes a share network from the configuration
	 * @param sharedNetwork the shared network to remove
	 * @return true
	 */
	virtual ReturnValue removeSharedNetwork(const QString &sharedNetwork);
	/**
	 *        Removes a host from the configuration
	 * @param hostname the host to remove
	 * @return true
	 */
	virtual ReturnValue removeHost(const QString &hostname);
	/**
	 *        Removes a group from the configuration
	 * @param group the group to remove
	 * @return true
	 */
	virtual ReturnValue removeGroup(const QString &group);
	/**
	 *        Removes a subnet from the configuration
	 * @param subnet the subnet to remove
	 * @return true
	 */
	virtual ReturnValue removeSubnet(const QString &subnet);
};

#endif
