/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU Lesser General Public License as         *
 *  published by the Free Software Foundation; either version 2 of the     *
 *  License, or (at your option) any later version.                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  Lesser General Public License for more details.                        *
 *                                                                         *
 *  You should have received a copy of the GNU Lesser General Public       *
 *  License along with this program; if not, write to the                  *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
***************************************************************************/
#ifndef RDSLDAPSESSION_H
#define RDSLDAPSESSION_H

#include <QObject>
#include <QHash>
#include <QString>
#include <QMetaType>
#include <QxtPimpl>
#include <ReturnValue>
#include <QStringList>
#include <ServiceProxy>
#include <RdsLdapActions>

struct ldap;
typedef ldap LDAP;




/**
	@author Brendan Powers <brendan@resara.com>
*/
class RdsLdapSession : public QtRpc::ServiceProxy
{
	QXT_DECLARE_PRIVATE(RdsLdapSession);
	Q_OBJECT
public:
	RdsLdapSession(QObject *parent = 0);
	RdsLdapSession(QString uri, QObject *parent = 0);
	~RdsLdapSession();

	/**
	 *         Binds to the ldap server anonymously. Only read operations allowed.
	 * @return True on success, check error string on failure
	 */
	virtual ReturnValue anonymousBind();
	/**
	*        Binds to the ldap server with authentication allowing write operations
	* @param dn the fulll DN of the user
	* @param password you know what this is...
	* @return True on success, check error string on failure
	*/
	virtual ReturnValue simpleBind(const QString &dn, const QString &password);
	/**
	 *        Binds to the ldap server with authentication allowing write operations
	 * @param user username
	 * @param password you know what this is...
	 * @return True on success, check error string on failure
	 */
	virtual ReturnValue bind(const QString &user, const QString &password);

	virtual ReturnValue kerberosBind();

	LDAP *handle();

	RdsLdapSession &operator =(const RdsLdapSession &object);

	/**
	 *        Gets a pointer to the global ldap session
	 * @return RdsLdapSession pointer
	 */
	static RdsLdapSession *globalSession();
	/**
	 *        Sets the global session pointer
	 * @param session A pointer to the new global ldap session
	 */
	static void setGlobalSession(RdsLdapSession *session);
	
	/**
	*        Returns true if the global session has been set
	* @return true/false
	*/
	static bool isGlobalSessionSet();

public slots:
	/**
		 *        Searches for objects in the basedn that match the filter (recursive)
		 * @param basedn the dn of the object to search in
		 * @param filter the search filter
		 * @return LdapResults containing the results of the search
		 */
	ReturnValue search(const QString &basedn, const QString &filter) const;
	/**
	 *        Searches for objects in the basedn that match the filter
	 * @param basedn the dn of the object to search in
	 * @param filter the search filter
	 * @param recursive whether the search should be recursive or not
	 * @return Ldap Results containing the results of the search
	 */
	ReturnValue search(const QString &basedn, const QString &filter, bool recursive) const;
	/**
	 *        Reads the values for the attributes of the object
	 * @param basedn the dn of the object to read the attributes from
	 * @return LdapResult containing the attrbutes and values
	 */
	ReturnValue read(const QString &basedn) const;
	/**
	 *        Reads the only the values for the attributes contained in attrs
	 * @param basedn the dn of the object to read the attributes from
	 * @param attrs list of attributes to read
	 * @return LdapResult containing the attributes and values
	 */
	ReturnValue read(const QString &basedn, const QStringList &attrs) const;
	/**
	 *        Lists the objects contained in the basedn (recursive)
	 * @param basedn the dn to list the contents of
	 * @return QStringList of dn's contained in the base dn
	 */
	ReturnValue list(const QString &basedn) const;
	/**
	 *        Lists the objects contained in the basedn that match the search filter (recursive)
	 * @param basedn the dn to list the matches from
	 * @param filter the search filter
	 * @return QString list of the dn's that matched the filter
	 */
	ReturnValue list(const QString &basedn, const QString &filter) const;
	/**
	 *        Lists the objects contained in the basedn that match the search filter
	 * @param basedn the dn to list the matches from
	 * @param filter the search filter
	 * @param recursive wheather to search recursively or not
	 * @return QStringList of dn's that matched the search criteria
	 */
	ReturnValue list(const QString &basedn, const QString &filter, bool recursive) const;
	/**
	 *        Adds the actions to a new object with distinguished name dn
	 * @param dn the distinguished name of the new object
	 * @param actions the actions to perfomr on the new object
	 * @return True on success, check error string on failure
	 */
	ReturnValue add(const QString &dn, const RdsLdapActions &actions);
	/**
	 *        Modifies an existing object
	 * @param dn the dn of the object to modify
	 * @param actions the changes to make
	 * @return True on success, check error string on failure
	 */
	ReturnValue modify(const QString &dn, const RdsLdapActions &actions);
	/**
	 *        Removes an object from the directory
	 * @param dn the dn of the object to remove
	 * @param recursive also delete child objects
	 * @return True on success, chech error string on failure
	 */
	ReturnValue remove(const QString &dn, bool recursive = false);
	/**
	 *        Re-names an object
	 * @param dn dn of the object to rename
	 * @param rdn the new rdn of the object
	 * @return True on success, check error string on failure
	 */
	ReturnValue rename(const QString &dn, const QString &rdn);
	/**
	 *        Moves an object to a new location in the directory
	 * @param dn the dn of the object to move
	 * @param newparent the dn of the new location in the directory
	 * @return True on success, check error string on failure
	 */
	ReturnValue move(const QString &dn, const QString &newparent);

private:
	ReturnValue reAuthenticate();
};

typedef QList<QByteArray> LdapValues;
typedef QHash<QString, LdapValues> LdapResult;
typedef QHash<QString, LdapResult> LdapResults;
Q_DECLARE_METATYPE(LdapResults);
Q_DECLARE_METATYPE(LdapResult);
Q_DECLARE_METATYPE(LdapValues);

inline RdsLdapSession *rdsLdapSession()
{
	return(RdsLdapSession::globalSession());
}

#endif
