/***************************************************************************
 *  Copyright (C) 2011 by Resara LLC                                       *
 *  brendan@resara.com                                                     *
 *                                                                         *
 *  This program is free software; you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation; either version 2 of the License, or (at  *
 *  your option) any later version.                                        *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful, but    *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU      *
 *  General Public License for more details.                               *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program; if not, write to the                          *
 *  Free Software Foundation, Inc.,                                        *
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                         *
 ***************************************************************************/
#include "rdsentitywidget.h"
#include "rdsentitywidget_p.h"
#include <QPushButton>
#include <QComboBox>
#include <QLineEdit>
#include <QTextEdit>
#include <QSpinBox>
#include <QDoubleSpinBox>
#include <QTimeEdit>
#include <QDateEdit>
#include <QDateTimeEdit>
#include <QDial>
#include <QSlider>
#include <QAbstractButton>
#include <QDebug>
#include <QMessageBox>
#include <QPlainTextEdit>
#include <QLabel>
#include <QRegExpValidator>


RdsEntityWidget::RdsEntityWidget(RdsEntityManager *manager, QWidget *parent)
		: QWidget(parent)
{
	qxt_d().manager = manager;
	QObject::connect(manager, SIGNAL(entityUpdated(QString)), this, SLOT(entityUpdated(QString)));
}


RdsEntityWidget::~RdsEntityWidget()
{
	if (unsavedChanges())
	{
		foreach(QString output, RdsEntityWidget::outputs())
		{
			manager()->stopEditing(output);
		}
	}
}

QPushButton *RdsEntityWidget::applyButton()
{
	return(qxt_d().applybutton);
}

void RdsEntityWidget::setApplyButton(QPushButton *button)
{
	button->setEnabled(false);
	QObject::connect(button, SIGNAL(clicked()), this, SLOT(apply()));
	qxt_d().applybutton = button;
}

QPushButton *RdsEntityWidget::discardButton()
{
	return(qxt_d().discardbutton);
}

void RdsEntityWidget::setDiscardButton(QPushButton *button)
{
	button->setEnabled(false);
	QObject::connect(button, SIGNAL(clicked()), this, SLOT(discard()));
	qxt_d().discardbutton = button;
}

void RdsEntityWidget::addWidget(QString name, QLabel *w, Flags flags)
{
	addWidget(name, w, "text", NULL, flags, false);
}

void RdsEntityWidget::addWidget(QString name, QAbstractButton *w, Flags flags)
{
	addWidget(name, w, "checked", SIGNAL(released()), flags, false);
}

void RdsEntityWidget::addWidget(QString name, QComboBox *w, Flags flags)
{
	bool editable = false;
	if (w->isEditable()) editable = true;
	w->setEditable(true);
	if (!editable) w->lineEdit()->setReadOnly(true);
	addWidget(name, w->lineEdit(), "text", SIGNAL(textChanged(QString)), flags, "");
}

void RdsEntityWidget::addWidget(QString name, QLineEdit *w, Flags flags, QString regex, QString errtext)
{
	addWidget(name, w, "text", SIGNAL(textEdited(QString)), flags, "");
	if(regex != "")
	{
		w->setValidator(new QRegExpValidator(QRegExp(regex),this));
		qxt_d().fields[name].errtext = errtext;
	}
}

void RdsEntityWidget::addWidget(QString name, QTextEdit *w, Flags flags)
{
	addWidget(name, w, "plainText", SIGNAL(textChanged()), flags, "");
}

void RdsEntityWidget::addWidget(QString name, QPlainTextEdit *w, Flags flags)
{
	addWidget(name, w, "plainText", SIGNAL(textChanged()), flags, "");
}

void RdsEntityWidget::addWidget(QString name, QSpinBox *w, Flags flags)
{
	addWidget(name, w, "value", SIGNAL(valueChanged(int)), flags, 0);
}

void RdsEntityWidget::addWidget(QString name, QDoubleSpinBox *w, Flags flags)
{
	addWidget(name, w, "value", SIGNAL(valueChanged(double)), flags, 0.0);
}

void RdsEntityWidget::addWidget(QString name, QDateTimeEdit *w, Flags flags)
{
	addWidget(name, w, "datetime", SIGNAL(dateTimeChanged(const QDateTime)), flags, 0);
}

void RdsEntityWidget::addWidget(QString name, QAbstractSlider *w, Flags flags)
{
	addWidget(name, w, "value", SIGNAL(valueChanged(int)), flags, 0);
}

void RdsEntityWidget::addWidget(QString name, QObject *w, QString property, const char * signal, Flags flags,  QVariant null)
{
	RdsEntityWidgetPrivate::Field field;
	field.name = name;
	field.property = property;
	field.flags = flags;
	field.def = null;
	field.enabled = true;

	if (signal != NULL) QObject::connect(w, signal, this, SLOT(fieldChanged()));
	qxt_d().fields[name] = field;
	qxt_d().widgets[name] = w;
	qxt_d().fieldmap[w] = name;
}

void RdsEntityWidget::fieldChanged()
{
	if (sender() == NULL) return;
	fieldChanged(sender());
}

void RdsEntityWidget::fieldChanged(QObject *widget)
{
	if (!qxt_d().fieldmap.contains(widget)) return;

	QString name = qxt_d().fieldmap[widget];
	if (!qxt_d().fields.contains(name)) return;

	RdsEntityWidgetPrivate::Field field = qxt_d().fields[name];

	//qDebug() << "Field Changed:" << name << field.value << widget->property(qPrintable(field.property)) <<
	(field.value == widget->property(qPrintable(field.property)));

	if (field.value == widget->property(qPrintable(field.property)))
	{
		if (qxt_d().modifiedfields.contains(name)) qxt_d().modifiedfields.removeAll(name);
		if (qxt_d().modifiedfields.count() == 0)
		{
			qxt_d().applybutton->setEnabled(false);
			qxt_d().discardbutton->setEnabled(false);

			foreach(QString output, outputs())
			{
				manager()->stopEditing(output);
			}
		}
	}
	else
	{
		if (!qxt_d().modifiedfields.contains(name)) qxt_d().modifiedfields << name;
		if (qxt_d().modifiedfields.count() == 1)
		{
			qxt_d().applybutton->setEnabled(true);
			qxt_d().discardbutton->setEnabled(true);
			foreach(QString output, outputs())
			{
				manager()->startEditing(output);
			}
		}
	}
}

void RdsEntityWidget::setInput(QString input)
{
	qxt_d().input = input;
	ReturnValue ret = getData();
	if (ret.isError())
	{
		qWarning() << "Warning: failed to get data:" << ret.errString();
		setEnabled(false);
		return;
	}

	QVariantMap map = ret.toMap();
	foreach(QString name, map.keys())
	{
		QVariant value = map[name];
		if (!qxt_d().fields.contains(name)) continue;
		RdsEntityWidgetPrivate::Field &field = qxt_d().fields[name];
		if (value == QVariant())
		{
			field.value = field.def;
		}
		else
		{
			field.value = value;
		}
	}

	foreach(QString name, map.keys())
	{
		QVariant value = map[name];
		if (!qxt_d().fields.contains(name)) continue;
		RdsEntityWidgetPrivate::Field &field = qxt_d().fields[name];
		qxt_d().widgets[name]->setProperty(qPrintable(field.property), field.value);
	}
}

void RdsEntityWidget::setOutputs(QStringList outputs)
{
	if (unsavedChanges())
	{
		foreach(QString output, RdsEntityWidget::outputs())
		{
			manager()->stopEditing(output);
		}

		foreach(QString output, outputs)
		{
			manager()->startEditing(output);
		}
	}

	foreach(RdsEntityWidgetPrivate::Field field, qxt_d().fields)
	{
		QObject *object = qxt_d().widgets[field.name];
		if (object == NULL) continue;
		object->setProperty("enabled", (!(((field.flags & NoMulti) != 0) && (outputs.size() > 1)) && (field.enabled)));
	}

	qxt_d().outputs = outputs;
}

QString RdsEntityWidget::input()
{
	return(qxt_d().input);
}

QStringList RdsEntityWidget::outputs()
{
	return(qxt_d().outputs);
}

RdsEntityManager *RdsEntityWidget::manager()
{
	return(qxt_d().manager);
}

bool RdsEntityWidget::unsavedChanges()
{
	//qDebug() << "Unsaved Fields:" << qxt_d().modifiedfields;
	foreach(QString name, qxt_d().modifiedfields)
	{
		RdsEntityWidgetPrivate::Field field = qxt_d().fields[name];
		//qDebug() << name << field.value.toString().toAscii().toHex() << qxt_d().widgets[name]->property(qPrintable(field.property));
		//qDebug() << (field.value == qxt_d().widgets[name]->property(qPrintable(field.property)));
	}
	return(qxt_d().modifiedfields.size() > 0);
}

void RdsEntityWidget::apply()
{
	QVariantMap fields;
	foreach(QString name, qxt_d().modifiedfields)
	{
		fields[name] = qxt_d().widgets[name]->property(qPrintable(qxt_d().fields[name].property));
	}
	
	//qDebug() << "Validating";
	
	foreach(QString name, fields.keys())
	{
		QObject *object = qxt_d().widgets[name];
		if(object == NULL) continue;
		
		if(object->metaObject()->className() == QString("QLineEdit"))
		{
			QLineEdit *edit = (QLineEdit *)object;
			//qDebug() << "Validating:" << name << edit->validator();
			QString text = edit->text();
			int size = edit->text().size();
			if(edit->validator() == NULL)
				continue;
			if (edit->validator()->validate(text, size) == QValidator::Acceptable)
				continue;
			else
			{
				QMessageBox::critical(NULL, "Error", qxt_d().fields[name].errtext);
				return;
			}
		}
	}

	if(!validate()) return;
	
	//qDebug() << "Done";

	bool success = beginSetData();

	if (success)
	{
		foreach(QString output, outputs())
		{
			//qDebug() << "Saving Data:" << output << fields;
			if (!setData(output, fields)) success = false;
			manager()->stopEditing(output);
		}
	}
	
	if(success) success = endSetData();

	if (!success)
		QMessageBox::critical(this, "Error", "Some changes failed to apply. Please try again later.");

	setInput(input()); //Reread values

	foreach(QString output, RdsEntityWidget::outputs())
	{
		manager()->stopEditing(output);
	}

	manager()->updateEntity(input());

	qxt_d().modifiedfields.clear();
	qxt_d().applybutton->setEnabled(false);
	qxt_d().discardbutton->setEnabled(false);
}

void RdsEntityWidget::discard()
{
	foreach(QString name, qxt_d().modifiedfields)
	{
		//qDebug() << "Discarding:" << name;
		RdsEntityWidgetPrivate::Field field = qxt_d().fields[name];
		qxt_d().widgets[name]->setProperty(qPrintable(field.property), field.value);
	}

	foreach(QString output, RdsEntityWidget::outputs())
	{
		manager()->stopEditing(output);
	}
	qxt_d().modifiedfields.clear();
	qxt_d().applybutton->setEnabled(false);
	qxt_d().discardbutton->setEnabled(false);
}

void RdsEntityWidget::setField(QString field, QVariant value)
{
	if (!qxt_d().fields.contains(field))
	{
		qWarning() << "There is no field with the name of:" << field;
		return;
	}

	qxt_d().fields[field].value = value;
	qxt_d().widgets[field]->setProperty(qPrintable(qxt_d().fields[field].property), value);
}

void RdsEntityWidget::setFieldEnabled(QString field, bool enabled)
{
	if (!qxt_d().fields.contains(field))
	{
		qWarning() << "There is no field with the name of:" << field;
		return;
	}
	
	qxt_d().fields[field].enabled = enabled;
}

void RdsEntityWidget::entityUpdated(QString id)
{
	if (id != input()) return;

	setInput(input());
}

bool RdsEntityWidget::beginSetData()
{
	return(true);
}

bool RdsEntityWidget::endSetData()
{
	return(true);
}

bool RdsEntityWidget::validate()
{
	return(true);
}
