#!/usr/bin/env python

# rbscrobbler.py
# Copyright (c) 2005 Alex Revo
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import pygtk; pygtk.require('2.0')
import gtk
import gconf
import gobject
import os
import signal
import string
import sys
import threading
import time
import types
import urllib

from Rhythmbox_Bonobo import Rhythmbox_Bonobo
from Audioscrobbler import Audioscrobbler
from MyQueue import MyQueue
from Debug import Debug
import Scrobbler_Musicbrainz

try:
    from egg.trayicon import TrayIcon
    HAS_TRAY = True
except:
    try:
        from trayicon.trayicon import TrayIcon
        HAS_TRAY = True
    except:
        print
        print("Unable to import trayicon.")
        print("Install gnome-python-extras or run 'make'.");
        print
        HAS_TRAY = False

GCONF_ROOT = "/apps/rbscrobbler"
VERSION = "0.0.9"
MYHOME = os.path.dirname(os.path.realpath(__file__))
DISABLE_HANDSHAKE = False

# allow ctrl-c to kill
signal.signal(signal.SIGINT, signal.SIG_DFL)


# initialise classes
RB = Rhythmbox_Bonobo()
RB.enable()
Scrobbler = Audioscrobbler()
queue = MyQueue.getInstance()
MB = Scrobbler_Musicbrainz.MB()


class GConfKey:
    def __init__(self):
        self.__client = gconf.client_get_default()
        self.__client.add_dir(GCONF_ROOT, gconf.CLIENT_PRELOAD_NONE)

    def __getitem__(self, key):
        ret_val = self.__client.get_string("%s/%s" % (GCONF_ROOT, key))
        if (str(ret_val) == "None"):
            return("")
        else:
            return(ret_val)

    def __setitem__(self, key, value):
        # no Python binding for gnome_config_private_set_string?
        # TODO: see libgnome/gnome-config.h
        # TODO: see http://developer.gnome.org/doc/API/libgnome/gnome-gnome-config.html
        self.__client.set_string("%s/%s" % (GCONF_ROOT, key), value)

    def get_muine_string(self, key):
        # TODO: there should be an application-independent location
        # to store this info (for rbscrobbler, Muine, etc.)

        try:
            ret_val = self.__client.get_string("%s/%s" %
            ("/apps/muine/plugins/audioscrobbler", key))
            if (str(ret_val) == "None"):
                return("")
            else:
                return(ret_val)
        except:
            return("")


class UserInfo:
    remember_password = False

    def __init__(self):
        self.gconf = GConfKey()

        self.__create_window()
        self.__update()
        self.login()

    def __create_window(self):
        # Create login dialog.
        self.dialog = gtk.Dialog("Rhythmbox Audioscrobbler Client")
        self.dialog.set_border_width(0)
        self.dialog.set_resizable(False)

        # Create accelerators.
        self.accel_group = gtk.AccelGroup()
        self.dialog.add_accel_group(self.accel_group)

        # Add delete/destroy signal handlers.
        self.dialog.connect("delete_event", self.__delete_event)
        self.dialog.connect("destroy", self.__destroy)

        # Add buttons
        # (Button) Cancel
        self.btnCancel = gtk.Button("Cancel", gtk.STOCK_CANCEL)
        stock_text = gtk.stock_lookup(gtk.STOCK_CANCEL)[1]
        accel_keyval = ord(stock_text[stock_text.find('_')+1])
        self.btnCancel.add_accelerator("grab_focus", self.accel_group, accel_keyval, gtk.gdk.CONTROL_MASK, gtk.ACCEL_VISIBLE)
        self.btnCancel.connect("clicked", self.btnCancel_clicked, "button 1")
        self.dialog.action_area.pack_start(self.btnCancel, True, True, 0)
        # (Button) OK
        self.btnOk = gtk.Button("_Ok", gtk.STOCK_OK)
        stock_text = gtk.stock_lookup(gtk.STOCK_OK)[1]
        accel_keyval = ord(stock_text[stock_text.find('_')+1])
        self.btnOk.add_accelerator("grab_focus", self.accel_group, accel_keyval, gtk.gdk.CONTROL_MASK, gtk.ACCEL_VISIBLE)
        self.btnOk.connect("clicked", self.btnOk_clicked, "button 1")
        self.dialog.action_area.pack_start(self.btnOk, True, True, 0)
        self.btnOk.set_sensitive(False)

        # Add table for username/password/remember checkbox
        self.tblLogin = gtk.Table(4, 2, False)
        self.tblLogin.set_row_spacings(6)
        self.tblLogin.set_col_spacings(12)
        self.tblLogin.set_border_width(12)
        self.dialog.vbox.pack_start(self.tblLogin, True, True, 0)
        # (Label) info
        self.lblInfo = gtk.Label("<b>Login to Audioscrobbler.com</b>")
        self.lblInfo.set_use_markup(True)
#        self.lblInfo.set_alignment(0, 0)
        self.tblLogin.attach(self.lblInfo, 0, 2, 0, 1)
        # Add username/password labels & boxes
        # (Label) Username
        self.lblUsername = gtk.AccelLabel("_Username:")
        self.lblUsername.set_use_underline(True)
        self.tblLogin.attach(self.lblUsername, 0, 1, 1, 2)
        # (Entry) Username
        self.entUsername = gtk.Entry(25)
        self.entUsername.connect("activate", self.entUsername_activate)
        self.entUsername.connect("changed", self.entUsername_changed)
        self.entUsername.add_accelerator("grab_focus", self.accel_group, ord("U"), gtk.gdk.CONTROL_MASK, gtk.ACCEL_VISIBLE)
        self.lblUsername.set_accel_widget(self.entUsername)
        self.tblLogin.attach(self.entUsername, 1, 2, 1, 2)
        # (Label) Password
        self.lblPassword = gtk.AccelLabel("_Password:")
        self.lblPassword.set_use_underline(True)
        self.tblLogin.attach(self.lblPassword, 0, 1, 2, 3)
        # (Entry) Password
        self.entPassword = gtk.Entry(25)
        self.entPassword.set_visibility(False)
        self.entPassword.set_sensitive(False)
        self.entPassword.connect("activate", self.entPassword_activate)
        self.entPassword.connect("changed", self.entPassword_changed)
        self.entPassword.add_accelerator("grab_focus", self.accel_group, ord("P"), gtk.gdk.CONTROL_MASK, gtk.ACCEL_VISIBLE)
        self.lblPassword.set_accel_widget(self.entPassword)
        self.tblLogin.attach(self.entPassword, 1, 2, 2, 3)
        # (CheckButton) Remember password
        self.chkRemember = gtk.CheckButton(label="_Remember password")
        self.chkRemember.add_accelerator("grab_focus", self.accel_group, ord("R"), gtk.gdk.CONTROL_MASK, gtk.ACCEL_VISIBLE)
        self.chkRemember.set_sensitive(False)
        self.chkRemember.connect("toggled", self.chkRemember_toggled)
        self.tblLogin.attach(self.chkRemember, 1, 2, 3, 4)
        self.dialog.action_area.show_all()
        self.tblLogin.show_all()


    def __delete_event(self, widget, event, data=None):
        self.hide()
        return(True)
    def __destroy(self, widget, data=None):
        self.hide()
        return(True)

    def hide(self):
        self.dialog.hide()
    def show(self):
        self.__update()
        self.dialog.show()

    def __update(self):
        self.username = str(self.gconf["username"])
        self.password = str(self.gconf["password"])

        if (not self.username):
            self.username = str(self.gconf.get_muine_string("username"))
            self.gconf["username"] = self.username
        if (not self.password):
            self.password = str(self.gconf.get_muine_string("password"))
            self.gconf["password"] = self.password

        if (self.username):
            self.entUsername.set_text(self.username)
            if (self.password):
                self.entPassword.set_text(self.password)
                self.btnOk.set_sensitive(True)
                self.btnOk.grab_focus()
                if (self.gconf["password"]):
                    self.chkRemember.set_active(True)
                    self.remember_password = True
                else:
                    self.chkRemember.set_active(False)
                    self.remember_password = False
            else:
                self.entPassword.set_text("")
                self.entPassword.grab_focus()
                self.chkRemember.set_active(False)
                self.remember_password = False


    def login(self):
        if (self.username and self.password):
            Scrobbler.set_username(self.username)
            Scrobbler.set_password(self.password)

            self.hide()
        else:
            self.__update()
            self.show()

    def btnCancel_clicked(self, widget, data=None):
        self.hide()

    def btnOk_clicked(self, widget, data=None):
        # Store username/password
        self.password = self.entPassword.get_text()
        self.username = self.entUsername.get_text()

        # update gconf values
        self.gconf["username"] = self.username
        # NOTE: this saves passwords in plain-text
        tmp_pw = ""
        if (self.remember_password):
            tmp_pw = self.password
        self.gconf["password"] = tmp_pw

        self.login()
        self.hide()

    def chkRemember_toggled(self, widget, data=None):
        self.remember_password = widget.get_active()

    def entUsername_activate(self, widget, data=None):
        self.entPassword.grab_focus()
    def entUsername_changed(self, widget, data=None):
        len_username = len(self.entUsername.get_text())
        len_password = len(self.entPassword.get_text())

        if (len_username > 0):
            self.entPassword.set_sensitive(True)

            if (len_password > 0):
                self.chkRemember.set_sensitive(True)
                self.btnOk.set_sensitive(True)
        else:
            self.entPassword.set_sensitive(False)
            self.chkRemember.set_sensitive(False)
            self.btnOk.set_sensitive(False)

    def entPassword_activate(self, widget, data=None):
        self.btnOk.grab_focus()
    def entPassword_changed(self, widget, data=None):
        len_username = len(self.entUsername.get_text())
        len_password = len(self.entPassword.get_text())

        if (len_username > 0 and len_password > 0):
            self.btnOk.set_sensitive(True)
            self.chkRemember.set_sensitive(True)
        else:
            self.btnOk.set_sensitive(False)
            self.chkRemember.set_sensitive(False)



class RunScrobbler(Debug):
    old_time = 0
    is_queued = False
    is_playing = False
    last_refresh = 0

    def __init__(self):
        # Add Rhythmbox callbacks
        RB.add_callback_song(self.__song_change)
        RB.add_callback_state(self.__playing_change)

        # run callbacks immediately
        if (RB.is_active()):
            self.__song_change()
            self.__playing_change()

    def __song_change(self, obj=None):
        self.is_queued = 0
        self.artist = RB["artist"].strip()
        self.album = RB["album"].strip()
        self.title = RB["title"].strip()
        self.duration = RB["duration"]
        self.uri = RB["uri"]

        # Ugly hack to update GUI
        try:
            Scrobbler.ugly_song_change_hack(RB)
        except:
            pass

        # drop tracks that violate the specification (1.2)
        if (self.duration < 30):
            self.is_queued = True
            Scrobbler.debug("  [!!] duration < 30 seconds; not submitting.")
        if (self.artist.lower() == "unknown"):
            self.is_queued = True
            Scrobbler.debug("  [!!] unknown artist; not submitting.")
        if (self.title.lower() == "unknown"):
            self.is_queued = True
            Scrobbler.debug("  [!!] unknown title; not submitting.")
        # TODO: really fix this. Doesn't matter at the moment as RB doesn't
        # report radio stuff (at the moment).
        #if (not (self.uri.startswith("file:") or self.uri.startswith("smb:"))):
        #    self.is_queued = True
        #    Scrobbler.debug("  [!!] URI is not local file; not submitting.")


    def __playing_change(self, obj=None):
        if (RB["playing"]):
            self.is_playing = True
        else:
            self.is_playing = False

        #print "playing status changed: %s" % self.is_playing
        return(0)


    def run(self):
        while(True):
            try:
                self.run_iter()
            except Exception, error:
                self.handle_exception(error)
            time.sleep(15)


    def run_iter(self):
        # Perform handshake if needed.
        if (not DISABLE_HANDSHAKE):
            Scrobbler.do_handshake()

        # refresh data
        RB.refresh()
        elapsed = int(RB["elapsed_time"])

        # Check for missed song change
        if (self.uri != RB["uri"]):
            self.__song_change()

        # return immediately if song had already been queued
        #  or is not playing
        if (self.is_queued or (not self.is_playing)):
            return

        # don't allow skipping
        if ((elapsed - self.old_time) > (int(time.time()) - int(self.last_refresh) + 5)):
            Scrobbler.debug("  Skipping detected; not submitting.")
            queue.push(("message", "Skipping detected; not submitting."))
            self.is_queued = True
            return

        # update times
        self.last_refresh = int(time.time())
        self.old_time = elapsed

        # check if we should submit this song
        if ((elapsed > self.duration / 2) or (elapsed >= 240)):
            # Format time correctly for Audioscrobbler.
            T = time.gmtime(time.time() - elapsed)
            queueTime = u"%04d-%02d-%02d %02d:%02d:%02d" % (T[0], T[1], T[2], T[3], T[4], T[5])
            ### 1.2 stuff:
            #queueTime = int(time.time()) - elapsed

            # get Musicbrainz id.
            uri = urllib.unquote(RB["uri"].replace("file://", ""))
            mbid = MB.getSignature(uri)
            if (not mbid):
                mbid = ""

            # Generate POST data for updates:
            #   u - username
            #   s - md5 response
            #   a - artist
            #   t - title
            #   b - album
            #   m - musicbrainz id
            #   l - length (secs)
            #   i - time (UTC)
            indexNumber = len(Scrobbler.get_queue())
            postData = ("a[%d]=%s&t[%d]=%s&b[%d]=%s&m[%d]=%s&l[%d]=%s&i[%d]=%s") % (
                indexNumber, Scrobbler.encode(self.artist),
                indexNumber, Scrobbler.encode(self.title),
                indexNumber, Scrobbler.encode(self.album),
                indexNumber, mbid,
                indexNumber, Scrobbler.encode(str(self.duration)),
                indexNumber, urllib.quote_plus(queueTime)
            )

            # Add it to the queue.
            Scrobbler.add_to_queue(postData)
            self.is_queued = True

            # run queue
            Scrobbler.run_queue()



class ScrobblerQueue:
    def __init__(self):
        self.gconf = GConfKey()
        self.login = UserInfo()
        self.status = "offline"
        self.__create_window()
        self.__create_icon()
        self.__create_menu()

    def __new_image(self, filename):
        retVal = None
        new_filename = os.path.join(MYHOME, "images", filename)
        if (os.path.isfile(new_filename)):
            try:
                retVal = gtk.gdk.pixbuf_new_from_file(new_filename)
            except Exception, error:
                retVal = None
                print "ERROR: %s" % error
        return(retVal)

    def __create_icon(self):
        self.__icon = gtk.Image()
        if (HAS_TRAY):
            self.tray = TrayIcon("rbscrobbler")
            pixbuf = self.__new_image("icon-offline.png")
            if (pixbuf):
                self.__tooltips = gtk.Tooltips()
                self.__icon.set_from_pixbuf(pixbuf)
                self.__ebox = gtk.EventBox()
                self.__ebox.add(self.__icon)
                self.__ebox.connect('button-press-event', self.__ebox_clicked)
                self.__ebox.show_all()
                self.settip("rbscrobbler")
                self.tray.add(self.__ebox)
                self.tray.show_all()
        else:
            self.status_push("Unable to import trayicon. Did you run 'make'?")

    def __create_menu(self):
        self.menu = gtk.Menu()
        item_blank = gtk.MenuItem()
        item_login = gtk.MenuItem("_Login...")
        item_login.connect("activate", self.__login_clicked)
        item_quit = gtk.ImageMenuItem(gtk.STOCK_QUIT)
        item_quit.get_children()[0].set_text_with_mnemonic("_Quit")
        item_quit.connect("activate", self.__quit_clicked)
        item_blank.show()
        item_login.show()
        item_quit.show()
        self.menu.append(item_login)
        self.menu.append(item_blank)
        self.menu.append(item_quit)
        self.menu.show_all()

    def __create_window(self):
        # create main window
        self.window = gtk.Window(gtk.WINDOW_TOPLEVEL)
        self.window.set_title("Audioscrobbler Queue")
        self.window.set_border_width(0)
        self.window.set_resizable(True)
        self.window.set_geometry_hints(self.window, 300, 100, -1, -1, 300, 100, -1, -1, -1, -1)
        self.window.set_default_size(600, 150)

        # Add delete/destroy signal handlers.
        self.window.connect("delete_event", self.__delete_event)
        self.window.connect("destroy", self.__destroy)

        # (VBox) vbox
        self.vbox = gtk.VBox()
        # (Statusbar) statusbar
        self.status_image = gtk.Image()
        pixbuf = self.__new_image("offline.png")
        if (pixbuf):
            self.status_image.set_from_pixbuf(pixbuf)
        self.shbox = gtk.HBox()
        self.status_button = gtk.Button()
        self.status_button.set_relief(gtk.RELIEF_NONE)
        self.status_button.add(self.status_image)
        self.status_button.connect("clicked", self.status_button_clicked, "button 1")
        self.statusbar = gtk.Statusbar()
        self.statusbar.set_has_resize_grip(True)
        self.shbox.pack_start(self.status_button, False, False, 0)
        self.shbox.pack_end(self.statusbar, True, True, 0)
        self.vbox.pack_end(self.shbox, False, False, 0)
        # (Label) info
        self.infohbox = gtk.HBox()
        self.imgSubmitted = gtk.Image()
        pixbuf = self.__new_image("notsubmitted.png")
        if (pixbuf):
            self.imgSubmitted.set_from_pixbuf(pixbuf)
        self.infohbox.pack_end(self.imgSubmitted, False, False, 6)
        self.lblInfo = gtk.Label("<b>not playing</b>")
        self.lblInfo.set_use_markup(True)
        self.lblInfo.set_padding(6, 0)
        self.lblInfo.set_alignment(0, 0)
        self.infohbox.pack_start(self.lblInfo, False, False, 0)
        self.vbox.pack_start(self.infohbox, False, False, 6)

        # (ScrolledWindow) scrolled_win
        self.scrolled_win = gtk.ScrolledWindow()
        self.scrolled_win.set_property('hscrollbar-policy', gtk.POLICY_AUTOMATIC)
        self.scrolled_win.set_property('vscrollbar-policy', gtk.POLICY_AUTOMATIC)
        # Create cells
        self.cell_time = gtk.CellRendererText()
        self.cell_artist = gtk.CellRendererText()
        self.cell_title = gtk.CellRendererText()
        self.cell_album = gtk.CellRendererText()
        self.cell_length = gtk.CellRendererText()
        self.cell_mb = gtk.CellRendererText()
        # (ListStore) ls_queue
        # (time, artist, title, album, length)
        self.ls_queue = gtk.ListStore(
            gobject.TYPE_STRING, 
            gobject.TYPE_STRING,
            gobject.TYPE_STRING)
        self.treeview = gtk.TreeView(self.ls_queue)
        self.treeview.set_rules_hint(True)
        self.tv_column_time = gtk.TreeViewColumn("Time", self.cell_time, text=0)
        self.tv_column_artist = gtk.TreeViewColumn("Artist", self.cell_artist, text=1)
        self.tv_column_title = gtk.TreeViewColumn("Title", self.cell_title, text=2)
        self.tv_column_time.set_property('resizable', True)
        self.tv_column_artist.set_property('resizable', True)
        self.tv_column_title.set_property('resizable', True)
        # setup treeview.append() properly
        self.treeview.append_column(self.tv_column_time)
        self.treeview.append_column(self.tv_column_artist)
        self.treeview.append_column(self.tv_column_title)
        # Allow sorting of cells
        self.treeview.set_search_column(0)
        self.tv_column_time.set_sort_column_id(0)
        self.tv_column_artist.set_sort_column_id(1)
        self.tv_column_title.set_sort_column_id(2)

        # Add treeview to window.
        self.scrolled_win.add(self.treeview)
        self.vbox.pack_start(self.scrolled_win, True, True, 0)
        self.vbox.show_all()
        self.window.add(self.vbox)

        if ((not HAS_TRAY) or (self.gconf["visible"] == "True")):
            self.show()


    def __delete_event(self, widget, event, data=None):
        if (not HAS_TRAY):
            return(False)
        else:
            self.hide()
            return(True)
    def __destroy(self, widget, data=None):
        if (not HAS_TRAY):
            gtk.main_quit()
            ###sys.exit(0)
        else:
            self.hide()
            return(True)


    def settip(self, tip):
        try:
            self.__tooltips.set_tip(self.__ebox, tip)
        except:
            pass
    def hide(self):
        self.gconf["visible"] = ""
        self.window.hide()
    def show(self):
        self.gconf["visible"] = "True"
        self.window.show_all()

    def status_push(self, message):
        id = self.statusbar.get_context_id("main")
        self.statusbar.pop(id)
        self.statusbar.push(id, str(message))

    def add(self, data):
        self.ls_queue.append(data)
    def clear(self):
        self.ls_queue.clear()

    def set_online(self):
        self.status = "online"
        pixbuf = self.__new_image("online.png")
        if (pixbuf):
            self.status_image.set_from_pixbuf(pixbuf)
        pixbuf = self.__new_image("icon-online.png")
        if (pixbuf):
            self.__icon.set_from_pixbuf(pixbuf)
    def set_offline(self):
        self.status = "offline"
        self.status_push("Unable to contact Audioscrobbler. Updates will be cached locally")
        pixbuf = self.__new_image("offline.png")
        if (pixbuf):
            self.status_image.set_from_pixbuf(pixbuf)
        pixbuf = self.__new_image("icon-offline.png")
        if (pixbuf):
            self.__icon.set_from_pixbuf(pixbuf)
    def set_submitted(self):
        pixbuf = self.__new_image("submitted.png")
        if (pixbuf):
            self.imgSubmitted.set_from_pixbuf(pixbuf)
    def set_not_submitted(self):
        pixbuf = self.__new_image("notsubmitted.png")
        if (pixbuf):
            self.imgSubmitted.set_from_pixbuf(pixbuf)

    def __ebox_clicked(self, src, event):
        if (event.button == 1):
            if (self.gconf["visible"] == "True"):
                self.hide()
            else:
                self.show()
        elif (event.button == 3):
            self.menu.popup(None, None, None, event.button, event.time)
    def __quit_clicked(self, data):
        gtk.main_quit()
        ###sys.exit(0)
    def __login_clicked(self, data):
        self.login.show()

    def status_button_clicked(self, widget, data=None):
        self.login.show()
        #if (self.status == "online"):
        #    self.status_push("Not implemented yet")
        #    #self.set_offline()
        #else:
        #    self.status_push("Not implemented yet")
        #    #self.status_push("Setting online mode...")
        #    #self.login.login()



class bad_wrapper:
    def __init__(self):
        self.swin = ScrobblerQueue()
        self.RunScrobbler = RunScrobbler()

    def __callback_add(self, data):
        self.queue_to_liststore(data)
        self.swin.set_submitted()
    def __callback_song(self, obj=None):
        try:
            self.swin.set_not_submitted()

            artist = obj["artist"].replace('&', '&amp;')
            title = obj["title"].replace('&', '&amp;')

            if (artist and title):
                self.swin.lblInfo.set_text("<b>%s - %s</b>" % (artist, title))
            else:
                self.swin.lblInfo.set_text("<b>not playing</b>")

            self.swin.lblInfo.set_use_markup(True)
        except Exception, error:
            print error

    def queue_to_liststore(self, data):
        # newData = (time, artist, title)
        # data = e.g., "a[3]=B-52s%2C+The&t[3]=Strobe+Light&b[3]=Wild+Planet&m[3]=&l[3]=241&i[3]=2004-09-30+13%3A16%3A08"

        max_num = 10
        if (len(Scrobbler.get_queue()) > max_num):
            max_num = len(Scrobbler.get_queue())

        while (self.swin.ls_queue.iter_n_children (None) > max_num):
            self.swin.ls_queue.remove (self.swin.ls_queue.get_iter_first ())

        tmp = []
        for i in string.split(data, '='):
            tmp.append(urllib.unquote_plus(str(i.split('&')[0])))
        artist = tmp[1]
        title = tmp[2]
        timestamp = "%s UTC" % tmp[6]
        self.swin.add([timestamp, artist, title])

    def run(self):
        gtk.threads_enter()
        # run background scrobbler loop in separate thread
        RB.add_callback_song(self.__callback_song)
        th_scrobbler = threading.Thread(target=self.RunScrobbler.run)
        th_scrobbler.setDaemon(True)
        th_scrobbler.start()

        #### Run message queue in separate thread
        ###th_mqueue = threading.Thread(target=self.run_message_queue)
        ###th_mqueue.setDaemon(True)
        ###th_mqueue.start()

        # Run message queue every 5 seconds
        self.timer = gobject.timeout_add (5000, self.run_message_queue)

        # Load queue into window
        for i in Scrobbler.get_queue():
            self.queue_to_liststore(i)

        gtk.main()

        gobject.source_remove (self.timer)
        RB.disable()

        gtk.threads_leave()


    def run_message_queue(self):
        # TODO: handle BADAUTH? (1.2 deals w/ this better)
        ###while (True):
        ###    queue.wait()
        if (queue.is_dirty()):
            while (queue.qsize()):
                (key, value) = queue.get()
                if (key == "message"):
                    self.swin.status_push(value)
                    self.swin.settip(value)
                elif (key == "mode"):
                    if (value == "online"):
                        self.swin.set_online()
                    else:
                        self.swin.set_offline()
                elif (key == "changed"):
                    self.__callback_song(RB)
                elif (key == "submitting"):
                    self.__callback_add(value)
            queue.clean()

        return True



if __name__ == "__main__":
    gtk.threads_init()

    if (len(sys.argv) >= 2):
        if (sys.argv[1] == "-f"):
            DISABLE_HANDSHAKE = True;
            print("\n\nHANDSHAKING DISABLED.\n")

    main = bad_wrapper()
    main.run()
