# Audioscrobbler.py
# Copyright (c) 2005 Alex Revo
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# TODO:
#    save last song between instances to avoid submitting twice
#    use protocol 1.2


# Import the required system modules.
import md5
import os
import re
import socket
import string
import time
import types
import urllib
import urllib2

# Import rbscrobbler modules.
from Debug import Debug
from MyQueue import MyQueue


CLIENT_ID = "rbx"
CLIENT_VERSION = "0.9"
MAX_QUEUE_SIZE = 1000
SCROBBLER_URL = "http://post.audioscrobbler.com/"
SCROBBLER_VERSION = "1.1"
#SCROBBLER_VERSION = "1.2"
QUEUE_PATH = os.path.join(os.environ['HOME'], ".rbscrobbler.queue")

queue = MyQueue.getInstance()


def set_status(message):
    queue.put(('message', str(message)))
def set_online():
    queue.put(('mode', 'online'))
def set_offline():
    queue.put(('mode', 'offline'))
def set_song_changed():
    queue.put(('changed', 'changed'))
def set_song_submitting(data):
    queue.put(('submitting', data))


class Audioscrobbler(Debug):
    # Initialise variables.
    __failures = 0
    __handshake = False
    __handshake_next = 0
    __md5_challenge = ""
    __password = ""
    __queue = []
    __run_queue = False
    __submit_url = "http://post.audioscrobbler.com/v1.1.php"
    __username = ""


    def __init__(self, *args):
        self.debug("Plugin ID: %s, Version %s (Protocol %s)" %
                   (CLIENT_ID, CLIENT_VERSION, SCROBBLER_VERSION))

        # Load saved queue.
        self.__load_queue()

        socket.setdefaulttimeout(60)
        # Make urllib2 use http_proxy
        http_proxy = os.getenv('http_proxy')
        if (http_proxy):
            self.debug("Using proxy: %s" % http_proxy)
            proxy_handler = urllib2.ProxyHandler({'http': http_proxy})
            opener = urllib2.build_opener(proxy_handler)
            urllib2.install_opener(opener)


    def get_queue(self):
        return(self.__queue)
    def set_password(self, data):
        if (self.__password != data):
            self.debug("Set password.")
            self.__password = data
            self.__handshake = False
    def set_username(self, data):
        if (self.__username != data):
            self.debug("Set username to \"%s\"." % data)
            self.__username = data
            self.__handshake = False


    def __parse_response(self, url, post_data = None):
        ret_val = ""

        try:
            resp = urllib2.urlopen(url, post_data)
        except Exception, error:
            self.debug("  error: %s" % error)
            return(error)

        resp_type = resp.readline().strip()
        if (resp_type.startswith("UPTODATE")):
            self.__md5_challenge = resp.readline().strip()
            self.__submit_url = resp.readline().strip()
        if (resp_type.startswith("UPDATE")):
            set_status("An updated version of rbscrobbler is available.")
            self.debug("  UPDATE: Updated version is available")
            self.__md5_challenge = resp.readline().strip()
            self.__submit_url = resp.readline().strip()
        if (resp_type.startswith("FAILED")):
            set_status("FAILED: " + resp_type[7:])
            self.debug("  [!!] FAILED: " + resp_type[7:])
            ret_val = resp_type[7:]
        if (resp_type.startswith("BADAUTH") or resp_type.startswith("BADUSER")):
            set_status("BADAUTH: Invalid username and/or password.")
            self.debug("  [!!] BADAUTH: Invalid username and/or password.")
            ret_val = "Invalid username and/or password"
        return(ret_val)


    def __load_queue(self):
        try:
            if (os.path.isfile(QUEUE_PATH)):
                cache = open(QUEUE_PATH, "r")
                map(lambda x: self.__queue.append(re.sub("\n$", "", x)),
                    cache.readlines())
                cache.close()

                if (len(self.__queue) > 0):
                    set_status("%s saved entries loaded." % len(self.__queue))
                    self.debug("%s saved entries loaded." % len(self.__queue))
                    self.__run_queue = True
        except Exception, error:
            self.handle_exception(error)

    def __save_queue(self):
        try:
            cache = open(QUEUE_PATH, "w")
            map(lambda x: cache.write("%s\n" % x), self.__queue)
            cache.close()
        except Exception, error:
            self.handle_exception(error)


    def ugly_song_change_hack(self, obj=None):
        set_song_changed()


    def do_handshake(self):
        if (self.__handshake):
            return
        if (int(time.time()) < self.__handshake_next):
            return
        if (not self.__username):
            return

        set_status("Performing handshake with Audioscrobbler server...")
        self.debug("Performing handshake...")

        # Wait at least 30 minutes between handshakes
        self.__handshake_next = int(time.time()) + 1800

        # Parse handshake response.
        url = "%s?hs=true&p=%s&c=%s&v=%s&u=%s&" % \
              (SCROBBLER_URL,
               SCROBBLER_VERSION,
               CLIENT_ID,
               CLIENT_VERSION,
               urllib.quote_plus(self.__username))
        ### 1.2 stuff:
        #timestamp = int(time.time())
        #url = "%s?hs=true&p=%s&c=%s&v=%s&u=%s&t=%d&a=%s" % \
        #      (SCROBBLER_URL,
        #       SCROBBLER_VERSION,
        #       CLIENT_ID,
        #       CLIENT_VERSION,
        #       urllib.quote_plus(self.__username),
        #       timestamp,
        #       self.handshake_hash(timestamp))
        error = self.__parse_response(url)
        if (not error):
            set_online()
            set_status("rbscrobbler is now online.")
            self.debug("  Handshake was successful.")

            # Reset failure count and run queue.
            self.__handshake = True
            self.__run_queue = True
            self.__failures = 0

            time.sleep(2)
            self.run_queue()
        else:
            set_offline()
            T = time.localtime(self.__handshake_next)
            next_try = "%02d:%02d %s" % (T[3], T[4], time.strftime("%Z"))
            set_status("Handshake failed: %s (will retry at %s)" %
                        (error, next_try))
            self.debug("  Handshake failed: %s (will retry at %s)" %
                        (error, next_try))

            self.__run_queue = False


    def add_to_queue(self, data):
        try:
            while (len(self.__queue) > MAX_QUEUE_SIZE):
                self.__queue.pop()

            # Don't add duplicate entries
            if (len(self.__queue) > 0):
                # Ignore time difference (last 26 chars)
                prev = re.sub("\[\d+?\]", "", self.__queue[0])[0:-26]
                new = re.sub("\[\d+?\]", "", data)[0:-26]
                if (prev == new):
                    return

            self.debug("Adding entry to queue: %s" % data)
            self.__queue.insert(0, data)
            self.__save_queue()

            set_song_submitting(data)
        except Exception, error:
            self.handle_exception(error)

    def run_queue(self):
        try:
            # return if we have no reason to run the queue
            if (not self.__handshake):
                return
            if (not self.__run_queue):
                return
            if (not len(self.__queue)):
                return

            self.debug("Running queue.")

            # If updating has failed >=3 times, renegotiate handshake and
            # stop running queue until new handshake.
            if (self.__failures >= 3):
                set_status("Queue run has failed 3+ times; caching locally.")
                self.debug("  [!!] Queue run has failed >=3 times; update has been cached locally.")
                self.__run_queue = False
                self.__handshake = False
                set_offline()
                return

            # Build POST data:
            new_data = ""
            entries = len(self.__queue)
            for i in self.__queue:
                new_data += "%s&" % i
            web_username= self.encode(self.__username)
            web_md5resp = self.encode(self.hashPassword(self.__password))
            new_data += "u=%s&s=%s" % (web_username, web_md5resp)

            # parse response
            error = self.__parse_response(self.__submit_url, new_data)
            if (not error):
                set_status("Last submission: %s" % time.strftime("%x %X %Z"))
                self.debug("  Success. %s song(s) submitted." % entries)

                # clear queue & reset failure count
                self.__queue = []
                self.__failures = 0
                self.__save_queue()
            else:
                self.__failures += 1

                set_status("Queue run has failed %s time(s): %s" %
                            (self.__failures, error))
                self.debug("  Queue run has failed %s time(s): %s" %
                            (self.__failures, error))
        except Exception, error:
            self.handle_exception(error)


    def hashPassword(self, password):
        # The MD5 response is md5(md5(password) + challenge), where MD5
        # is the ascii-encoded MD5 representation, and + represents
        # concatenation.
        tmp = md5.new(password).hexdigest()
        return(md5.new(tmp + self.__md5_challenge).hexdigest())
    ### 1.2 stuff:
    #def handshake_hash(self, timestamp = int(time.time())):
    #    # passcode is md5(md5(<your password>)+unix_timestamp) where +
    #    # represents concatanation.
    #    tmp = "%s%s" % \
    #           (md5.new(self.__password).hexdigest(),
    #            int(time.time()))
    #    return(md5.new(tmp).hexdigest())


    def encode(self, input):
        if (type(input) == types.UnicodeType):
            return(urllib.quote_plus(input))
        return(urllib.quote_plus(unicode(input, 'iso-8859-1')))
