/*

  rawtran -  RAW to FITS converter
  Copyright (C) 2007-x  Filip Hroch, Masaryk University, Brno, CZ

  Rawtran is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  Rawtran is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Rawtran.  If not, see <http://www.gnu.org/licenses/>.

  Ideas:

  * read infos from files to prevent direct run onf dcraw (no needs
    interprocess communication + nice fine tunning parameters)

*/

#define _XOPEN_SOURCE

#include "pixmap.h"
#include "rawtran.h"

#include <time.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <float.h>
#include <stdlib.h>
#include <fitsio.h>

#include <assert.h>


inline unsigned short array(int xdim,int ydim, unsigned short *pole, int i, int j)
{
  assert((0 <= i && i < xdim) && (0 <= j && j < ydim));
  return pole[j*xdim + i];
}

inline void matmul(float cmatrix[][3], float x[], float y[])
{
  int i,j;
  float s;

  for(i = 0; i < 3; i++) {
    s = 0.0;
    for(j = 0; j < 3; j++)
      s += cmatrix[i][j]*x[j];
    y[i] = s;
  }
}

int rawtran(char *raw, char *fitsname, int type, char *filter, int bitpix,
	    char *opts, char *execs, char *adds, char *dark)
{
  fitsfile *file;
  int status, npixels, iso, naxis;
  long naxes[3];
  float *flux,*f;
  int i,j,k,m,jj,width,height,nexif,idx,ix,jx;
  float maxval,minval,g1,g2,x,y,exp,focus=-1.0;
  unsigned short int *d;
  const int maxexif = 21;
  char exif[maxexif][FLEN_CARD+1], stime[FLEN_CARD+1], etime[FLEN_CARD+1],
    camera[FLEN_CARD+1], line[FLEN_CARD+1], aperture[FLEN_CARD+1];
  const char *ver = "Created by rawtran: http://integral.physics.muni.cz/rawtran";

  /*  matrix for XYZ to BVR transformation, D65 */
  float cmatrix[][3] = {{ 0.200308,  -0.017427,   0.012508},
			{ 0.013093,   0.789019,  -0.328110},
			{-0.224010,  -0.465060,   1.381969}};
  /* "matrix" from XYZ D65 to scotopic curve */
  float scovec[3] = { 0.36169, 1.18214, -0.80498 };
  float bvr[3],zyx[3];

  PIXMAP *p;
  FILE *dcraw;
  struct tm tm;

  const char *dcraw_info_command = "dcraw -i -v -c ";
  const char *dcraw_command = "dcraw -c ";
  const char *dark_comment = "Darkframe subtracted: ";
  const char *rum_comment = "Command: ";
  char *com, *photsys, *add;

  assert(execs);

  flux = NULL;

  m = adds ? strlen(adds) : 0;
  k = dark ? strlen(dark) : 0;
  add = malloc(strlen(execs)+k+m+7);
  /* 7 stand for " -K " (and waste space when dark is unused) */

  /* add dark to options */
  strcpy(add,execs);
  if( dark ) {
    strcat(add," -K ");
    strcat(add,dark);
    strcat(add," ");
  }
  if( adds ) {
    strcat(add," ");
    strcat(add,adds);
  }

  /* run dcraw to get EXIF info */
  com = malloc(strlen(dcraw_info_command)+strlen(raw)+1);
  strcpy(com,dcraw_info_command);
  strcat(com,raw);
  if( (dcraw = popen(com,"r")) == NULL ) {
    fprintf(stderr,"Failed to invoke: `%s'\n",dcraw_info_command);
    free(com);
    return(1);
  }
  for( nexif = 0, i = 0; i < maxexif && fgets(exif[i],FLEN_CARD,dcraw); i++) {
    for( j = 0; exif[i][j] != '\n'; j++);
    exif[i][j] = '\0';
  }
  nexif = i;
  pclose(dcraw);
  free(com);

  /* decode exif data */
  for( i = 0; i < nexif; i++) {

    /* time of start of exposure ? */
    if( strstr(exif[i],"Timestamp:") ) {
      strncpy(stime,exif[i]+strlen("Timestamp: "),FLEN_CARD);
      if( strptime(stime,"%a %b %d %T %Y",&tm) != NULL )
	strftime(stime, sizeof(stime), "%Y-%m-%dT%T", &tm);
      else
	strcpy(stime,"");
    }

    /* exposure time */
    if( strstr(exif[i],"Shutter:") ) {
      strncpy(etime,exif[i]+strlen("Shutter: "),FLEN_CARD);
      if( strstr(etime,"/") ){
	  sscanf(etime,"%f/%f",&x,&y);
	  exp = x/y;
      }
      else
	  sscanf(etime,"%f",&exp);
    }

    /* camera */
    if( strstr(exif[i],"Camera:") )
      strncpy(camera,exif[i]+strlen("Camera: "),FLEN_CARD);

    /* ISO speed */
    if( strstr(exif[i],"ISO speed:") ) {
      strncpy(line,exif[i]+strlen("ISO speed:"),FLEN_CARD);
      sscanf(line,"%d",&iso);
    }

    /* aperture */
    if( strstr(exif[i],"Aperture:") )
      strncpy(aperture,exif[i]+strlen("Aperture: "),FLEN_CARD);

    /* focus */
    if( strstr(exif[i],"Focal length:") ) {
      strncpy(line,exif[i]+strlen("Focal length:"),FLEN_CARD);
      if( sscanf(line,"%f",&focus) != 1 )
	focus = -1.0;
    }

  }


  /* run dcraw to get PIXMAP */
  com = malloc(strlen(dcraw_command)+strlen(opts)+strlen(add)+strlen(raw)+3);
  strcpy(com,dcraw_command);
  strcat(com,opts);
  strcat(com," ");
  strcat(com,add);
  strcat(com," ");
  strcat(com,raw);
  free(add);
  if( (dcraw = popen(com,"r")) == NULL ) {
    fprintf(stderr,"Failed to invoke: `%s'\n",com);
    free(com);
    return(1);
  }

  if( (p = pixmap(dcraw)) == NULL ) {
    pclose(dcraw);
    fprintf(stderr,"Failed to import data by `%s'.\n",com);
    fprintf(stderr,"Please, check that dcraw can properly decode file `%s'.\n",raw);
    free(com);
    return(1);
  }
  pclose(dcraw);


  /* data decoding */
  if( type == 0 ) {
    /* standard band */

    if( p->colors != 3 ) {
      pixmap_free(p);
      fprintf(stderr,"A wrong type of transformation requested (ncolours != 3).\n");
      return(1);
    }

    if( filter == NULL ) {
      /* colour frame */

      width = p->width;
      height = p->height;
      npixels = 3*width*height;

      naxis = 3;
      naxes[0] = width;
      naxes[1] = height;
      naxes[2] = 3;
      flux = malloc(npixels*sizeof(float));

      /* loop over all XYZ pixels, prepared by dcraw, to get all tree colours */
      f = flux;
      d = p->data;
      for( j = 0; j < npixels; j++)
	*f++ = *d++;
    }
    else {

      width = p->width;
      height = p->height;

      naxis = 2;
      naxes[0] = width;
      naxes[1] = height;
      flux = malloc(width*height*sizeof(float));
      npixels = width*height;

      /* select index */
      if( strcmp(filter,"X") == 0 || strcmp(filter,"B") == 0 )
	idx = 2;
      else if( strcmp(filter,"Y") == 0 || strcmp(filter,"V") == 0 )
	idx = 1;
      else if( strcmp(filter,"Z") == 0 || strcmp(filter,"R") == 0 )
	idx = 0;
      else
	idx = 1;

      /* selecting XYZ */
      if( strcmp(filter,"X") == 0 || strcmp(filter,"Y") == 0 ||
	  strcmp(filter,"Z") == 0 ) {

	f = flux;
	d = (p->data + idx*npixels);

	for( j = 0; j < npixels; j++)
	  *f++ = *d++;

      }
      else if( strcmp(filter,"R") == 0 || strcmp(filter,"V") == 0 ||
	       strcmp(filter,"B") == 0 ) {

	k = 0;
	for(j = 0; j < height; j++) {
	  jj = j*width;
	  for(i = 0; i < width; i++) {
	    for(m = 0; m < 3; m++)
	      zyx[m] = p->data[m*npixels + jj + i];
	    matmul(cmatrix,zyx,bvr);
	    flux[k++] = bvr[idx];
	  }
	}
      }
      else if( strcmp(filter,"s") == 0 ) {

	k = 0;
	for(j = 0; j < height; j++) {
	  jj = j*width;
	  for(i = 0; i < width; i++) {
	    for(m = 0; m < 3; m++)
	      zyx[m] = p->data[m*npixels + jj + i];
	    flux[k++] = zyx[0]*scovec[0] + zyx[1]*scovec[1] + zyx[2]*scovec[2];
	  }
	}
      }
      else if( strcmp(filter,"u") == 0 ) {

	k = 0;
	for(j = 0; j < height; j++) {
	  jj = j*width;
	  for(i = 0; i < width; i++) {
	    for(m = 0; m < 3; m++)
	      zyx[m] = p->data[m*npixels + jj + i];
	    flux[k++] = zyx[0] + zyx[1] + zyx[2];
	  }
	}
      }
    }
  }

  else if( type == 1 ) {
    /* instrumental */

    if( strcmp(filter,"all") == 0 ) {

      width = p->width/2;
      height = p->height/2;

      naxis = 3;
      naxes[0] = width;
      naxes[1] = height;
      naxes[2] = 4;
      flux = malloc(4*width*height*sizeof(float));

      /* loop over all pixels and separate instrumental RGGB to four bands */
      /* colours are arranged as: */
      /* flux[0..npixels] (red), */
      /* flux[npixels+1..2*npixels] (green1), */
      /* flux[2*npixels+1..3*npixels] (green2), */
      /* flux[3*npixels+1..4*npixels] (blue) */

      k = 0;
      npixels = width*height;
      for(j = 0; j < 2*height; j = j + 2)
	for(i = 0; i < 2*width; i = i + 2) {

	  flux[k] = array(p->width,p->height,p->data,i+1,j);
	  flux[npixels+k] = array(p->width,p->height,p->data,i+1,j+1);
	  flux[2*npixels+k] = array(p->width,p->height,p->data,i,j);
	  flux[3*npixels+k] = array(p->width,p->height,p->data,i,j+1);

	  k++;
	}

    }

    else if( strcmp(filter,"P") == 0 ) {

      width = p->width;
      height = p->height;

      naxis = 2;
      naxes[0] = width;
      naxes[1] = height;

      flux = malloc(width*height*sizeof(float));

      /* loop over all RGGB pixels, prepared by dcraw, to get all tree colours*/
      f = flux;
      d = p->data;
      npixels = width*height;
      for( j = 0; j < npixels; j++)
	*f++ = *d++;

    }
    else if( strcmp(filter,"Ri") == 0 || strcmp(filter,"Gi") == 0 ||
	     strcmp(filter,"Bi") == 0 || strcmp(filter,"Gi1") == 0 ||
	     strcmp(filter,"Gi2") == 0 ) {

      width = p->width/2;
      height = p->height/2;

      naxis = 2;
      naxes[0] = width;
      naxes[1] = height;
      flux = malloc(width*height*sizeof(float));
      npixels = width*height;

      if( strcmp(filter,"Gi") == 0 ) {

	k = 0;
	for(j = 0; j < 2*height; j = j + 2 )
	  for(i = 0; i < 2*width; i = i + 2 ) {
	    g1 = array(p->width,p->height,p->data,i+1,j+1);
	    g2 = array(p->width,p->height,p->data,i,j);
	    flux[k++] = (g1 + g2)/2.0;
	  }
      }
      else {

	ix = 0; jx = 0;

	if( strcmp(filter,"Ri") == 0 ) { ix = 0; jx = 1; }
	if( strcmp(filter,"Bi") == 0 ) { ix = 1; jx = 0; }
	if( strcmp(filter,"Gi1") == 0 ){ ix = 1; jx = 1; }
	if( strcmp(filter,"Gi2") == 0 ){ ix = 0; jx = 0; }

	k = 0;
	for(j = 0; j < 2*height; j = j + 2)
	  for(i = 0; i < 2*width; i = i + 2)
	    flux[k++] = array(p->width,p->height,p->data,i+ix,j+jx);
      }
    }
  }
  pixmap_free(p);

  assert(flux != NULL);

  /* output to FITS file */
  npixels = 1;
  for(k = 0; k < naxis; k++)
    npixels = npixels*naxes[k];


  /* cutoff values to a specified range */
  if( bitpix > 0 ) {
    maxval = pow(2.0,1.0*bitpix) - 1.0 - DBL_EPSILON;
    minval = DBL_EPSILON;
    for(i = 0; i < npixels; i++) {
      if( flux[i] > maxval )
	flux[i] = maxval;
      if( flux[i] < minval )
	flux[i] = minval;
    }
  }

  /* select output data representation */
  switch(bitpix) {
  case 8:   bitpix = BYTE_IMG;  break;
  case 16:  bitpix = USHORT_IMG; break;
  case 32:  bitpix = ULONG_IMG;  break;
  case -32: bitpix = FLOAT_IMG;  break;
  default:  bitpix = USHORT_IMG; break;
  }

  status = 0;
  fits_create_file(&file, fitsname, &status);

  fits_create_img(file, bitpix, naxis, naxes, &status);

  /* determine photometry system */
  if( filter == NULL ) {
    fits_update_key(file,TSTRING,"CSPACE","CIE 1931 XYZ",
		    "Colour space (CIE 1931 XYZ colour space)",&status);
  }
  else {
    photsys = "Instrumental";
    if( type == 0 ) {
      if ( strcmp(filter,"X") == 0 || strcmp(filter,"Y") == 0 || strcmp(filter,"Z") == 0 )
	photsys = "CIE 1931 XYZ";
      else if( strcmp(filter,"B") == 0 || strcmp(filter,"V") == 0 || strcmp(filter,"R") == 0 )
	photsys = "Johnson";
    }
    fits_update_key(file,TSTRING,"PHOTSYS",photsys,"photometry filter system",&status);
    fits_update_key(file,TSTRING,"FILTER",filter,
		    "Spectral filter or colour space component",&status);
  }

  fits_update_key(file,TSTRING,"DATE-OBS",stime,"time of exposure",&status);
  fits_update_key_fixflt(file,"EXPTIME",exp,5,"[s] Exposure time",&status);
  fits_update_key(file,TSHORT,"ISO",&iso,"ISO speed",&status);
  fits_update_key(file,TSTRING,"INSTRUME",camera,
		  "Camera manufacturer and model",&status);
  fits_update_key(file,TSTRING,"APERTURE",aperture,"Aperture",&status);
  if( focus > 0.0 )
    fits_update_key_fixflt(file,"FOCUS",focus,1,"[mm] Focal length",&status);
  if( dark ) {
    add = malloc(strlen(dark_comment)+strlen(dark)+1);
    strcpy(add,dark_comment);
    strcat(add,dark);
    fits_write_comment(file,add,&status);
    free(add);
  }
  if( com ) {
    add = malloc(strlen(rum_comment)+strlen(com)+1);
    strcpy(add,rum_comment);
    strcat(add,com);
    fits_write_comment(file,add,&status);
    free(add);
  }
  fits_update_key(file,TSTRING,"CREATOR","rawtran " VERSION,
		  "Created by rawtran " VERSION,&status);
  fits_write_comment(file,ver,&status);

  fits_write_comment(file,"EXIF data info - begin",&status);
  for( i = 0; i < nexif; i++)
    fits_write_comment(file,exif[i],&status);
  fits_write_comment(file,"EXIF data info - end",&status);

  fits_write_img(file, TFLOAT, 1, npixels, flux, &status);

  fits_close_file(file, &status);

  fits_report_error(stderr,status);

  free(flux);
  free(com);

  return(status);
}
