/*
   Name: $RCSfile: strutil.c,v $
   Author: Alan Moran
   $Date: 2005/11/13 20:57:21 $
   $Revision: 1.11 $
   $Id: strutil.c,v 1.11 2005/11/13 20:57:21 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief String related utility functions.

   Since so much of what rapple does is concerned with the manipulation of
   strings it is appropriate that there exists a string module that defines a
   string type (rpl_str_t) along with operations that frequently arise during
   processing (e.g., concatentation, spliting etc.)

 */

#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "globals.h"

/**
   Returns the concatenation of a RPL_STR_EOC terminated variadic list of strings.

   @param start

   @note This function has variable arguments, last argument must be RPL_STR_EOC.

   @return the concatenation of a RPL_STR_EOC terminated variadic list of strings.
 */
rpl_str_t
rpl_str_concat(rpl_c_str_t start, ...) {
    size_t length;
    rpl_str_t buff;
    va_list argp;
    rpl_str_t strp;

    assert(start != NULL);

    /* scan the vargs and allocate sufficient memory */
    length = strlen(start);
    va_start(argp, start);
    while((strp = va_arg(argp, rpl_str_t)) != NULL)
        length += strlen(strp);
    va_end(argp);
    buff = (rpl_str_t)rpl_me_malloc(length + 1);

    /* perform concatenation */
    strcpy(buff, start);
    va_start(argp, start);
    while((strp=va_arg(argp, rpl_str_t)) != NULL)
        strcat(buff, strp);
    va_end(argp);

    return buff;
}

/**
   Eeturns the position of the character c in the string str.

   @param str
   @param c

   @return the position of the character c in the string str.
 */
size_t
rpl_str_index_of(rpl_c_str_t str, char c) {
    size_t i;
    for(i=0;str[i];i++)
        if(str[i] == c)
            return i;
    return -1;
}

/**
   Splits into key/value pair the string str at the first occurence of character delim.

   @param str
   @param delim
   @param key
   @param value
 */
void
rpl_str_split(rpl_c_str_t str, char delim, rpl_str_t *key, rpl_str_t *value) {
    rpl_str_t kp, vp;
    size_t length, k_len, v_len;

    /* locate the key/value pointers */
    length = strlen(str) + 1;
    kp = (rpl_str_t)rpl_me_malloc(length);
    snprintf(kp, length, "%s", str);
    /* special care is required if the delimiter is not found */
    if((vp = strchr(kp, delim)) != NULL) {
        *vp = '\0';
        vp++;
    } else {
        /* note: string null not NULL */
        vp = RPL_STR_NUL;
    }

    /* assign the key */
    k_len = strlen(kp) + 1;
    *key = (rpl_str_t)rpl_me_malloc(k_len);
    snprintf(*key, k_len, "%s", kp);

    /* assign the value */
    v_len = strlen(vp) + 1;
    *value = (rpl_str_t)rpl_me_malloc(v_len);
    snprintf(*value, v_len, "%s", vp);

    /* free resources (only kp was allocated) */
    rpl_me_free(kp);

}

/**
   Splits into key/value pair the string str at the last occurence of character delim.

   @param str
   @param delim
   @param key
   @param value
 */
void
rpl_str_rsplit(rpl_c_str_t str, char delim, rpl_str_t *key, rpl_str_t *value) {
    rpl_str_t kp, vp;

	assert(str != NULL);

	kp = strdup(str);

    /* special care is required if the delimiter is not found */
    if((vp = strrchr(kp, delim)) != NULL) {
        *vp = '\0';
        vp++;
    } else {
        vp = kp;
        kp = RPL_STR_NUL;
    }

	*key = strdup(kp);
	*value = strdup(vp);

	if(strlen(kp) > 0)	
		rpl_me_free(kp);
}
