/*
   Name: $RCSfile: mime.c,v $
   Author: Alan Moran
   $Date: 2005/10/25 13:53:03 $
   $Revision: 1.11 $
   $Id: mime.c,v 1.11 2005/10/25 13:53:03 a_j_moran Exp $

   Description:


   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Web Asset file manager.

   The behaviour of rapple for the most part is determined by the type of asset
   it is processing.  For example whilst XHTML files are transformated, binary
   files (e.g., images or PDF documents) are simply copied without any
   transformation.  The type of an asset can be determined depending on the
   context e.g., if it is delivered by HTTP then the MIME type can be used
   otherwise if it is a local file then its type can be inferred from its
   extension.  The mime module provides supporting functionality that is used
   by rapple when processing assets (e.g., it can inform the processor whether
   or not a given type is deemed transformable etc.)

   Note that the mime module is not concerned with the parsing of MIME messages
   or related functionality as this is already covered by the http module.

*/

#include "globals.h"
#include "mime.h"

static const rpl_mime_type mime_types[] = {
    { "text/html", 1 },
    { "application/x-httpd-php" , 1 },
    { "text/css", 0 },
    { "image/jpeg", 0 },
    { "image/png", 0 },
    { "image/gif", 0 },
    { "application/pdf", 0 },
    { "text/xml", 0 },
    { "application/x-shockwave-flash", 0 },
};
static int  mime_types_size = sizeof(mime_types) / sizeof(mime_types[0]);

static const rpl_mime_mapping mime_mappings[] = {
    { "css",    "text/css" },
    { "html",   "text/html" },
    { "htm",    "text/html" },
    { "php",    "application/x-httpd-php" },
    { "png",    "image/png" },
    { "jpeg",   "image/jpeg" },
    { "jpg",    "image/jpeg" },
    { "gif",    "image/gif" },
    { "pdf",    "application/pdf" },
    { "xsl",    "text/xml" },
    { "xslt",   "text/xml" },
    { "xml",    "text/xml" },
    { "swf",    "application/x-shockwave-flash" },
};
static int  mime_mappings_size = sizeof(mime_mappings) / sizeof(mime_mappings[0]);

/**
   Returns the mime_type based on extension of filename. Returns NULL if type cannot be
   determined.

   @param filename

   @return the mime_type based on extension of filename or NULL if type cannot be
   determined.
 */
rpl_str_t
rpl_mime_get_type(rpl_c_str_t filename)
{
    int i;
    rpl_str_t mime_type = NULL, ext;

    assert(filename != NULL);

    if((ext = strrchr(filename,'.')) != NULL)
    {
        ext++;
        /* lookup MIME type using ext as key */
        for(i = 0; i<mime_mappings_size; i++)
        {
            if(strcmp(ext, mime_mappings[i].extension) == 0)
                break;
        }

        /* if found, assign the appropriate mime_type */
        if(i < mime_mappings_size)
        {
            mime_type = (rpl_str_t) rpl_me_malloc(strlen(mime_mappings[i].mime_type) + 1);
            strcpy(mime_type, mime_mappings[i].mime_type);
        }
    }

    if (mime_type == NULL)
    {
        /* need to do this so that the string returned by this function can be rpl_me_free'd */
        mime_type = (rpl_str_t) rpl_me_malloc (strlen ("n/a") + 1);
        strcpy (mime_type, "n/a");
    }
    return mime_type;
}

/**
   Determines whether an asset corresponding to mime_type can be transformed (tidy, XSLT e
   etc.).

   @param mime_type

   @return 1 (transformable), 0 (non-transformable), -1 (unknown type).
 */
int
rpl_mime_is_transformable(rpl_str_t mime_type)
{
    int i;

    assert(mime_type != NULL);

    for(i=0; i<mime_types_size; i++)
       if(strcmp(mime_types[i].name, mime_type) == 0)
          return mime_types[i].transformable;

    /* MIME type not found */
    return -1;
}

/**
   Determines whether an asset possesses a supported MIME type.

   @param mime_type

   @return 1 (supported), 0 (unsupported).
 */
int
rpl_mime_is_supported_type(rpl_str_t mime_type)
{
    int i;

    assert(mime_type != NULL);

    for(i=0; i<mime_types_size; i++)
       if(strcmp(mime_types[i].name, mime_type) == 0)
           return 1;

    return 0;
}
