/*
   Name: $RCSfile: html.c,v $
   Author: Alan Moran
   $Date: 2005/09/18 15:12:46 $
   $Revision: 1.11 $
   $Id: html.c,v 1.11 2005/09/18 15:12:46 a_j_moran Exp $

   Legal Notice:

   This program is free software; you can redistribute it and/or
   modify it under the terms of the license contained in the
   COPYING file that comes with this distribution.

 */

/**
   @file

   @brief Functions to support generation of HTML documents.

   The html module caters for the need to perform rudimentary generation of
   HTML fragments (e.g. the creation of a hyperlink) or the processing of
   documents (e.g. the replacement of newlines with HTML style break elements
   etc.).  It is most often invoking by modules concerned with the generation
   of webpages (e.g. the cataloging functionality).
*/

#include <stdarg.h>

#include "globals.h"
#include "html.h"

static rpl_c_str_t br = "<br />\n";

/**
   Returns the contents of the string src with \n characters replaced by "<br />\n".
   intended for HTML interpretation of textual content.

   @param src

   @return the contents of the string src with \n characters replaced by "<br />\n".
 */
rpl_str_t
rpl_html_nl2br(rpl_c_str_t src) {
    rpl_str_t dest;
    size_t i, s_length, j = 0, br_length = strlen(br);
    int no_nl = 0;

    assert(src != NULL);

    /* count the number of newlines */
    s_length = strlen(src);
    for(i=0; i<s_length; i++)
        if(src[i] == '\n')
            no_nl++;

    /* perform replacements */
    dest = (rpl_str_t)rpl_me_malloc(s_length + no_nl * br_length + 1);
    *dest = '\0';
    for(i=0; i<s_length; i++) {
        if(src[i] == '\n') {
            snprintf(dest+j, br_length, "%s", br);
            j += br_length - 1;
        } else if (src[i] == '\r') {
            dest[j++] = ' ';
        } else {
            dest[j++] = src[i];
        }
    }
    dest[j] = '\0';

    return dest;
}

/**
   Constructs a hyperlink to filename (relative to datastore subdirectory) using hypertext
   provided. Original intent as part of digest/catalog functionality.

   @param filename
   @param hypertext

   @return hyperlink to filename (relative to datastore subdirectory).
 */
rpl_str_t
rpl_html_create_hyperlink(rpl_str_t filename, rpl_str_t hypertext) {
    assert((filename != NULL) && (hypertext != NULL));
    return rpl_str_concat("<a href=\"", rpl_cfg_get_trf_tpl_domain(), filename, "\">", hypertext, "</a>", RPL_STR_EOC);
}

/**
   Convenience function that returns a "more" hyperlink. Original intent as part of
   digest/catalog functionality.

   @param filename

   @return a "more" hyperlink.
 */
rpl_str_t
rpl_html_create_more_hyperlink(rpl_str_t filename) {
    return rpl_html_create_hyperlink(filename, "more");
}
