/*Randim: generates image from user-specified number of affine
  transformations in 2- or 3-d space which are generated randomly
  according to a given template.  The tranformations are then applied in
  random sequence to a starting point which is returned to outside a given
  range.

  Written by: Adrian Robert, based on work of Michael Barnsley with
    conceptual inspiration from a program by Graham Macnamara.
    Animation mode inspired by a mode by Massimino Pascal for xlock.

  Mods, from version 1.0 on:
  Version 2.0 - added autoscaling, modified to load and save pictures,
  		added intelligent save mode - 1/2/91.
  Version 3.0 - added features to load, enhanced transformation possibilities.
  Version 3.1 - added runtime user determination of transformation
  		number and type, temporarily removed load/save capability.
  Version 3.2 - added scaling display and zoom mode - 5/26/92.
  Version 3.3 - added point zoom mode, added load/save for any zoom
  		level with standard record format - 7/25/92.
  Version 4.0 - ported to Sunview and enhanced, 9/30/92.
  		4.01 - added save as datafile feature, 11/13/92.
  Version 4.1 - added 3d images - 11/15/92.
  Version 4.2 - added multiple transformation alteration feature,
  		probability table - 11/20/92.
  Version 4.3 - ported to XWindows and restored zoom features - 2/20/95.
  Version 4.4 - added graphical display and edit of individual
		transformations	- 9/15/95.
  Version 4.5 - added slideshow mode, color plotting - 10/16/96.
  Version 4.6 - updated display of individual transformations
  Version 4.7 - fixed some bugs with interface, added animation mode,
		cleaned up code - 11/23/97.
  Version 4.8 - added depth color mode 4/4/00.
  Version 4.9a - added multi-colormap capability and many new maps - 3/25/01.
  Version 5.0 - added log-hits based intensity mode - 4/7/01.
*/

/* This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.  Please see the file "COPYING"
   for details.  If you have not received it along with this program,
   please write to the Free Software Foundation, Inc., 675 Mass. Ave.,
   Cambridge, MA 02139, USA. */

#include "randim.h"
#include "randefs.h"


/**********************************************************************/
/* initialization */

/* main() initializes Xt main widget, calls init() to set up windows and
   variables. */

int
main(int argc, char **argv)
{
  int	hsz,vsz;

  /* check for a colorfile specification */
  if (argc > 1) {
    strcpy(cmapName, argv[1]);
  } else {
    cmapName[0] = '\0';
  }

  hsz = HSIZE, vsz = VSIZE;
  printf("\n\nWelcome to the random transformation image generator.\n");
  printf("This is version 5.0.\n");

  /*Xtoolkit initialization - returns parent widget*/
  pwidget = XtVaAppInitialize(&appcontext,"Randim",NULL,0,&argc,argv,
			      NULL,NULL);
  if (appcontext==NULL) {		/*something didn't work*/
    fprintf(stderr,"\nProblems, mate.\n");
    exit(-1);
      }

  init(); /*set up windows and variables*/

  /*Now just let Xt run everything, calling sections of code on events.*/
  XtAppMainLoop(appcontext);

  return(0);

}


void
init()
{
  float	sq3o2 = 0.866025403;	/*square root of 3 over 2*/
  int rlength;			/*bit length of random numbers*/

  /*set value of a shift variable used later based on length of rand()
    numbers on this machine*/
  rlength = rint(log((double) RAND_MAX + 1)/log(2.0));
  RANDSHIFT = rlength - 15;
  /* if using qrand() in apply_trans() increment RANDSHIFT: */
  RANDSHIFT++;	/* NOTE: do only if using qrand() */

  /*Set initial screensize, initial transrep subwindow size*/
  hsize = HSIZE, vsize = VSIZE;
  hsize2 = HSIZE2, vsize2 = VSIZE2;
  rhsize2 = 25, rvsize2 = 25;

  /* general image plotting parameters */
  ploopn = N/PN;	/* number used in draw()'s plot loop*/
  slidepoints = animpoints = denspoints = 0; /* we start out in normal mode*/
  animstep = 0.1, animpersist = 20, n_mods = 3;

  /*set 3d-2d projection angles*/
  _cx = sq3o2;
  _sx = 0.5;	/*30 degrees*/
  _cy = sq3o2;
  _sy = 0.5;	/*30 degrees*/
  _cz = 0;
  _sz = 1.0;	/*90 degrees*/

  isLargeCmap = 0;
  win_setup();	/*set up the window structures and request mapping*/

  /*Set the initial values for generation parameters*/
  ttype = 7;
  logtn = 2;
  loghtn = 0;
  tmodn = atmodn = 0;
  seed = 77;
  dimension = 2;
  asize = 12;
  zmag = 2.0;

  /*Set some flags and toggles.*/
  drawflag = 0;	/*won't draw until window mapped*/
  bsstate = 0;	/*start out with no backing store*/
  cmode = 0;	/*start in blackandwhite*/
  diFlag = 0;   /*not in density-intensity mode*/
  diffEqFlag = 0; /*start in conventional mode*/

}

/**********************************************************************/
/* utilities */

/* dr() returns a double-precision random number uniform on [0,1) (because
   not all systems seem to have a drand48())*/
double
dr()
{
  return ((double) random()/(RAND_MAX + 1.0));
}

/* shuffle() puts the integers 0..n-1 in the array list */
void
shuffle(int *list, int n)
{
  int		i,r;

  for (i=0;i<n;i++) list[i] = -1;
  for (i=0;i<n;i++) {
    for (r = nrand(n); list[r] != -1; r = (r + 1) % n);
    list[r] = i;
    }
}

/*Function memfail prints a failure to allocate message and exits.*/
void
memfail(char *item)
{
    fprintf(stderr,"\n***Memory allocation failed for %s.  Sorry, dude.\n",
	    item);
    exit(1);
}

