# Copyright 2014: Rackspace UK
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import novaclient.exceptions

from rally.benchmark.context import base
from rally.benchmark.context.cleanup import manager as resource_manager
from rally.common.i18n import _
from rally.common import log as logging
from rally.common import utils
from rally import osclients


LOG = logging.getLogger(__name__)


@base.context(name="keypair", order=310)
class Keypair(base.Context):
    KEYPAIR_NAME = "rally_ssh_key"

    def _generate_keypair(self, endpoint):
        keypair_name = "%s_%s" % (
            self.KEYPAIR_NAME, self.context["task"]["uuid"])

        nova_client = osclients.Clients(endpoint).nova()

        # NOTE(hughsaunders): If keypair exists, it must be deleted as we can't
        # retrieve the private key
        try:
            nova_client.keypairs.delete(keypair_name)
        except novaclient.exceptions.NotFound:
            pass

        keypair = nova_client.keypairs.create(keypair_name)
        return {"private": keypair.private_key,
                "public": keypair.public_key,
                "name": keypair_name,
                "id": keypair.id}

    @utils.log_task_wrapper(LOG.info, _("Enter context: `keypair`"))
    def setup(self):
        for user in self.context["users"]:
            user["keypair"] = self._generate_keypair(user["endpoint"])

    @utils.log_task_wrapper(LOG.info, _("Exit context: `keypair`"))
    def cleanup(self):
        # TODO(boris-42): Delete only resources created by this context
        resource_manager.cleanup(names=["nova.keypairs"],
                                 users=self.context.get("users", []))
