/*
 *  Copyright 2005 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifndef _GENDATA_H
#define _GENDATA_H

#include <iostream>
#include "redfsm.h"
#include "fsmcodegen.h"

struct NameInst;
typedef DList<Action> ActionList;

/* The minimum chars are the absolute value of the real minimum because the
 * sign is stored separately in integers read in so they are compared in the
 * positive. Each is casted to an unsigned because the data part of the number
 * is in unsigned int size. 
 */
#define RL_CHAR_MIN    ((long)((char)CHAR_MIN))
#define RL_CHAR_MAX    ((long)((char)CHAR_MAX))
#define RL_UCHAR_MIN   ((unsigned long)((unsigned char)0))
#define RL_UCHAR_MAX   ((unsigned long)((unsigned char)UCHAR_MAX))
#define RL_SHORT_MIN   ((long)((short)SHRT_MIN))
#define RL_SHORT_MAX   ((long)((short)SHRT_MAX))
#define RL_USHORT_MIN  ((unsigned long)((unsigned short)0))
#define RL_USHORT_MAX  ((unsigned long)((unsigned short)USHRT_MAX))
#define RL_INT_MIN     ((long)((int)INT_MIN))
#define RL_INT_MAX     ((long)((int)INT_MAX))
#define RL_UINT_MIN    ((unsigned long)((unsigned int)0))
#define RL_UINT_MAX    ((unsigned long)((unsigned int)UINT_MAX))
#define RL_LONG_MIN    ((long)LONG_MIN)
#define RL_LONG_MAX    ((long)LONG_MAX)
#define RL_ULONG_MIN   ((unsigned long)0)
#define RL_ULONG_MAX   ((unsigned long)LONG_MAX)

/* Types of alphabet supported by Ragel. */
enum AlphType
{
	AT_Char,
	AT_UnsignedChar,
	AT_Short,
	AT_UnsignedShort,
	AT_Int,
	AT_UnsignedInt
};


/* Context statement, kept unique by name. */
struct Context
{
	Context( char *data, int id )
		: data(data), id(id), declared(false) {}

	char *data;
	int id;
	bool declared;

	Context *prev, *next;
};
typedef AvlMap<char *, Context*, CmpStr> ContextMap;
typedef AvlMapEl<char *, Context*> ContextMapEl;
typedef DList<Context> ContextList;

typedef AvlMap<char *, CodeGenData*, CmpStr> CodeGenMap;
typedef AvlMapEl<char *, CodeGenData*> CodeGenMapEl;

#define WO_NOEND    0x01
#define WO_NOERROR  0x02
#define WO_NOPREFIX 0x04
#define WO_NOFF     0x08

struct CodeGenData
{
	CodeGenData( char *fileName, char *fsmName, bool wantComplete )
	:
		fileName(fileName),
		fsmName(fsmName), 
		redFsm(0), 
		allActions(0),
		allActionTables(0),
		allStates(0),
		nameIndex(0),
		startState(0),
		alphType(AT_Char),
		getKeyExpr(0),
		accessExpr(0),
		curStateExpr(0),
		lowKey(0),
		highKey(0),
		codeGen(0),
		wantComplete(wantComplete),
		writeOps(0),
		writeData(false),
		writeInit(false),
		writeExec(false),
		writeEOF(false),
		hasLongestMatch(false),
		hasEnd(true),
		dataPrefix(true),
		writeFirstFinal(true),
		writeErr(true),
		hasBeenPrepared(false)
	{ }

	/* 
	 * Collecting the machine.
	 */

	char *fileName;
	char *fsmName;
	RedFsmAp *redFsm;
	Action *allActions;
	RedAction *allActionTables;
	RedStateAp *allStates;
	NameInst **nameIndex;
	int startState;
	ActionList actionList;
	ContextMap contextMap;
	AlphType alphType;
	InlineList *getKeyExpr;
	InlineList *accessExpr;
	InlineList *curStateExpr;
	long lowKey, highKey;
	FsmCodeGen *codeGen;
	KeyOps keyOps;
	bool wantComplete;
	int writeOps;
	bool writeData;
	bool writeInit;
	bool writeExec;
	bool writeEOF;
	EntryIdVect entryPointIds;
	EntryNameVect entryPointNames;
	bool hasLongestMatch;

	/* Write options. */
	bool hasEnd;
	bool dataPrefix;
	bool writeFirstFinal;
	bool writeErr;

	void createMachine();
	void initActionList( unsigned long length );
	void newAction( int anum, char *name, int line, int col, InlineList *inlineList );
	void initActionTableList( unsigned long length );
	void initStateList( unsigned long length );
	void setStartState( unsigned long startState );
	void addEntryPoint( char *name, unsigned long entryState );
	void setFinal( int snum );
	void initTransList( int snum, unsigned long length );
	void newTrans( int snum, int tnum, long lowKey, long highKey, 
			long targ, long act );
	void finishTransList( int snum );
	void setStateActions( int snum, long toStateAction, 
			long fromStateAction, long eofAction );
	void finishMachine();
	void setForcedErrorState()
		{ redFsm->forcedErrorState = true; }

	bool isAlphSigned();
	bool setAlphType( char *data );
	void setKeyOps();

	void makeCodeGen();
	void generateGraphviz();
	void resolveTargetStates( InlineList *inlineList );

	void generate();
	void generateCode();
	void prepareMachine();
	bool hasBeenPrepared;
};

extern CodeGenData *cgd;

void lineDirective( ostream &out, char *fileName, int line );
void genLineDirective( ostream &out );

#endif /* _GENDATA_H */
