/*
 *  Copyright 2001-2005 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <iostream>
#include <fstream>
#include <unistd.h>
#include <sstream>

/* Parsing. */
#include "ragel.h"
#include "rlparse.h"
#include "parsetree.h"

/* Parameters and output. */
#include "pcheck.h"
#include "vector.h"
#include "version.h"

using std::ostream;
using std::ofstream;
using std::cout;
using std::cerr;
using std::endl;

/* Io globals. */
ostream *outStream = 0;
char *outputFileName = 0;

/* Controls minimization. */
MinimizeLevel minimizeLevel = MinimizePartition2;
bool minimizeEveryOp = false;

/* Graphviz dot file generation. */
char *machineSpec = 0, *machineName = 0;
bool machineSpecFound = false;

/* Print a summary of the options. */
void usage()
{
	cout <<
"usage: ragel [options] file\n"
"general:\n"
"   -h, -H, -?, --help    Print this usage and exit\n"
"   -v, --version         Print version information and exit\n"
"   -o <file>             Write output to <file>\n"
"fsm minimization:\n"
"   -m                    Minimize the number of states (default)\n"
"   -n                    Do not perform minimization\n"
"output:\n"
"   -S <spec>             FSM specification to output for -V\n"
"   -M <machine>          Machine definition/instantiation to output for -V\n"
	;	
}

/* Print version information. */
void version()
{
	cout << "Ragel State Machine Compiler version " VERSION << " " PUBDATE << endl <<
			"Copyright (c) 2001-2005 by Adrian Thurston" << endl;
}

/* Global parse data pointer. */
extern InputData *id;

/* Total error count. */
int gblErrorCount = 0;

/* Print the opening to a program error, then return the error stream. */
ostream &error()
{
	/* Keep the error count. */
	if ( id != 0 && id->pd != 0 )
		id->pd->errorCount += 1;
	gblErrorCount += 1;
	
	cerr << PROGNAME ": ";
	return cerr;
}

/* Print the opening to an error in the input, then return the error ostream. */
ostream &error( const BISON_YYLTYPE &loc )
{
	/* Keep the error count. */
	if ( id->pd != 0 )
		id->pd->errorCount += 1;
	gblErrorCount += 1;

	cerr << id->fileName << ":" << loc.first_line << ":" << 
			loc.first_column << ": ";
	return cerr;
}

/* Print the opening to an error in the input, then return the error ostream. */
ostream &error( const InputLoc &loc )
{
	/* Keep the error count. */
	if ( id->pd != 0 )
		id->pd->errorCount += 1;
	gblErrorCount += 1;

	cerr << id->fileName << ":" << loc.line << ":" << 
			loc.col << ": ";
	return cerr;
}

ostream &error( int first_line, int first_column )
{
	/* Keep the error count. */
	if ( id->pd != 0 )
		id->pd->errorCount += 1;
	gblErrorCount += 1;

	cerr << id->fileName << ":" << first_line << ":" << first_column << ": ";
	return cerr;
}

/* Print the opening to a warning, then return the error ostream. */
ostream &warning( )
{
	cerr << id->fileName << ": warning: ";
	return cerr;
}

/* Print the opening to a warning in the input, then return the error ostream. */
ostream &warning( const InputLoc &loc )
{
	cerr << id->fileName << ":" << loc.line << ":" << 
			loc.col << ": warning: ";
	return cerr;
}

/* Print the opening to a warning in the input, then return the error ostream. */
std::ostream &warning( int first_line, int first_column )
{
	cerr << id->fileName << ":" << first_line << ":" << 
			first_column << ": warning: ";
	return cerr;
}

void escapeLineDirectivePath( std::ostream &out, char *path )
{
	for ( char *pc = path; *pc != 0; pc++ ) {
		if ( *pc == '\\' )
			out << "\\\\";
		else
			out << *pc;
	}
}

/* Construct a new parameter checker with for paramSpec. */
ParamCheck::ParamCheck(char *paramSpec, int argc, char **argv)
:
	state(noparam),
	argOffset(0),
	curArg(0),
	iCurArg(1),
	paramSpec(paramSpec), 
	argc(argc), 
	argv(argv)
{
}

/* Check a single option. Returns the index of the next parameter.  Sets p to
 * the arg character if valid, 0 otherwise.  Sets parg to the parameter arg if
 * there is one, NULL otherwise. */
bool ParamCheck::check()
{
	bool requiresParam;

	if ( iCurArg >= argc ) {            /* Off the end of the arg list. */
		state = noparam;
		return false;
	}

	if ( argOffset != 0 && *argOffset == 0 ) {
		/* We are at the end of an arg string. */
		iCurArg += 1;
		if ( iCurArg >= argc ) {
			state = noparam;
			return false;
		}
		argOffset = 0;
	}

	if ( argOffset == 0 ) {
		/* Set the current arg. */
		curArg = argv[iCurArg];

		/* We are at the beginning of an arg string. */
		if ( argv[iCurArg] == 0 ||        /* Argv[iCurArg] is null. */
			 argv[iCurArg][0] != '-' ||   /* Not a param. */
			 argv[iCurArg][1] == 0 ) {    /* Only a dash. */
			parameter = 0;
			parameterArg = 0;

			iCurArg += 1;
			state = noparam;
			return true;
		}
		argOffset = argv[iCurArg] + 1;
	}

	/* Get the arg char. */
	char argChar = *argOffset;
	
	/* Loop over all the parms and look for a match. */
	char *pSpec = paramSpec;
	while ( *pSpec != 0 ) {
		char pSpecChar = *pSpec;

		/* If there is a ':' following the char then
		 * it requires a parm.  If a parm is required
		 * then move ahead two in the parmspec. Otherwise
		 * move ahead one in the parm spec. */
		if ( pSpec[1] == ':' ) {
			requiresParam = true;
			pSpec += 2;
		}
		else {
			requiresParam = false;
			pSpec += 1;
		}

		/* Do we have a match. */
		if ( argChar == pSpecChar ) {
			if ( requiresParam ) {
				if ( argOffset[1] == 0 ) {
					/* The param must follow. */
					if ( iCurArg + 1 == argc ) {
						/* We are the last arg so there
						 * cannot be a parameter to it. */
						parameter = argChar;
						parameterArg = 0;
						iCurArg += 1;
						argOffset = 0;
						state = invalid;
						return true;
					}
					else {
						/* the parameter to the arg is the next arg. */
						parameter = pSpecChar;
						parameterArg = argv[iCurArg + 1];
						iCurArg += 2;
						argOffset = 0;
						state = match;
						return true;
					}
				}
				else {
					/* The param for the arg is built in. */
					parameter = pSpecChar;
					parameterArg = argOffset + 1;
					iCurArg += 1;
					argOffset = 0;
					state = match;
					return true;
				}
			}
			else {
				/* Good, we matched the parm and no
				 * arg is required. */
				parameter = pSpecChar;
				parameterArg = 0;
				argOffset += 1;
				state = match;
				return true;
			}
		}
	}

	/* We did not find a match. Bad Argument. */
	parameter = argChar;
	parameterArg = 0;
	argOffset += 1;
	state = invalid;
	return true;
}

/* Main, process args and call yyparse to start scanning input. */
int main(int argc, char **argv)
{
	ParamCheck pc("o:enmabjkS:M:vHh?-:", argc, argv);
	char *inputFileName = 0;

	while ( pc.check() ) {
		switch ( pc.state ) {
		case ParamCheck::match:
			switch ( pc.parameter ) {
			/* Output. */
			case 'o':
				if ( *pc.parameterArg == 0 ) {
					/* Complain, someone used -o "" */
					error() << "zero length output file name" << endl;
					exit(1);
				}
				else if ( outputFileName != 0 ) {
					/* Complain, two output files given. */
					error() << "more than one output file" << endl;
					exit(1);
				}
				else {
					/* Ok, remember the output file name. */
					outputFileName = pc.parameterArg;
				}
				break;

			/* Minimization, mostly hidden options. */
			case 'e':
				minimizeEveryOp = true;
				break;
			case 'n':
				minimizeLevel = MinimizeNone;
				break;
			case 'm':
				minimizeLevel = MinimizePartition2;
				break;
			case 'a':
				minimizeLevel = MinimizeApprox;
				break;
			case 'b':
				minimizeLevel = MinimizeStable;
				break;
			case 'j':
				minimizeLevel = MinimizePartition1;
				break;
			case 'k':
				minimizeLevel = MinimizePartition2;
				break;

			/* Machine spec. */
			case 'S':
				if ( *pc.parameterArg == 0 ) {
					/* Complain, need a graph spec. */
					error() << "please specify an argument to -S" << endl;
					exit(1);
				}
				else if ( machineSpec != 0 ) {
					error() << "FSM specification to generate already given" << endl;
					exit(1);
				}
				else {
					/* Ok, remember the path to the machine to generate. */
					machineSpec = pc.parameterArg;
				}
				break;

			/* Machine path. */
			case 'M':
				if ( *pc.parameterArg == 0 ) {
					/* Complain, need a graph spec. */
					error() << "please specify an argument to -M" << endl;
					exit(1);
				}
				else if ( machineName != 0 ) {
					error() << "machine to generate already given" << endl;
					exit(1);
				}
				else {
					/* Ok, remember the machine name to generate. */
					machineName = pc.parameterArg;
				}
				break;

			/* Version and help. */
			case 'v':
				version();
				exit(0);
			case 'H': case 'h': case '?':
				usage();
				exit(0);
			case '-':
				if ( strcasecmp(pc.parameterArg, "help") == 0 ) {
					usage();
					exit(0);
				}
				else if ( strcasecmp(pc.parameterArg, "version") == 0 ) {
					version();
					exit(0);
				}
				else {
					error() << "invalid parameter" << endl;
					exit(1);
				}
			}
			break;

		case ParamCheck::invalid:
			error() << "invalid parameter" << endl;
			exit(1);

		case ParamCheck::noparam:
			/* It is interpreted as an input file. */
			if ( *pc.curArg == 0 ) {
				error() << "zero length input file name" << endl;
				exit(1);
			}
			if ( inputFileName != 0 ) {
				error() << "more than one input file" << endl;
				exit(1);
			}
			/* Remember the filename. */
			inputFileName = pc.curArg;
			break;
		}
	}

	/* If the user wants minimization at every op, but no minimization level 
	 * is given, then print a warning. */
	if ( minimizeEveryOp && minimizeLevel == MinimizeNone )
		warning() << "-e given but minimization is not enabled" << endl;

	/* Look for no input file specified. */
	if ( inputFileName == 0 )
		error() << "no input file" << endl;

	/* Bail on above errors. */
	if ( gblErrorCount > 0 )
		exit(1);

	/* Make sure we are not writing to the same file as the input file. */
	if ( outputFileName != 0 && strcmp( inputFileName, outputFileName  ) == 0 ) {
		error() << "output file \"" << outputFileName  << 
				"\" is the same as the input file" << endl;
	}

	/* Open the input file for reading. */
	yyin = fopen( inputFileName, "rt" );
	if ( yyin == 0 ) {
		error() << "could not open " << inputFileName << " for reading" << endl;
	}

	/* Bail on above errors. */
	if ( gblErrorCount > 0 )
		exit(1);

	std::ostringstream outputBuffer;
	outStream = &outputBuffer;

	/* Create the input data for the top level file. */
	id = new InputData( inputFileName, 0, 0 );

	if ( machineSpec == 0 && machineName == 0 )
		*outStream << "<host line=\"1\" col=\"1\">";

	/* Parse the input! */
	yyparse();

	if ( machineSpec == 0 && machineName == 0 )
		*outStream << "</host>\n";

	/* Finished, final check for errors.. */
	if ( gblErrorCount > 0 )
		return 1;

	checkMachines();

	if ( gblErrorCount > 0 )
		return 1;
	
	ostream *outputFile = 0;
	if ( outputFileName != 0 )
		outputFile = new ofstream( outputFileName );
	else
		outputFile = &cout;

	/* Write the machines, then the surrounding code. */
	writeMachines( *outputFile, outputBuffer.str(), inputFileName );

	if ( outputFileName != 0 )
		delete outputFile;

	return 0;
}
