/*
 *  Copyright 2001-2005 Adrian Thurston <thurston@cs.queensu.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifndef _FSMCODEGEN_H
#define _FSMCODEGEN_H

#include <iostream>

/* Integer array line length. */
#define IALL 8

/* Forwards. */
struct RedFsmAp;
struct ParseData;
struct Action;
struct InlineBlock;
struct NameInst;
struct InlineItem;
struct InlineList;
struct RedAction;
struct LongestMatch;

/*
 * class FsmCodeGen
 */
class FsmCodeGen
{
public:
	FsmCodeGen() : out(*(std::ostream*)0) {}
	FsmCodeGen( char *fsmName, ParseData *parseData, 
			RedFsmAp *redFsm, std::ostream &out );
	virtual ~FsmCodeGen() {}

	virtual void writeOutInterface() = 0;
	virtual void writeOutHeader() = 0;
	virtual void writeOutCode() = 0;
	
	/* Gather various info on the machine. */
	void analyzeAction( Action *act, InlineList *inlineList );
	void analyzeActionList( RedAction *redAct, InlineList *inlineList );
	void analyzeMachine();
	
	void startCodeGen();
	void endCodeGen( const InputLoc &loc );

protected:
	friend struct ParseData;

	std::ostream &ARRAY_TYPE( unsigned long maxVal );
	std::ostream &C_INTERFACE();
	std::ostream &C_HEADER();
	std::ostream &CPP_INTERFACE();
	std::ostream &CPP_HEADER();
	std::ostream &OBJC_INTERFACE();
	std::ostream &OBJC_HEADER();
	std::ostream &FSM_NAME();
	std::ostream &BASE_CLAUSE();
	std::ostream &USER_INIT( InlineBlock *ilBlock );
	std::ostream &INIT_CODE();
	std::ostream &START_STATE_ID();
	std::ostream &ACTIONS_ARRAY();
	std::ostream &ALPH_TYPE();
	std::ostream &EL_TYPE();
	std::ostream &GET_KEY();
	std::ostream &TABS( int level );
	std::ostream &KEY( long key );
	std::ostream &LDIR_PATH( char *path );
	std::ostream &INLINE_LIST( InlineList *inlineList, int targState, bool inFinish );
	std::ostream &ACTION( Action *action, int targState, bool inFinish );
	std::ostream &LM_SWITCH( LongestMatch *longestMatch, int targState, int inFinish );

	virtual std::ostream &CALL( NameInst *name, int targState, bool inFinish ) = 0;
	virtual std::ostream &GOTO( NameInst *name, bool inFinish ) = 0;
	virtual std::ostream &RET( bool inFinish ) = 0;
	virtual std::ostream &CURS( bool inFinish ) = 0;
	virtual std::ostream &TARGS( bool inFinish, int targState ) = 0;
	virtual std::ostream &GET_ENTRY( NameInst *name, bool inFinish ) = 0;
	virtual std::ostream &CALLE( InlineItem *ilItem, int targState, bool inFinish ) = 0;
	virtual std::ostream &GOTOE( InlineItem *ilItem, bool inFinish ) = 0;
	virtual std::ostream &NEXT( NameInst *name, bool inFinish ) = 0;
	virtual std::ostream &NEXTE( InlineItem *ilItem, bool inFinish ) = 0;
	virtual std::ostream &SELF() = 0;

	int arrayTypeSize( unsigned long maxVal );

	bool anyActions();
	bool anyEofActions()            { return bAnyEofActions; }
	bool anyActionCalls()           { return bAnyActionCalls; }
	bool anyActionRets()            { return bAnyActionRets; }
	bool anyRegActionRets()         { return bAnyRegActionRets; }
	bool anyRegActionByValControl() { return bAnyRegActionByValControl; }
	bool anyRegNextStmt()           { return bAnyRegNextStmt; }
	bool anyRegCurStateRef()        { return bAnyRegCurStateRef; }
	bool anyEofActionControl()      { return bAnyEofActionControl; }
	bool anyEofActionCharRef()      { return bAnyEofActionCharRef; }
	bool anyEofActionHold()         { return bAnyEofActionHold; }

	/* Set up labelNeeded flag for each state. Differs for each goto style so
	 * is virtual. */
	virtual void setLabelsNeeded() {}

	/* Determine if we should use indicies. */
	virtual void calcIndexSize() {}

	void findFinalActionRefs();
	void assignActionIds();
	void setValueLimits();

	/* Are there any regular transition functions, any out transition functions. */
	char *fsmName;
	ParseData *parseData;
	RedFsmAp *redFsm;

protected:
	std::ostream &out;

	bool bAnyEofActions;
	bool bAnyActionCalls;
	bool bAnyActionRets;
	bool bAnyRegActionRets;
	bool bAnyRegActionByValControl;
	bool bAnyRegNextStmt;
	bool bAnyRegCurStateRef;
	bool bAnyEofActionControl;
	bool bAnyEofActionCharRef;
	bool bAnyEofActionHold;

	int maxState;
	int maxSingleLen;
	int maxRangeLen;
	int maxKeyOffset;
	int maxIndexOffset;
	int maxIndex;
	int maxActListId;
	int maxActionLoc;
	int maxActArrItem;
	unsigned long long maxSpan;
	int maxFlatIndexOffset;

	bool useIndicies;
};

class CCodeGen : virtual public FsmCodeGen
{
public:
	virtual void writeOutInterface() { C_INTERFACE(); }
	virtual void writeOutHeader() { C_HEADER(); }
	std::ostream &SELF();
};

class CppCodeGen : virtual public FsmCodeGen
{
public:
	virtual void writeOutInterface() { CPP_INTERFACE(); }
	virtual void writeOutHeader() { CPP_HEADER(); }
	std::ostream &SELF();
};

class ObjCCodeGen : virtual public FsmCodeGen
{
public:
	virtual void writeOutInterface() { OBJC_INTERFACE(); }
	virtual void writeOutHeader() { OBJC_HEADER(); }
	std::ostream &SELF();
};

#endif /* _FSMCODEGEN_H */
