/*
 *  Copyright 2001-2004 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "gotocodegen.h"
#include "redfsm.h"
#include "parsetree.h"
#include "bstmap.h"

/* Init base data. */
GotoCodeGen::GotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	FsmCodeGen(fsmName, parseData, redFsm, out)
{ }

/* Emit the goto to take for a given transition. */
std::ostream &GotoCodeGen::TRANS_GOTO( RedTransAp *trans, int level )
{
	TABS(level) << "goto tr" << trans->id << ";";
	return out;
}

std::ostream &GotoCodeGen::FIRST_FINAL_STATE()
{
	if ( redFsm->firstFinState != 0 )
		out << redFsm->firstFinState->id;
	else
		out << redFsm->nextStateId;
	return out;
}

std::ostream &GotoCodeGen::ERROR_STATE()
{
	out << redFsm->errState->id;
	return out;
}

std::ostream &GotoCodeGen::ACTION_SWITCH()
{
	/* Walk the list of functions, printing the cases. */
	for ( ActionList::Iter act = parseData->actionList; act.lte(); act++ ) {
		/* Write out referenced actions. */
		if ( act->numTransRefs > 0 || act->numEofRefs > 0 ) {
			/* Get the function data. Print the case label.  */
			out << "\tcase " << act->actionId << ":\n";

			/* Write the preprocessor line info for going into the source file. */
			out << "#line " << act->loc.line << " \""; LDIR_PATH(inputFile) << "\"\n";

			/* Write the block and close it off. */
			out << "\t{"; INLINE_LIST(act->inlineList, 0, false) << "} break;\n";
		}
	}

	/* Write the directive for going back into the output file. The line
	 * number is for the next line, so add one. */
	out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	return out;
}

std::ostream &GotoCodeGen::GOTO_HEADER( RedStateAp *state )
{
	/* Label the state. */
	out << "case " << state->id << ":\n";
	return out;
}


void GotoCodeGen::emitSingleSwitch( RedStateAp *state )
{
	/* Load up the singles. */
	int numSingles = state->outSingle.length();
	RedTransEl *data = state->outSingle.data;

	if ( numSingles == 1 ) {
		/* If there is a single single key then write it out as an if. */
		out << "\tif ( "; GET_KEY() << " == "; KEY(data[0].lowKey) << " )\n\t\t"; 

		/* Virtual function for writing the target of the transition. */
		TRANS_GOTO(data[0].value, 0) << "\n";
	}
	else if ( numSingles > 1 ) {
		/* Write out single keys in a switch if there is more than one. */
		out << "\tswitch( "; GET_KEY() << " ) {\n";

		/* Write out the single indicies. */
		for ( int j = 0; j < numSingles; j++ ) {
			out << "\t\tcase "; KEY(data[j].lowKey) << ": ";
			TRANS_GOTO(data[j].value, 0) << "\n";
		}
		/* Close off the transition switch. */
		out << "\t}\n";
	}
}

void GotoCodeGen::emitRangeBSearch( RedStateAp *state, int level, int low, int high )
{
	/* Get the mid position, staying on the lower end of the range. */
	int mid = (low + high) >> 1;
	RedTransEl *data = state->outRange.data;

	/* Determine if we need to look higher or lower. */
	bool anyLower = mid > low;
	bool anyHigher = mid < high;

	/* Determine if the keys at mid are the limits of the alphabet. */
	bool limitLow = parseData->keyOps.eq( data[mid].lowKey, parseData->lowKey );
	bool limitHigh = parseData->keyOps.eq( data[mid].highKey, parseData->highKey );

	if ( anyLower && anyHigher ) {
		/* Can go lower and higher than mid. */
		TABS(level) << "if ( "; GET_KEY() << " < "; KEY(data[mid].lowKey) << " ) {\n";
		emitRangeBSearch( state, level+1, low, mid-1 );
		TABS(level) << "} else if ( "; GET_KEY() << " > "; KEY(data[mid].highKey) << " ) {\n";
		emitRangeBSearch( state, level+1, mid+1, high );
		TABS(level) << "} else\n";
		TRANS_GOTO(data[mid].value, level+1) << "\n";
	}
	else if ( anyLower && !anyHigher ) {
		/* Can go lower than mid but not higher. */
		TABS(level) << "if ( "; GET_KEY() << " < "; KEY(data[mid].lowKey) << " ) {\n";
		emitRangeBSearch( state, level+1, low, mid-1 );

		/* if the higher is the highest in the alphabet then there is no
		 * sense testing it. */
		if ( limitHigh ) {
			TABS(level) << "} else\n";
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
		else {
			TABS(level) << "} else if ( "; GET_KEY() << " <= "; KEY(data[mid].highKey) << " )\n";
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
	}
	else if ( !anyLower && anyHigher ) {
		/* Can go higher than mid but not lower. */
		TABS(level) << "if ( "; GET_KEY() << " > "; KEY(data[mid].highKey) << " ) {\n";
		emitRangeBSearch( state, level+1, mid+1, high );

		/* If the lower end is the lowest in the alphabet then there is no
		 * sense testing it. */
		if ( limitLow ) {
			TABS(level) << "} else\n";
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
		else {
			TABS(level) << "} else if ( "; GET_KEY() << " >= "; KEY(data[mid].lowKey) << " )\n";
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
	}
	else {
		/* Cannot go higher or lower than mid. It's mid or bust. What
		 * tests to do depends on limits of alphabet. */
		if ( !limitLow && !limitHigh ) {
			TABS(level) << "if ( "; KEY(data[mid].lowKey) << " <= "; GET_KEY() << 
				" && "; GET_KEY() << " <= "; KEY(data[mid].highKey) << " )\n";
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
		else if ( limitLow && !limitHigh ) {
			TABS(level) << "if ( "; GET_KEY() << " <= "; KEY(data[mid].highKey) << " )\n";
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
		else if ( !limitLow && limitHigh ) {
			TABS(level) << "if ( "; KEY(data[mid].lowKey) << " <= "; GET_KEY() << " )\n";
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
		else {
			/* Both high and low are at the limit. No tests to do. */
			TRANS_GOTO(data[mid].value, level+1) << "\n";
		}
	}
}

std::ostream &GotoCodeGen::STATE_GOTOS()
{
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Writing code above state gotos. */
		GOTO_HEADER( st );

		/* Try singles. */
		if ( st->outSingle.length() > 0 )
			emitSingleSwitch( st );

		/* Default case is to binary search for the ranges, if that fails then */
		if ( st->outRange.length() > 0 )
			emitRangeBSearch( st, 1, 0, st->outRange.length() - 1 );

		/* Write the default transition. */
		TRANS_GOTO( st->defTrans, 1 ) << "\n";
	}
	return out;
}

std::ostream &GotoCodeGen::TRANSITIONS()
{
	/* Emit any transitions that have functions and that go to 
	 * this state. */
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ ) {
		/* Write the label for the transition so it can be jumped to. */
		out << "\ttr" << trans->id << ": ";

		/* Destination state. */
		if ( trans->action != 0 && trans->action->anyCurStateRef() )
			out << "_ps = _cs;";
		out << "_cs = " << trans->targ->id << "; ";

		if ( trans->action != 0 ) {
			/* Write out the transition func. */
			out << "goto f" << trans->action->actListId << ";\n";
		}
		else {
			/* No code to execute, just loop around. */
			out << "goto again;\n";
		}
	}
	return out;
}

/* Set up labelNeeded flag for each state. */
void GotoCodeGen::setLabelsNeeded()
{
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ )
		st->labelNeeded = false;
}

std::ostream &GotoCodeGen::EXEC_FUNCS()
{
	/* Make labels that set acts and jump to execFuncs. Loop func indicies. */
	for ( ActionTableMap::Iter act = redFsm->actionMap; act.lte(); act++ ) {
		out << "\tf" << act->actListId << ": _acts = "; FSM_NAME() << "_a+" << 
				act->location+1 << "; goto execFuncs;\n";
	}

	out <<
		"\n"
		"execFuncs:\n"
		"	_nacts = *_acts++;\n"
		"	while ( _nacts-- > 0 ) {\n"
		"		switch ( *_acts++ ) {\n";
		ACTION_SWITCH() <<
		"		}\n"
		"	}\n"
		"	goto again;\n";
	return out;
}

std::ostream &GotoCodeGen::FINISH_CASES()
{
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* States that are final and have an out action need a case. */
		if ( st->eofAction != 0 ) {
			/* Write the case label. */
			out << "\t\tcase " << st->id << ": ";

			/* Write the goto func. */
			out << "goto f" << st->eofAction->actListId << ";\n";
		}
	}
	
	return out;
}

std::ostream &GotoCodeGen::GOTO( NameInst *name, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{_cs = " << targ->value->id << "; goto again;}";
	return out;
}

std::ostream &GotoCodeGen::GOTOE( InlineItem *ilItem, bool inFinish )
{
	out << "{_cs = (";
	INLINE_LIST( ilItem->children, 0, inFinish );
	out << "); goto again;}";
	return out;
}


std::ostream &GotoCodeGen::STACK( int size )
{
	out << "int _top, _st[" << size << "];";
	return out;
}

std::ostream &GotoCodeGen::CURS( bool inFinish )
{
	out << "(_ps)";
	return out;
}

std::ostream &GotoCodeGen::TARGS( bool inFinish, int targState )
{
	out << "(_cs)";
	return out;
}

std::ostream &GotoCodeGen::GET_ENTRY( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << targ->value->id;
	return out;
}

std::ostream &GotoCodeGen::NEXT( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "_cs = " << targ->value->id << ";";
	return out;
}

std::ostream &GotoCodeGen::NEXTE( InlineItem *ilItem, bool inFinish )
{
	out << "_cs = (";
	INLINE_LIST( ilItem->children, 0, inFinish );
	out << ");";
	return out;
}

std::ostream &GotoCodeGen::CONTEXT( int contextId )
{
	/* Make a vect to fill with context indexed by state. */
	int numVals = (redFsm->nextStateId>>3)+1;
	unsigned char *vals = new unsigned char[numVals];
	memset( vals, 0, sizeof(unsigned char)*numVals );

	/* Fill the context vect. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		if ( st->contextSet.find( contextId ) )
			vals[st->id>>3] |= 1 << (st->id&0x7);
	}

	/* Write out the vect. */
	for ( int i = 0; i < numVals; ) {
		out << (unsigned int)vals[i];

		i += 1;
		if ( i < numVals ) {
			out << ", ";
			if ( i % IALL == 0 )
				out << "\n\t";
		}
	}
	out << "\n";

	delete[] vals;
	return out;
}


/* Init base data. */
CGotoCodeGen::CGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out ) 
: 
	GotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &CGotoCodeGen::RET( bool inFinish )
{
	out << "{_cs = fsm->_st[--fsm->_top]; goto again;}";
	return out;
}

std::ostream &CGotoCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{fsm->_st[fsm->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

void CGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActArrItem) << " "; FSM_NAME() << "_a[] = {\n";
			ACTIONS_ARRAY() <<
			"};\n"
			"\n";
	}

	out <<
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->_top = 0;\n";

	INIT_CODE() <<
		"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n"
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() << 
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = fsm->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps = 0";
	out << ";\n";

	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out << 
		"	fsm->curs = -1;\n"
		"\n";

	out << 
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out << 
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	out <<
		"	switch ( _cs ) {\n";
		STATE_GOTOS() <<
		"	}\n"
		"\n";
		TRANSITIONS() <<
		"\n";

	if ( anyActions() )
		EXEC_FUNCS() << "\n";

	if ( anyEofActions() ) {
		out << 
			"eofActions:\n"
			"	_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"	switch( _cs ) {\n";
			FINISH_CASES() << 
			"	}\n"
			"\n";
	}
	
	out <<
		"out:\n"
		"	fsm->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out << 
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return "; FSM_NAME() << "_execute( fsm, 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( fsm->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( fsm->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		/* Write out the vect. */
		out << "unsigned char _"; FSM_NAME() << "_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base data. */
CppGotoCodeGen::CppGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
: 
	GotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CppGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{this->_st[this->_top++] = _cs; _cs = " << 
			targ->value->id << "; goto again;}";
	return out;
}

std::ostream &CppGotoCodeGen::RET( bool inFinish )
{
	out << "{_cs = this->_st[--this->_top]; goto again;}";
	return out;
}

std::ostream &CppGotoCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{this->_st[this->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

void CppGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActArrItem) << " "; FSM_NAME() << "_a[] = {\n";
			ACTIONS_ARRAY() <<
			"};\n"
			"\n";
	}

	out <<
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->_top = 0;\n";

	INIT_CODE() <<
		"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";
	
	out <<
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = this->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps = 0";
	out << ";\n";

	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out << 
		"	this->curs = -1;\n"
		"\n";

	out << 
		"again:\n"
		"	if ( ++_p == _pe )\n"
		"		goto out;\n";

	if ( anyEofActions() ) {
		out <<
			"	if ( _p == 0 )\n"
			"		goto eofActions;\n"
			"\n";
	}

	out <<
		"	switch ( _cs ) {\n";
		STATE_GOTOS() <<
		"	}\n"
		"\n";
		TRANSITIONS() <<
		"\n";

	if ( anyActions() )
		EXEC_FUNCS() << "\n";

	if ( anyEofActions() ) {
		out <<
			"eofActions:\n"
			"	_p = 0; _pe = (("; EL_TYPE() << "*)0)+1;\n"
			"	switch( _cs ) {\n";
			FINISH_CASES() << 
			"	}\n"
			"\n";
	}

	out <<
		"out:\n"
		"	this->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";
	
	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";
	
	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out << "	return execute( 0, 1 );\n";
	}
	else {
		if ( redFsm->errState != 0 ) {
			out << 
				"	if ( this->curs == "; ERROR_STATE() << " )\n"
				"		return -1;\n";
		}

		out <<
			"	if ( this->curs >= "; FIRST_FINAL_STATE() << " )\n"
			"		return 1;\n"
			"	return 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		out << "unsigned char "; FSM_NAME() << "::_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
}


/* Init base data. */
ObjCGotoCodeGen::ObjCGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
: 
	GotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &ObjCGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	out << "{self->_st[self->_top++] = _cs; _cs = "
		<< targ->value->id << "; goto again;}";
	return out;
}

std::ostream &ObjCGotoCodeGen::RET( bool inFinish )
{
	out << "{_cs = self->_st[--self->_top]; goto again;}";
	return out;
}

std::ostream &ObjCGotoCodeGen::CALLE( InlineItem *ilItem, int targState, bool inFinish )
{
	out << "{self->_st[self->_top++] = _cs; _cs = (";
	INLINE_LIST( ilItem->children, targState, inFinish );
	out << "); goto again;}";
	return out;
}

void ObjCGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n";

	if ( anyActions() ) {
		out <<
			"static "; ARRAY_TYPE(maxActArrItem) << " "; FSM_NAME() << "_a[] = {\n";
			ACTIONS_ARRAY() <<
			"};\n"
			"\n";
	}

	out <<
		"@implementation "; FSM_NAME() << "\n\n"

		"- (int)initFsm;\n"
		"{\n"
		"\tself->curs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "\tself->_top = 0;\n\n";

	INIT_CODE() <<
		"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";
	
	out <<
		"- (int) executeWithData:("; EL_TYPE() << " *)_data len:(int)_len;\n"
		"{\n"
		"\t"; EL_TYPE() << " *_p = _data - 1;\n"
		"\t"; EL_TYPE() << " *_pe = _data + _len;\n"
		"\tint _cs = self->curs";

	if ( anyRegCurStateRef() )
		out << ", _ps = 0";
	out << ";\n";

	if ( anyActions() ) {
		out << "\t";
		ARRAY_TYPE(maxActArrItem) << " *_acts, _nacts;\n";
	}

	out << 
		"\tself->curs = -1;\n"
		"\n";

	out << 
		"again:\n"
			"\tif ( ++_p == _pe )\n"
				"\t\tgoto out;\n";

	if ( anyEofActions() ) {
		out <<
			"\tif ( _p == 0 )\n"
				"\t\tgoto eofActions;\n"
			"\n";
	}

	out <<
			"\tswitch ( _cs ) {\n";
				STATE_GOTOS() <<
			"\t}\n"
		"\n";
		TRANSITIONS() <<
		"\n";

	if ( anyActions() ) EXEC_FUNCS() << "\n";

	if ( anyEofActions() ) {
		out <<
		"eofActions:\n"
			"\t_p = 0; _pe = ( ("; EL_TYPE() << "*)0 ) + sizeof("; EL_TYPE() << ");\n"
			"\tswitch( _cs ) {\n";
				FINISH_CASES() << 
			"\t}\n"
		"\n";
	}

	out <<
		"out:\n"
			"\tself->curs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"\tif ( _cs == "; ERROR_STATE() << " ) return -1;\n";
	}

	out <<
			"\treturn ( _cs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n"
		"}\n"
		"\n";
	
	out <<
		"- (int)finish;\n"
		"{\n";
	
	if ( anyEofActions() ) {
		/* May need to execute code in the action loop. Must use special
		 * params to execute func. */
		out <<
			"\treturn [self executeWithData:0 len:1];\n";
	}
	else {
		// XXX this can be optimized to take out the double self->curs read in
		// the case where there is an errState
		if ( redFsm->errState != 0 ) {
			out << 
			"\tif ( self->curs == "; ERROR_STATE() << " ) return -1;\n";
		}

		out <<
			"\treturn ( self->curs >= "; FIRST_FINAL_STATE() << " ) ? 1 : 0;\n";
	}

	out <<
		"}\n"
		"\n";

	for ( ContextMap::Iter ctx = parseData->contextMap; ctx.lte(); ctx++ ) {
		/* Write out the vect. */
		out << "unsigned char _"; FSM_NAME() << "_ctxdata_" << ctx->key << "[] = {\n\t";
		CONTEXT( ctx->value->id );
		out << "};\n\n";
	}
	
	out <<
		"@end\n\n";
}

