/*
 *  Copyright 2001-2004 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "ipgotocodegen.h"
#include "redfsm.h"
#include "parsetree.h"
#include "bstmap.h"

/* Inits the code and data pointers. There are no code generation 
 * functions to register. */
IpGotoCodeGen::IpGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	GotoCodeGen(fsmName, parseData, redFsm, out)
{ }

/* Called from GotoCodeGen::STATE_GOTOS just before writing the gotos for each
 * state. */
std::ostream &IpGotoCodeGen::GOTO_HEADER( RedStateAp *state )
{
	bool anyWritten = false;

	/* Emit any transitions that have actions and that go to 
	 * this state. */
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans ++ ) {
		if ( trans->targ == state && trans->action != 0 ) {
			/* Remember that we wrote a trans so we know to write the
			 * line directive for going back to the output. */
			anyWritten = true;

			/* Write the label for the transition so it can be jumped to. */
			out << "tr" << trans->id << ":\n";

			/* If the action contains a next, then we must preload the current
			 * state since the action may or may not set it. */
			if ( trans->action->anyNextStmt() )
				out << "	_cs = " << trans->targ->id << ";\n";

			for ( ActionTable::Iter item = trans->action->key; item.lte(); item++ ) {
				/* Get the function data. */
				Action *action = parseData->actionIndex[item->value];

				/* Write the preprocessor line info for going into the 
				 * source file. */
				out << "#line " << action->loc.line << " \""; LDIR_PATH(inputFile) << "\"\n";

				/* Wrap the block in brakets. */
				out << "\t{"; ACTION(action, trans->targ->id, false) << "}\n";
			}

			/* If the action contains a next then we need to reload, otherwise
			 * jump directly to the target state. */
			if ( trans->action->anyNextStmt() )
				out << "\tgoto again;\n";
			else
				out << "\tgoto st" << trans->targ->id << ";\n";
		}
	}

	if ( anyWritten ) {
		/* Write the directive for going back into the output file. The line
		 * number is for the next line, so add one. */
		out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	}

	if ( state->labelNeeded ) {
		out << 
			"st" << state->id << ":\n"
			"	if ( ++_p == _pe )\n"
			"		goto out" << state->id << ";\n";
	}

	/* Label the state and recored the prev state. */
	out << "case " << state->id << ":\n";
	if ( state->anyRegCurStateRef() )
		out << "	_ps = " << state->id << ";\n";
	return out;
}

/* Emit the goto to take for a given transition. */
std::ostream &IpGotoCodeGen::TRANS_GOTO( RedTransAp *trans, int level )
{
	if ( trans->action != 0 ) {
		/* Go to the transition which will go to the state. */
		TABS(level) << "goto tr" << trans->id << ";";
	}
	else {
		/* Go directly to the target state. */
		TABS(level) << "goto st" << trans->targ->id << ";";
	}
	return out;
}

std::ostream &IpGotoCodeGen::EXIT_STATES()
{
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		if ( st->labelNeeded ) {
			out << "\tout" << st->id << ": _cs = " << st->id;
			out << "; goto out;\n";
		}
	}
	return out;
}

std::ostream &IpGotoCodeGen::FINISH_CASES()
{
	bool anyWritten = false;

	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Cased needed if there are out actions. */
		if ( st->eofAction != 0 ) {
			out << "\tcase " << st->id << ": ";

			/* If there are out functions, write them. */
			/* Remember that we wrote a trans so we know to write the
			 * line directive for going back to the output. */
			anyWritten = true;

			out << "\n";
			for ( ActionTable::Iter item = st->eofAction->key; item.lte(); item++ ) {
				/* Get the action item. */
				Action *action = parseData->actionIndex[item->value];
				
				/* Write the preprocessor line info for going into the 
				 * source file. */
				out << "#line " << action->loc.line << " \""; LDIR_PATH(inputFile) << "\"\n";

				/* Wrap the block in brakets. */
				out << "\t{"; ACTION(action, STATE_ERR_STATE, true) << "}\n";
			}
			out << "\tbreak;\n";
		}
	}

	if ( anyWritten ) {
		/* Write the directive for going back into the output file. The line
		 * number is for the next line, so add one. */
		out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
	}
	return out;
}

/* Set up labelNeeded flag for each state. */
void IpGotoCodeGen::setLabelsNeeded()
{
	/* Default all labels needed to false. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ )
		st->labelNeeded = false;

	/* Walk all transitions and set only those that have targs. */
	for ( TransApSet::Iter trans = redFsm->transSet; trans.lte(); trans++ ) {
		/* If there is no action with a next statement, then the label will be
		 * needed. */
		if ( trans->action == 0 || !trans->action->anyNextStmt() )
			trans->targ->labelNeeded = true;

		/* Need labels for states that have goto or calls in action code
		 * invoked on characters (ie, not from out action code). */
		if ( trans->action != 0 ) {
			/* Loop the actions. */
			for ( ActionTable::Iter act = trans->action->key; act.lte(); act++ ) {
				/* Get the action. */
				Action *action = parseData->actionIndex[act->value];

				/* Loop it's block items. */
				for ( BlockIter item = action->data; item.lte(); item++ ) {
					if ( item.type == CB_GOTO || item.type == CB_CALL ) {
						/* Get the target. */
						NameInst *name = action->nameTargs[item.name];
						RedEntryMapEl *targ = redFsm->entryMap.find( name->id );

						/* Need a label for this target. */
						targ->value->labelNeeded = true;
					}
				}
			}
		}
	}
}

std::ostream &IpGotoCodeGen::CURS( bool inFinish )
{
	out << "(_ps)";
	return out;
}

std::ostream &IpGotoCodeGen::TARGS( bool inFinish, int targState )
{
	out << targState;
	return out;
}

/* Init base data. */
CIpGotoCodeGen::CIpGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	IpGotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CIpGotoCodeGen::GOTO( NameInst *name, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	if ( !inFinish )
		out << "{goto st" << targ->value->id << ";}";
	else {
		out << "{fsm->_cs = " << targ->value->id;
		if ( anyEofActionHold() )
			out << "; goto again;}";
		else 
			out << "; goto out;}";
	}
	return out;
}

std::ostream &CIpGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	if ( !inFinish ) {
		out << "{fsm->_st[fsm->_top++] = " << 
				targState << "; goto st" << targ->value->id << ";}";
	}
	else {
		out << "{fsm->_st[fsm->_top++] = " << 
				targState << "; fsm->_cs = " << targ->value->id;
		if ( anyEofActionHold() )
			out << "; goto again;}";
		else 
			out << "; goto out;}";
	}
	return out;
}

std::ostream &CIpGotoCodeGen::RET( bool inFinish )
{
	if ( !inFinish )
		out << "{_cs = fsm->_st[--fsm->_top]; goto again;}";
	else {
		out << "{fsm->_cs = fsm->_st[--fsm->_top]";
		if ( anyEofActionHold() )
			out << "; goto again;}";
		else 
			out << "; goto out;}";
	}
	return out;
}

std::ostream &CIpGotoCodeGen::GOTOE( char *stateExpr, bool inFinish )
{
	if ( !inFinish ) {
		out << "{_cs = (" << stateExpr << "); goto again;}";
	}
	else {
		out << "{fsm->_cs = (" << stateExpr;
		if ( anyEofActionHold() )
			out << "); goto again;}";
		else 
			out << "); goto out;}";
	}
	return out;
}

std::ostream &CIpGotoCodeGen::CALLE( char *stateExpr, int targState, bool inFinish )
{
	if ( !inFinish ) {
		out << "{fsm->_st[fsm->_top++] = " << 
				targState << "; _cs = (" << 
				stateExpr << "); goto again;}";
	}
	else {
		out << "{fsm->_st[fsm->_top++] = " << 
				targState << "; fsm->_cs = (" << stateExpr;
		if ( anyEofActionHold() )
			out << "); goto again;}";
		else 
			out << "); goto out;}";
	}
	return out;
}

std::ostream &CIpGotoCodeGen::NEXT( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	if ( ! inFinish )
		out << "_cs = " << targ->value->id << ";";
	else
		out << "fsm->_cs = " << targ->value->id << ";";
	return out;
}

std::ostream &CIpGotoCodeGen::NEXTE( char *stateExpr, bool inFinish )
{
	if ( !inFinish )
		out << "_cs = (" << stateExpr << ");";
	else
		out << "fsm->_cs = (" << stateExpr << ");";
	return out;
}

void CIpGotoCodeGen::writeOutHeader()
{
	out <<
		"/* Only non-static data: current state. */\n"
		"struct "; FSM_NAME() << "\n"
		"{\n"
		"	int _cs;\n";
		STRUCT_DATA() <<
		"};\n"
		"\n"
		"/* Init the fsm. */\n"
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm );\n"
		"\n"
		"/* Execute some chunk of data. */\n"
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() << 
				" *data, int len );\n"
		"\n"
		"/* Indicate to the fsm tha there is no more data. */\n"
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm );\n"
		"\n";
}

void CIpGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n"
		"int "; FSM_NAME() << "_init( struct "; FSM_NAME() << " *fsm )\n"
		"{\n"
		"	fsm->_cs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	fsm->_top = 0;\n";

	INIT_CODE() <<
		"	if ( fsm->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n"
		"int "; FSM_NAME() << "_execute( struct "; FSM_NAME() << " *fsm, "; EL_TYPE() << 
				" *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = fsm->_cs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n\n";

	/* Only needed by return statements. */
	if ( anyRegActionRets() || anyRegActionByValControl() || anyRegNextStmt() )
		out << "again:\n";

	out <<
		"	if ( ++_p == _pe )\n"
		"		goto out;\n"
		"	switch ( _cs ) {\n";
		STATE_GOTOS() << 
		"	}\n";
		EXIT_STATES() << 
		"\n";
	
	out <<
		"out:\n"
		"	fsm->_cs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";

	out <<
		"int "; FSM_NAME() << "_finish( struct "; FSM_NAME() << " *fsm )\n"
		"{\n";

	if ( anyEofActionCharRef() ) {
		out << "	"; 
		if ( ! anyEofActionHold() ) {
			EL_TYPE() << " *_p = 0;\n";
		}
		else {
			EL_TYPE() << " *_p = (("; EL_TYPE() << "*)0)-1" << 
					", *_pe = (("; EL_TYPE() << "*)0)+1;\n";
		}
	}

	if ( anyEofActionHold() ) {
		out <<
			"again:\n"
			"	if ( ++_p == _pe )\n"
			"		goto out;\n";
	}

	out <<
		"	switch ( fsm->_cs ) {\n";
		FINISH_CASES() <<
		"	}\n"
		"\n";

	if ( anyEofActionHold() )
		out << "	goto again;\n";
	
	if ( anyEofActionControl() || anyEofActionHold() )
		out << "	out:\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( fsm->_cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( fsm->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";
}

/* Init base data. */
CCIpGotoCodeGen::CCIpGotoCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, std::ostream &out )
:
	IpGotoCodeGen(fsmName, parseData, redFsm, out)
{ }

std::ostream &CCIpGotoCodeGen::GOTO( NameInst *name, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	if ( !inFinish )
		out << "{goto st" << targ->value->id << ";}";
	else {
		out << "{this->_cs = " << targ->value->id;
		if ( anyEofActionHold() )
			out << "; goto again;}";
		else 
			out << "; goto out;}";
	}
	return out;
}

std::ostream &CCIpGotoCodeGen::CALL( NameInst *name, int targState, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	if ( !inFinish ) {
		out << "{this->_st[this->_top++] = " << 
				targState << "; goto st" << targ->value->id << ";}";
	}
	else {
		out << "{this->_st[this->_top++] = " << 
				targState << "; this->_cs = " << targ->value->id;
		if ( anyEofActionHold() )
			out << "; goto again;}";
		else 
			out << "; goto out;}";
	}
	return out;
}

std::ostream &CCIpGotoCodeGen::RET( bool inFinish )
{
	if ( !inFinish )
		out << "{_cs = this->_st[--this->_top]; goto again;}";
	else {
		out << "{this->_cs = this->_st[--this->_top]";
		if ( anyEofActionHold() )
			out << "; goto again;}";
		else 
			out << "; goto out;}";
	}
	return out;
}

std::ostream &CCIpGotoCodeGen::GOTOE( char *stateExpr, bool inFinish )
{
	/* Lookup the target. Always guaranteed to return just one target. */
	if ( !inFinish ) {
		out << "{_cs = (" << stateExpr << "); goto again;}";
	}
	else {
		out << "{this->_cs = (" << stateExpr;
		if ( anyEofActionHold() )
			out << "); goto again;}";
		else 
			out << "); goto out;}";
	}
	return out;
}

std::ostream &CCIpGotoCodeGen::CALLE( char *stateExpr, int targState, bool inFinish )
{
	if ( !inFinish ) {
		out << "{this->_st[this->_top++] = " << 
				targState << "; _cs = (" << 
				stateExpr << "); goto again;}";
	}
	else {
		out << "{this->_st[this->_top++] = " << 
				targState << "; this->_cs = (" << stateExpr;
		if ( anyEofActionHold() )
			out << "); goto again;}";
		else 
			out << "); goto out;}";
	}
	return out;
}

std::ostream &CCIpGotoCodeGen::NEXT( NameInst *name, bool inFinish )
{
	RedEntryMapEl *targ = redFsm->entryMap.find( name->id );
	if ( ! inFinish )
		out << "_cs = " << targ->value->id << ";";
	else
		out << "this->_cs = " << targ->value->id << ";";
	return out;
}

std::ostream &CCIpGotoCodeGen::NEXTE( char *stateExpr, bool inFinish )
{
	if ( !inFinish )
		out << "_cs = (" << stateExpr << ");";
	else
		out << "this->_cs = (" << stateExpr << ");";
	return out;
}

void CCIpGotoCodeGen::writeOutHeader()
{
	out <<
		"/* Only non-static data: current state. */\n"
		"class "; FSM_NAME() << "\n"
		"{\n"
		"public:\n"
		"	int _cs;\n"
		"\n"
		"	/* Init the fsm. */\n"
		"	int init( );\n"
		"\n"
		"	/* Execute some chunk of data. */\n"
		"	int execute( "; EL_TYPE() << " *data, int len );\n"
		"\n"
		"	/* Indicate to the fsm tha there is no more data. */\n"
		"	int finish( );\n";
		STRUCT_DATA() <<
		"};\n"
		"\n";
}

void CCIpGotoCodeGen::writeOutCode()
{
	out <<
		"static int "; FSM_NAME() << "_start = "; START_STATE_ID() << ";\n"
		"\n"
		"int "; FSM_NAME() << "::init( )\n"
		"{\n"
		"	this->_cs = "; FSM_NAME() << "_start;\n";

	/* If there are any calls, then the stack top needs initialization. */
	if ( anyActionCalls() || anyActionRets() )
		out << "	this->_top = 0;\n";

	INIT_CODE() <<
		"	if ( this->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n"
		"int "; FSM_NAME() << "::execute( "; EL_TYPE() << " *_data, int _len )\n"
		"{\n"
		"	"; EL_TYPE() << " *_p = _data-1;\n"
		"	"; EL_TYPE() << " *_pe = _data+_len;\n"
		"	int _cs = this->_cs";

	if ( anyRegCurStateRef() )
		out << ", _ps";
	out << ";\n\n";

	/* Needed by returns, by-value control and next statements. */
	if ( anyRegActionRets() || anyRegActionByValControl() || anyRegNextStmt() )
		out << "again:\n";

	out <<
		"	if ( ++_p == _pe )\n"
		"		goto out;\n"
		"	switch ( _cs ) {\n";
		STATE_GOTOS() <<
		"	}\n";
		EXIT_STATES() << 
		"\n";

	out <<
		"out:\n"
		"	this->_cs = _cs;\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( _cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( _cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";
	
	out <<
		"int "; FSM_NAME() << "::finish( )\n"
		"{\n";
	
	if ( anyEofActionCharRef() ) {
		out << "	"; 
		if ( ! anyEofActionHold() ) {
			EL_TYPE() << " *_p = 0;\n";
		}
		else {
			EL_TYPE() << " *_p = (("; EL_TYPE() << "*)0)-1" << 
					", *_pe = (("; EL_TYPE() << "*)0)+1;\n";
		}
	}

	if ( anyEofActionHold() ) {
		out <<
			"again:\n"
			"	if ( ++_p == _pe )\n"
			"		goto out;\n";
	}

	out <<
		"	switch ( this->_cs ) {\n";
		FINISH_CASES() <<
		"	}\n"
		"\n";

	if ( anyEofActionHold() ) {
		out << "	goto again;\n";
	}
	
	if ( anyEofActionControl() || anyEofActionHold() )
		out << "	out:\n";

	if ( redFsm->errState != 0 ) {
		out <<
			"	if ( this->_cs == "; ERROR_STATE() << " )\n"
			"		return -1;\n";
	}

	out <<
		"	if ( this->_cs >= "; FIRST_FINAL_STATE() << " )\n"
		"		return 1;\n"
		"	return 0;\n"
		"}\n"
		"\n";
}
