/*
 *  Copyright 2001-2004 Adrian Thurston <adriant@ragel.ca>
 */

/*  This file is part of Ragel.
 *
 *  Ragel is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Ragel is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Ragel; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include "ragel.h"
#include "fsmcodegen.h"
#include "parsetree.h"
#include "redfsm.h"

/* Determine if a string is only whitespace. Code blocks that are only
 * whitespace need not be output. */
bool onlyWhitespace( char *str )
{
	while ( *str != 0 ) {
		if ( *str != ' ' && *str != '\t' && *str != '\n' &&
				*str != '\v' && *str != '\f' && *str != '\r' )
			return false;
		str += 1;
	}
	return true;
}

using std::ostream;

/* Init code gen with in parameters. */
FsmCodeGen::FsmCodeGen( char *fsmName, ParseData *parseData, 
		RedFsmAp *redFsm, ostream &out )
:
	fsmName(fsmName), 
	parseData(parseData), 
	redFsm(redFsm),
	out(out),

	bAnyEofActions(false),
	bAnyActionCalls(false),
	bAnyActionRets(false),
	bAnyRegActionRets(false),
	bAnyRegActionByValControl(false),
	bAnyRegNextStmt(false),
	bAnyRegCurStateRef(false),
	bAnyEofActionControl(false),
	bAnyEofActionCharRef(false),
	bAnyEofActionHold(false)
{
}

void FsmCodeGen::startCodeGen()
{
	/* Write the preprocessor line info for going to the output file. */
	out << "#line " << outFilter->line + 1 << " \""; LDIR_PATH(outputFile) << "\"\n";
}

void FsmCodeGen::endCodeGen()
{
	/* Write the preprocessor line info for to the input file. */
	out << "#line " << parseData->fsmEndSecLoc.line  << " \""; LDIR_PATH(inputFile) << "\"\n";
}

/* Does the machine have any functions. */
bool FsmCodeGen::anyActions()
{
	/* If there is more than one action then one must be non-empty. If there
	 * is only one then it may be nonzero. */
	return redFsm->actionMap.length() > 0;
}

void FsmCodeGen::findFinalActionRefs()
{
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Rerence count out of single transitions. */
		for ( RedTransList::Iter rtel = st->outSingle; rtel.lte(); rtel++ ) {
			if ( rtel->value->action != 0 ) {
				for ( ActionTable::Iter item = rtel->value->action->key; item.lte(); item++ ) {
					Action *action = parseData->actionIndex[item->value];
					action->numTransRefs += 1;
				}
			}
		}

		/* Reference count out of range transitions. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			if ( rtel->value->action != 0 ) {
				for ( ActionTable::Iter item = rtel->value->action->key; item.lte(); item++ ) {
					Action *action = parseData->actionIndex[item->value];
					action->numTransRefs += 1;
				}
			}
		}

		/* Reference count default transition. */
		if ( st->defTrans != 0 && st->defTrans->action != 0 ) {
			for ( ActionTable::Iter item = st->defTrans->action->key; 
					item.lte(); item++ ) {
				Action *action = parseData->actionIndex[item->value];
				action->numTransRefs += 1;
			}
		}

		/* Reference count out of eof actions. */
		if ( st->eofAction != 0 ) {
			for ( ActionTable::Iter item = st->eofAction->key; item.lte(); item++ ) {
				Action *action = parseData->actionIndex[item->value];
				action->numEofRefs += 1;
			}
		}
	}
}

/* Gather various info on the machine. */
void FsmCodeGen::analyzeMachine()
{
	/* Find the true count of action references.  */
	findFinalActionRefs();

	/* Check if there are any calls in action code. */
	for ( ActionList::Iter act = parseData->actionList; act.lte(); act++ ) {
		/* Only consider actions that are referenced. */
		if ( act->numTransRefs > 0 || act->numEofRefs > 0 ) {
			for ( BlockIter item = act->data; item.lte(); item++ ) {
				if ( item.type == CB_CALL )
					bAnyActionCalls = true;
				else if ( item.type == CB_RET )
					bAnyActionRets = true;
			}
		}

		/* Check for various things in regular actions. */
		if ( act->numTransRefs > 0 ) {
			for ( BlockIter item = act->data; item.lte(); item++ ) {
				/* Any returns in regular actions? */
				if ( item.type == CB_RET )
					bAnyRegActionRets = true;

				/* Any next statements in the regular actions? */
				if ( item.type == CB_NEXT || item.type == CB_NEXTE )
					bAnyRegNextStmt = true;

				/* Any by value control in regular actions? */
				if ( item.type == CB_CALLE || item.type == CB_GOTOE )
					bAnyRegActionByValControl = true;

				/* Any references to the current state in regular actions? */
				if ( item.type == CB_CURS )
					bAnyRegCurStateRef = true;
			}
		}

		/* Check for various things in eof actions. */
		if ( act->numEofRefs > 0 ) {
			/* Note that there are eof actions. */
			bAnyEofActions = true;

			/* Loop it's block items. */
			for ( BlockIter item = act->data; item.lte(); item++ ) {
				/* Any Control flow? */
				if ( item.type == CB_GOTO || item.type == CB_CALL || 
						item.type == CB_RET || item.type == CB_GOTOE ||
						item.type == CB_CALLE )
					bAnyEofActionControl = true;

				/* Any holds? */
				if ( item.type == CB_HOLD )
					bAnyEofActionHold = true;

				/* References to the current character? */
				if ( item.type == CB_CHAR || item.type == CB_PCHAR || 
						item.type == CB_HOLD )
					bAnyEofActionCharRef = true;
			}
		}
	}

	/* Analyze reduced action lists. */
	for ( ActionTableMap::Iter at = redFsm->actionMap; at.lte(); at++ ) {
		for ( ActionTable::Iter act = at->key; act.lte(); act++ ) {
			Action *action = parseData->actionIndex[act->value];
			for ( BlockIter item = action->data; item.lte(); item++ ) {
				/* Any next statements in the action table? */
				if ( item.type == CB_NEXT || item.type == CB_NEXTE )
					at->bAnyNextStmt = true;

				/* Any references to the current state. */
				if ( item.type == CB_CURS )
					at->bAnyCurStateRef = true;
			}
		}
	}

	/* Find states that have transitions with actions that have next
	 * statements. */
	for ( RedStateList::Iter st = redFsm->stateList; st.lte(); st++ ) {
		/* Check any actions out of outSinge. */
		for ( RedTransList::Iter rtel = st->outSingle; rtel.lte(); rtel++ ) {
			if ( rtel->value->action != 0 && rtel->value->action->anyCurStateRef() )
				st->bAnyRegCurStateRef = true;
		}

		/* Check any actions out of outRange. */
		for ( RedTransList::Iter rtel = st->outRange; rtel.lte(); rtel++ ) {
			if ( rtel->value->action != 0 && rtel->value->action->anyCurStateRef() )
				st->bAnyRegCurStateRef = true;
		}

		/* Check any action out of default. */
		if ( st->defTrans != 0 && st->defTrans->action != 0 && 
				st->defTrans->action->anyCurStateRef() )
			st->bAnyRegCurStateRef = true;
	}

	/* Set up the label needed flag for each state */
	setLabelsNeeded();
}


/* Write out the fsm name. */
std::ostream &FsmCodeGen::FSM_NAME()
{
	out << fsmName;
	return out;
}

std::ostream &FsmCodeGen::USER_STRUCT( char *data )
{
	for ( BlockIter item = data; item.lte(); item++ ) {
		switch ( item.type ) {
		case CB_STACK:
			STACK(atoi(item.data));
			break;
		case CB_TEXT:
			out << item.data;
			break;
		}
	}
	/* There may not be a trailing newline. Add one. */
	out << "\n";
	return out;
}

std::ostream &FsmCodeGen::STRUCT_DATA()
{
	bool anyBlocksWritten = false;

	/* Walk the list of data, printing the cases. */
	InlineBlock *ilBlock = parseData->dataList.head;
	while ( ilBlock != NULL ) {
		/* Don't bother with blocks that are only whitespace. */
		if ( ! onlyWhitespace( ilBlock->data ) ) {
			/* Remember that we wrote a blcok so that we know to write a line
			 * directive back to the output file. */
			anyBlocksWritten = true;

			/* Write the preprocessor line info for going into the 
			 * source file. */
			out << "#line " << ilBlock->loc.line << " \"";
			LDIR_PATH(inputFile) << "\"\n";

			/* Write the block. */
			USER_STRUCT( ilBlock->data );
		}
		ilBlock = ilBlock->next;
	}

	/* If any blocks were written, then write the directive for going back
	 * into the output file. The line number is for the next line, so add one. */
	if ( anyBlocksWritten ) {
		out << "#line " << outFilter->line + 1 << " \"";
		LDIR_PATH(outputFile) << "\"\n";
	}

	return out;
}

std::ostream &FsmCodeGen::USER_INIT( InlineBlock *ilBlock )
{
	out << "{";
	for ( BlockIter item = ilBlock->data; item.lte(); item++ ) {
		switch ( item.type ) {
		case CB_STATE:
			GET_STATE( ilBlock->nameTargs[item.name], false );
			break;
		case CB_TEXT:
			out << item.data;
			break;
		}
	}
	out << "}\n";
	return out;
}

std::ostream &FsmCodeGen::INIT_CODE()
{
	bool anyBlocksWritten = false;

	/* Walk the list of pre funcs, printing the sections. */
	InlineBlock *ilBlock = parseData->initCodeList.head;
	while ( ilBlock != NULL ) {
		/* Don't bother with blocks that are only whitespace. */
		if ( ! onlyWhitespace( ilBlock->data ) ) {
			/* Remember that we wrote a blcok so that we know to write a line
			 * directive back to the output file. */
			anyBlocksWritten = true;

			/* Write the preprocessor line info for going into the 
			 * source file. */
			out << "#line " << ilBlock->loc.line << " \"";
			LDIR_PATH(inputFile) << "\"\n";

			/* Write the block. */
			USER_INIT( ilBlock );
		}
		ilBlock = ilBlock->next;
	}

	/* If any blocks were written, then write the directive for going back
	 * into the output file. The line number is for the next line, so add one. */
	if ( anyBlocksWritten ) {
		out << "#line " << outFilter->line + 1 << " \"";
		LDIR_PATH(outputFile) << "\"\n";
	}

	return out;
}

/* Emit the offset of the start state as a decimal integer. */
std::ostream &FsmCodeGen::START_STATE_ID()
{
	out << redFsm->startState->id;
	return out;
};

/* Write out the array of actions. */
std::ostream &FsmCodeGen::ACTIONS_ARRAY()
{
	out << "\t0, ";
	int totalActions = 1;
	for ( ActionTableMap::Iter act = redFsm->actionMap; act.lte(); act++ ) {
		/* Write out the length, which will never be the last character. */
		out << act->key.length() << ", ";
		/* Put in a line break every 8 */
		if ( totalActions++ % 8 == 7 )
			out << "\n\t";

		for ( ActionTable::Iter item = act->key; item.lte(); item++ ) {
			out << item->value;
			if ( ! (act.last() && item.last()) )
				out << ", ";

			/* Put in a line break every 8 */
			if ( totalActions++ % 8 == 7 )
				out << "\n\t";
		}
	}
	out << "\n";
	return out;
}


/* Emit the alphabet data type. */
std::ostream &FsmCodeGen::ALPH_TYPE()
{
	switch ( parseData->alphType ) {
	case AT_Char:
		out << "char";
		break;
	case AT_UnsignedChar:
		out << "unsigned char";
		break;
	case AT_Short:
		out << "short";
		break;
	case AT_UnsignedShort:
		out << "unsigned short";
		break;
	case AT_Int:
		out << "int";
		break;
	case AT_UnsignedInt:
		out << "unsigned int";
		break;
	}

	return out;
}

std::ostream &FsmCodeGen::EL_TYPE()
{
	if ( parseData->elementType != 0 ) {
		/* Emit the element type. */
		out << parseData->elementType;
	}
	else {
		/* No element type specified, just using alph type. */
		ALPH_TYPE();
	}
	return out;
}

std::ostream &FsmCodeGen::GET_KEY()
{
	if ( parseData->getKeyExpr != 0 ) { 
		/* Emit the user supplied method of retrieving the key. */
		out << "(";

		for ( BlockIter item = parseData->getKeyExpr; item.lte(); item++ ) {
			switch ( item.type ) {
			case CB_CHAR:
				out << "(*_p)";
				break;
			case CB_PCHAR:
				out << "_p";
				break;
			case CB_TEXT:
				out << item.data;
				break;
			default:
				/* FIXME: This should be validated. */
				assert(false);
				break;
			}
		}

		out << ")";
	}
	else {
		/* Expression for retrieving the key, use simple dereference. */
		out << "(*_p)";
	}
	return out;
}

/* Write out level number of tabs. Makes the nested binary search nice
 * looking. */
std::ostream &FsmCodeGen::TABS( int level )
{
	while ( level-- > 0 )
		out << "\t";
	return out;
}

/* Write out a key from the fsm code gen. Depends on wether or not the key is
 * signed. */
std::ostream &FsmCodeGen::KEY( long key )
{
	if ( parseData->isAlphSigned() )
		out << key;
	else
		out << (unsigned long) key << 'u';
	return out;
}

/* Write out action code which is segmented into statements interpreted by
 * ragel for rewriting and text that is passed through as is. */
std::ostream &FsmCodeGen::ACTION( Action *action, int targState, bool inFinish )
{
	for ( BlockIter item = action->data; item.lte(); item++ ) {
		switch ( item.type ) {
		case CB_HOLD:
			out << "_p--;";
			break;
		case CB_CHAR:
			out << "(*_p)";
			break;
		case CB_PCHAR:
			out << "_p";
			break;
		case CB_TEXT:
			out << item.data;
			break;
		case CB_GOTO:
			GOTO( action->nameTargs[item.name], inFinish );
			break;
		case CB_CALL:
			CALL( action->nameTargs[item.name], targState, inFinish );
			break;
		case CB_RET:
			RET( inFinish );
			break;
		case CB_CURS:
			CURS( inFinish );
			break;
		case CB_TARGS:
			TARGS( inFinish, targState );
			break;
		case CB_STATE:
			GET_STATE( action->nameTargs[item.name], inFinish );
			break;
		case CB_GOTOE:
			GOTOE( item.data, inFinish );
			break;
		case CB_CALLE:
			CALLE( item.data, targState, inFinish );
			break;
		case CB_NEXT:
			NEXT( action->nameTargs[item.name], inFinish );
			break;
		case CB_NEXTE:
			NEXTE( item.data, inFinish );
			break;
		}
	}
	return out;
}

/* Write out paths in line directives. Escapes any special characters. */
std::ostream &FsmCodeGen::LDIR_PATH( char *path )
{
	for ( char *pc = path; *pc != 0; pc++ ) {
		if ( *pc == '\\' )
			out << "\\\\";
		else
			out << *pc;
	}
	return out;
}
