% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cubintegrate.R
\name{cubintegrate}
\alias{cubintegrate}
\title{Unified Cubature Integration Interface}
\usage{
cubintegrate(
  f,
  lower,
  upper,
  fDim = 1,
  method = c("hcubature", "pcubature", "cuhre", "divonne", "suave", "vegas"),
  relTol = 1e-05,
  absTol = 1e-12,
  maxEval = 10^6,
  nVec = 1L,
  ...
)
}
\arguments{
\item{f}{The function (integrand) to be integrated. Can be
vectorized version, but the additional arguments \code{...}
must indicate via either \code{vectorInterface = TRUE} for
\code{hcubature} and \code{pcubature}, or a value for \code{nVec}. See
details on each method.}

\item{lower}{The lower limit of integration, a vector for
hypercubes.}

\item{upper}{The upper limit of integration, a vector for
hypercubes.}

\item{fDim}{The number of components of f, default 1, bears no
relation to the dimension of the hypercube over which
integration is performed.}

\item{method}{the method to use should be one of "hcubature",
"pcubature", "cuhre", "divonne", "suave" or "vegas".}

\item{relTol}{The maximum tolerance, default 1e-5.}

\item{absTol}{the absolute tolerance, default 1e-12.}

\item{maxEval}{The maximum number of function evaluations needed,
default 10^6.  Note that the actual number of function
evaluations performed is only approximately guaranteed not to
exceed this number.}

\item{nVec}{the number of vectorization points for Cuba C library,
default 1, but can be set to an integer > 1 for vectorization,
for example, 1024. The function f above needs to handle the
vector of points appropriately; see vignette examples. Unlike
Cuba, the cubature C library manages the number of points on
its own and can vary between calls. Therefore, any value for
nVec greater than one implies vectorization for a cubature
method.}

\item{...}{All other arguments which may include integration
method specific parameters and those for f. Unrecognized
parameters for integration method are presumed to be intended
for f and so processed.}
}
\value{
The returned value is a list of items:
\describe{
\item{integral}{the value of the integral}
\item{error}{the estimated absolute error}
\item{neval}{the number of times the function was evaluated}
\item{returnCode}{the actual integer return code of the C routine; a non-zero value usually indicates problems; further interpretation depends on method}
\item{nregions}{forcCuba routines, the actual number of subregions needed}
\item{prob}{the \eqn{\chi^2}{Chi2}-probability (not the \eqn{\chi^2}{Chi2}-value itself!) that \code{error} is not a reliable estimate of the true integration error.}
}
}
\description{
Integrate a function within specified limits using method
specified. Further arguments specific to method as well as other
arguments to f may be passed. For defaults used in each method, see
help on the method or \code{\link[=default_args]{default_args()}}.
}
\examples{
I.1d <- function(x) {
  sin(4*x) *
    x * ((x * ( x * (x*x-4) + 1) - 1))
}
I.1d_v <- function(x) {
   matrix(apply(x, 2, function(z)
       sin(4 * z) *
       z * ((z * ( z * (z * z - 4) + 1) - 1))),
       ncol = ncol(x))
}
cubintegrate(f = I.1d, lower = -2, upper = 2, method = "pcubature")
cubintegrate(f = I.1d, lower = -2, upper = 2, method = "cuhre", flags=list(verbose = 2))
cubintegrate(f = I.1d_v, lower = -2, upper = 2, method = "hcubature", nVec = 2L)
cubintegrate(f = I.1d_v, lower = -2, upper = 2, method = "cuhre", nVec = 128L)

}
\seealso{
\code{\link[=default_args]{default_args()}}, \code{\link[=hcubature]{hcubature()}},
\code{\link[=pcubature]{pcubature()}}, \code{\link[=cuhre]{cuhre()}},
\code{\link[=vegas]{vegas()}}, \code{\link[=suave]{suave()}}, \code{\link[=divonne]{divonne()}}
}
