% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data.R
\docType{data}
\name{miniACC}
\alias{miniACC}
\title{Adrenocortical Carcinoma (ACC) MultiAssayExperiment}
\format{
A \code{MultiAssayExperiment} with 5 experiments, providing:
\describe{
    \item{RNASeq2GeneNorm}{RNA-seq count data: an \code{ExpressionSet}
    with 198 rows and 79 columns}
    \item{gistict}{Reccurent copy number lesions identified by GISTIC2:
    a \code{SummarizedExperiment} with 198 rows and 90 columns}
    \item{RPPAArray}{Reverse Phase Protein Array: an \code{ExpressionSet}
    with 33 rows and 46 columns. Rows are indexed by genes,
    but protein annotations are available from
    \code{featureData(miniACC[["RPPAArray"]])}. The source of these
    annotations is noted in \code{abstract(miniACC[["RPPAArray"]])}}
    \item{Mutations}{Somatic mutations: a \code{matrix} with 223 rows and
    90 columns. 1 for any kind of non-silent mutation, zero for silent
    (synonymous) or no mutation.}
    \item{miRNASeqGene}{microRNA sequencing: an \code{ExpressionSet} with
    471 rows and 80 columns. Rows not having at least 5 counts in at least
    5 samples were removed.}
}
}
\source{
https://github.com/waldronlab/multiassayexperiment-tcga
}
\usage{
miniACC
}
\description{
A \code{\link{MultiAssayExperiment}} object providing a reduced version of
the TCGA ACC dataset for all 92 patients. RNA-seq, copy number, and somatic
mutations are included only for genes whose proteins are included in the
reverse-phase protein array. The MicroRNA-seq dataset is also included,
with infrequently expressed microRNA removed. Clinical, pathological, and
subtype information are provided by \code{colData(miniACC)}, and some
additional details are provided by metadata(miniACC).
}
\examples{
miniACC
metadata(miniACC)
colnames(colData(miniACC))
table(miniACC$vital_status)
longFormat(miniACC["MAPK3", , ], colDataCols = c("vital_status", "days_to_death"))
wideFormat(miniACC["MAPK3", , ], colDataCols = c("vital_status", "days_to_death"))

##
## The following is the code used to create this mini dataset from the full ACC dataset.
## The full ACC MultiAssayExperiment was created by the pipeline at
## https://github.com/waldronlab/multiassayexperiment-tcga.
\dontrun{
    ## See www.tinyurl.com/MAEOurls for more pre-built TCGA MultiAssayExperiment objects
    download.file("http://s3.amazonaws.com/multiassayexperiments/accMAEO.rds",
                  destfile = "accMAEO.rds")
    library(MultiAssayExperiment)
    library(RaggedExperiment) #needed for RaggedExperiment objects by updateObject()
    library(Biobase)

    acc <- readRDS("accMAEO.rds")
    acc <- updateObject(acc)
    protmap <- read.csv(paste0("http://genomeportal.stanford.edu/",
                        "pan-tcga/target_selection_send_data",
                        "?filename=Allprotein.txt"), as.is = TRUE
        )

    RPPAgenes <- Filter(function(x) x != "", protmap$Genes)
    RPPAgenes <- unlist(strsplit(RPPAgenes, ","))
    RPPAgenes <- unique(RPPAgenes)

    miniACC <-
        acc[RPPAgenes, , c("RNASeq2GeneNorm", "gistict", "RPPAArray", "Mutations")]
    mut <- assay(miniACC[["Mutations"]], i = "Variant_Classification")
    mut <- ifelse(is.na(mut) | mut == "Silent", 0, 1)

    miniACC[["Mutations"]] <- mut
    colData(miniACC) <- colData(miniACC)[, c(1:17, 810:822)]

    rpparowData <-
        protmap[match(rownames(miniACC[["RPPAArray"]]), protmap$Genes),]
    rpparowData <- AnnotatedDataFrame(rpparowData)
    featureData(miniACC[["RPPAArray"]]) <- rpparowData

    md <-
        list(
            title = "Comprehensive Pan-Genomic Characterization of Adrenocortical Carcinoma",
            PMID = "27165744",
            sourceURL = "http://s3.amazonaws.com/multiassayexperiments/accMAEO.rds",
            RPPAfeatureDataURL = paste0("http://genomeportal.stanford.edu/",
                                        "pan-tcga/show_target_selection_file",
                                        "?filename=Allprotein.txt"),
            colDataExtrasURL = "http://www.cell.com/cms/attachment/2062093088/2063584534/mmc3.xlsx"
        )
    metadata(miniACC) <- md

    mirna <- acc[["miRNASeqGene"]]
    mirna <- mirna[rowSums(assay(mirna) >= 5) >= 5, ]
    experimentData(mirna)@abstract <-
        "Note: Rows not having at least 5 counts in at least 5 samples were removed."
    miniACC <- c(miniACC,
                 list(miRNASeqGene = mirna),
                 sampleMap = sampleMap(acc)[sampleMap(acc)$assay == "miRNASeqGene",])

     miniACC[["RNASeq2GeneNorm"]] <-
         as(miniACC[["RNASeq2GeneNorm"]], "SummarizedExperiment")
     miniACC[["RPPAArray"]] <-
         as(miniACC[["RPPAArray"]], "SummarizedExperiment")
     miniACC[["miRNASeqGene"]] <-
         as(miniACC[["miRNASeqGene"]], "SummarizedExperiment")

    save(miniACC, file = "data/miniACC.RData", compress = "bzip2")
}
}
\references{
Zheng S *et al.*: Comprehensive Pan-Genomic Characterization of
Adrenocortical Carcinoma. Cancer Cell 2016, 29:723-736.
}
\author{
Levi Waldron \email{lwaldron.research@gmail.com}
}
\keyword{data}
