// jithash.c: hash table for jit.c
//
// This supports the standard put and get but also
// fast retrieval of all stored elements.
// It is used to keep track of SEXPs that have the JITTED_BIT or
// related bits set so we can clear the bits at the end of JIT section.
//
// R : A Computer Language for Statistical Data Analysis
// Copyright (C) 1995, 1996    Robert Gentleman and Ross Ihaka
// Copyright (C) 1998--2006    The R Development Core Team.
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, a copy is available at
// http://www.r-project.org/Licenses/

#ifdef HAVE_CONFIG_H
  #include "config.h"
#endif
#include "Defn.h"
#include "Print.h"
#define JIT_INTERNAL 1          // need Dassert and NBR_ELEMS
#include "jit.h"
#include "jithash.h"
#include "printsxp.h"

static SEXP tab[HASH_LEN];      // the hash table
static int entries[HASH_LEN];   // tab indices, for fast retrieval
static int nentries;            // number of used elements in "entries"
static int ientries;            // index into "entries"

#if DEBUG_JIT
static int nputs;
static int ncollisions;
#endif

#if DEBUG_JIT
static void printTab(void)
{
    if (jitTrace >= 5) {
        printf("Hash table: nentries %d nputs %d collisions %g%%\n",
               nentries, nputs,
               (nputs? 100 * (double)ncollisions / nputs : 0.0));

        if (jitTrace >= 6) {
            int i;
            for (i = 0; i < nentries; i++) {
                printf("  [%d]\t", entries[i]);
                printBinding(tab[entries[i]]);
                printf("\n");
            }
        }
    }
}
#endif

void jitInitHash(void)          // clears the hash table
{
    if (nentries) {
        memset(tab, 0, sizeof tab);
        nentries = 0;
#if DEBUG_JIT
        nputs = ncollisions = 0;
#endif
        assert(sizeof(SEXP) == 4);      // see hash() function below
    }
}

// This hash func is optimized for SEXPs.
// It assumes pointers are 4 bytes.
// If not, the assert in jitInitHash will fail.

static R_INLINE int hash(SEXP p)
{
    return ((unsigned)p >> 2) % NBR_ELEMS(tab);
}

void jitPutHash(SEXP p)         // put p into the table
{
    int i = hash(p);
#if DEBUG_JIT
    nputs++;
#endif
    if (tab[i] == p)                    // found it
        return;
    if (tab[i] == NULL)                 // new entry
        tab[i] = p;
    else {                              // collision
        int istart = i;
        do {
#if DEBUG_JIT
            ncollisions++;
#endif
            if (++i == NBR_ELEMS(tab))
                i = 0;
            if (i == istart)            // hash tab full?
                error("too many symbols in JIT block");
            if (tab[i] == p)            // found it
                return;
        } while (tab[i]);
        tab[i] = p;                     // new entry
    }
    entries[nentries++] = i;
    Dassert(nentries <= NBR_ELEMS(entries));
}

Rboolean jitInHash(SEXP p)      // TRUE if p is in the hash table
{
    int i = hash(p);
    while (tab[i] != p && tab[i])
        if (++i == NBR_ELEMS(tab))
            i = 0;
    return tab[i] != NULL;
}

Rboolean jitInitHashNext(void)  // call before using jitHashNext
{
#if DEBUG_JIT
    printTab();
#endif
    ientries = 0;
    return nentries != 0;
}

// Each call returns the next hash entry, NULL when all done.
// Remember to call jitInitHashNext first.

SEXP jitHashNext(void)
{
    if (ientries >= nentries)
       return NULL;

    return tab[entries[ientries++]];
}
