/*  eval.c
 *
 *  R : A Computer Language for Statistical Data Analysis
 *  Copyright (C) 1995, 1996    Robert Gentleman and Ross Ihaka
 *  Copyright (C) 1998--2006    The R Development Core Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, a copy is available at
 *  http://www.r-project.org/Licenses/
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include "Defn.h"
#include "Rinterface.h"
#include "Fileio.h"
#define JIT_INTERNAL 1
#include "jit.h"
#include "printsxp.h"

/* NEEDED: A fixup is needed in browser, because it can trap errors,
 *         and currently does not reset the limit to the right value.
 */

void attribute_hidden check_stack_balance(SEXP op, int save)
{
    if (save == R_PPStackTop)
        return;

    REprintf("Warning: stack imbalance in '%s', %d then %d\n",
             PRIMNAME(op), save, R_PPStackTop);
}

/* inlined version of check_stack_balance, for speed */

static void R_INLINE check_stack_balance1(SEXP op, int save)
{
    if (save == R_PPStackTop)
        return;
    check_stack_balance(op, save);
}

static R_INLINE void forcePromise(SEXP e)   /* call only from evalPromise */
{
    RPRSTACK prstack;
    SEXP val;
    if(PRSEEN(e)) {
        if (PRSEEN(e) == 1)
            errorcall(R_GlobalContext->call,
                      _("promise already under evaluation: recursive "
                        "default argument reference or earlier problems?"));
        else warningcall(R_GlobalContext->call,
                         _("restarting interrupted promise evaluation"));
    }
    assert(PRVALUE(e) == R_UnboundValue);

    /* Mark the promise as under evaluation and push it on a stack
       that can be used to unmark pending promises if a jump out
       of the evaluation occurs. */

    SET_PRSEEN(e, 1);
    prstack.promise = e;
    prstack.next = R_PendingPromises;
    R_PendingPromises = &prstack;
    jitSuspend("forcePromise"); /* prevent jit gen while evaluating promise */
    val = eval(PRCODE(e), PRENV(e));
    jitUnsuspend();             /* reallow jit generation */

    /* Pop the stack, unmark the promise and set its value field.
       Also set the environment to RNIL to allow GC to
       reclaim the promise environment; this is also useful for
       fancy games with delayedAssign() */

    R_PendingPromises = prstack.next;
    SET_PRSEEN(e, 0);
    SET_PRVALUE(e, val);
    SET_PRENV(e, RNIL);
}

SEXP evalPromise(SEXP e)
{
    if (PRVALUE(e) == R_UnboundValue)
        forcePromise(e);
    return PRVALUE(e);
}

static R_INLINE SEXP evalSelf(SEXP e, Rboolean push)
{
    /* Make sure constants in expressions are NAMED before being
       used as values.  Setting NAMED to 2 makes sure weird calls
       to assignment functions won't modify constants in
       expressions.  */

    if (NAMED(e) != 2)  /* RA_TODO is the "if" necessary? */
        SET_NAMED(e, 2);
    if (push && jitCompiling())
        genjitPush(e);
    return e;
}

static R_INLINE SEXP evalSym(SEXP e, SEXP rho)
{
    SEXP loc = NULL, val;
    jitUnresolved++;   /* genjitPushsym below will decr if can compile e */

    if (e == R_DotsSymbol)
        error(_("'...' used in an incorrect context"));

    if (DDVAL(e))
        val = ddfindVar(e,rho);
    else if (!jitCompiling() || is_user_database(rho))
        val = findVar(e, rho);
    else {
        loc = findVarLoc(e, rho); /* jit compiling so need loc */
        val = getSymValFromLoc(loc);
        checkLocationConsistency(e, rho, loc, val);
    }
    if (val == R_UnboundValue)
        error(_("object \"%s\" not found"), CHAR(PRINTNAME(e)));
    else if (val == R_MissingArg && !DDVAL(e)) {
        if (CHAR(PRINTNAME(e))[0])
            error(_("argument \"%s\" is missing, with no default"),
                  CHAR(PRINTNAME(e)));
        else
            error(_("argument is missing, with no default"));
    }
    else if (TYPEOF(val) == PROMSXP) {
        PROTECT(val);
        val = eval(val, rho);
        SET_NAMED(val, 2);
        UNPROTECT(1);
    }
    if (!isNull(val)) {
        if (NAMED(val) < 1)
            SET_NAMED(val, 1);
        if (jitCompiling())
            genjitPushsym(loc, rho);
    }
    return val;
}

static R_INLINE SEXP evalLangSpecial(SEXP e, SEXP rho, SEXP op)
{
    SEXP tmp;
    int save = R_PPStackTop, flag = PRIMPRINT(op);
    void *vmax = vmaxget();
    PROTECT(CDR(e));
    R_Visible = flag != 1;
    tmp = PRIMFUN(op) (e, op, CDR(e), rho);
#ifdef CHECK_VISIBILITY
    if(flag < 2 && R_Visible == flag) {
        char *nm = PRIMNAME(op);
        if(strcmp(nm, "for")
           && strcmp(nm, "repeat") && strcmp(nm, "while")
           && strcmp(nm, "[[<-") && strcmp(nm, "on.exit"))
            printf("vis: special %s\n", nm);
    }
#endif
    if (flag < 2) R_Visible = flag != 1;
    UNPROTECT(1);
    check_stack_balance1(op, save);
    vmaxset(vmax);
    return tmp;
}

static R_INLINE SEXP evalLangBuiltin(SEXP e, SEXP rho, SEXP op)
{
    extern int R_Profiling;
    SEXP tmp;
    int save = R_PPStackTop, flag = PRIMPRINT(op);
    void *vmax = vmaxget();
    RCNTXT cntxt;
    PROTECT(tmp = evalList(CDR(e), rho, op));
    if (flag < 2) R_Visible = flag != 1;
    /* We used to insert a context only if profiling,
       but helps for tracebacks on .C etc. */
    if (R_Profiling || (PPINFO(op).kind == PP_FOREIGN)) {
        begincontext(&cntxt, CTXT_BUILTIN, e,
                     R_BaseEnv, R_BaseEnv, RNIL, RNIL);
        PROTECT(tmp = PRIMFUN(op) (e, op, tmp, rho));
        endcontext(&cntxt);
    } else
        PROTECT(tmp = PRIMFUN(op) (e, op, tmp, rho));

#ifdef CHECK_VISIBILITY
    if(flag < 2 && R_Visible == flag) {
        char *nm = PRIMNAME(op);
        printf("vis: builtin %s\n", nm);
    }
#endif
    if (flag < 2) R_Visible = flag != 1;
    UNPROTECT(2);
    check_stack_balance1(op, save);
    vmaxset(vmax);
    return tmp;
}

static R_INLINE SEXP evalLangClos(SEXP e, SEXP rho, SEXP op)
{
    SEXP tmp;
    PROTECT(tmp = promiseArgs(CDR(e), rho));
    tmp = applyClosure(e, op, tmp, rho, R_BaseEnv);
    UNPROTECT(1);
    return tmp;
}

static R_INLINE SEXP evalLang(SEXP e, SEXP rho)
{
    SEXP op, tmp;
    const unsigned prevJitState = jitProlog(e, "evalLang");
    if (TYPEOF(CAR(e)) == SYMSXP)
        /* findFun will call error if the function is not found */
        PROTECT(op = findFun(CAR(e), rho));
    else
        PROTECT(op = eval(CAR(e), rho));
    if(TRACE(op) && R_current_trace_state()) {
        Rprintf("trace: ");
        PrintValue(e);
    }
    switch(TYPEOF(op)) {
        case SPECIALSXP:
            tmp = evalLangSpecial(e, rho, op);
            break;
        case BUILTINSXP:
            tmp = evalLangBuiltin(e, rho, op);
            break;
        case CLOSXP:
            tmp = evalLangClos(e, rho, op);
            break;
        default:
            error(_("attempt to apply non-function"));
            tmp = NULL;  /* -Wall */
    }
    if (prevJitState)
        jitEpilog(tmp, "evalLang");
    UNPROTECT(1);
    return tmp;
}

/* Win32: FNINIT sets the i86 floating point state to the default i.e. round
   to nearest, all exceptions masked, and 64-bit precision.  It is is needed
   because code elsewhere (perhaps in a DLL) may change the FPU control word.
   It is perhaps now unneeded because dynload.c:R_loadLibrary() fixes that.
   But for safety we call FNINIT before any func called by eval that could
   use floating point.  We avoid calling it needlessly because it is expensive.
*/

#ifdef Win32
  #ifdef _MSC_VER               /* microsoft compiler */
    #define FNINIT __asm{ fninit }
  #else
    #define FNINIT __asm__("fninit")
  #endif
#else                           /* not Win32 */
    #define FNINIT
#endif

#define PROLOG {                                                        \
    /* The use of depthsave below is necessary because of the           \
       possibility of non-local returns from evaluation.  Without this  \
       an "expression too complex error" is quite likely. */            \
                                                                        \
    depthsave = R_EvalDepth++;                                          \
                                                                        \
    /* We need to explicit set a RNIL call here to circumvent attempts  \
       to deparse the call in the error-handler */                      \
                                                                        \
    if (R_EvalDepth > R_Expressions) {                                  \
        R_Expressions = R_Expressions_keep + 500;                       \
        errorcall(RNIL,                                                 \
                  _("evaluation nested too deeply: "                    \
                    "infinite recursion / options(expressions=)?"));    \
    }                                                                   \
    R_CheckStack();                                                     \
    if (++evalcount > 1000) { /* was 100 before 2.8.0 */                \
        R_CheckUserInterrupt();                                         \
        evalcount = 0 ;                                                 \
    }                                                                   \
    FNINIT;                                                             \
}

#define EPILOG R_EvalDepth = depthsave;

/* Return value of "e" evaluated in "rho". */

SEXP eval(SEXP e, SEXP rho)
{
    SEXP tmp;
    static int evalcount = 0;
    int depthsave;

#if 0 /* RA_TODO will remove */
    e = traceEval(e, rho, "EVAL");
#endif

    R_Visible = TRUE;
    switch (TYPEOF(e)) {
    case NILSXP:
    case LISTSXP:
    case STRSXP:
    case CPLXSXP:
    case RAWSXP:
    case S4SXP:
    case SPECIALSXP:
    case BUILTINSXP:
    case ENVSXP:
    case CLOSXP:
    case VECSXP:
    case EXTPTRSXP:
    case WEAKREFSXP:
    case EXPRSXP:
        tmp = evalSelf(e, FALSE);
        break;
    case LGLSXP:
    case INTSXP:
    case REALSXP:
        tmp = evalSelf(e, TRUE);
        break;
    case SYMSXP:
        tmp = evalSym(e, rho);
        break;
    case PROMSXP:
        tmp = evalPromise(e);
        break;
    case LANGSXP:
        PROLOG;
        tmp = evalLang(e, rho);
        EPILOG;
        break;
    case JITSXP:
        PROLOG;
        tmp = evalJit(e);
        EPILOG;
        break;
#ifdef BYTECODE
    case BCODESXP:
        PROLOG;
        tmp = bcEval(e, rho);
        EPILOG;
        break;
#endif
    case DOTSXP:
        error(_("'...' used in an incorrect context"));
        tmp = RNIL;     /* -Wall */
    default:
        UNIMPLEMENTED_TYPE("eval", e);
        tmp = RNIL;     /* -Wall */
    }
    return (tmp);
}
