# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 2 of the License.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package MpgDumper;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (dev dvd dvd_maker title_list_items vts_list_items
		       vap_fact base_path ignore_errors);
use QVamps qw (read_setting replace error tr);
use File::Path;


# MpgDumper (dev, dvd, dvd_maker, title_list_items, vts_list_items,
#            vap_fact, base_path, ignore_errors, parent, name)
sub NEW
{
  my $this             = shift;
  my $dev              = shift;
  my $dvd              = shift;
  my $dvd_maker        = shift;
  my $title_list_items = shift;
  my $vts_list_items   = shift;
  my $vap_fact         = shift;
  my $base_path        = shift;
  my $ignore_errors    = shift;

  $this -> SUPER::NEW (@_);

  dev              = $dev;
  dvd              = $dvd;
  dvd_maker        = $dvd_maker;
  title_list_items = $title_list_items;
  vts_list_items   = $vts_list_items;
  vap_fact         = $vap_fact;
  base_path        = $base_path;
  ignore_errors    = $ignore_errors;
}


sub DESTROY
{
  parent () -> removeChild (this);

  dev              = undef;
  dvd              = undef;
  dvd_maker        = undef;
  title_list_items = undef;
  vts_list_items   = undef;
  vap_fact         = undef;
  base_path        = undef;

  SUPER -> DESTROY ();
}


sub run
{
  my $rm_temps = shift;

  my $base_path = base_path;

  rmtree ("$base_path/mpegs");
  mkpath ("$base_path/mpegs");
  unlink ("$base_path/vamps.inj");
  dvd_maker -> update_status (1);

  # clear program items' "copied" flag
  vts_list_items -> clear_all_copied_flags ();

  # all programs may have been de-selected
  unless (vts_list_items -> any_program_selected ())
  {
    error (&tr ("Neither any title nor any program selected to dump"));

    return -1;
  }

  my $selected_sectors = vts_list_items -> nr_of_selected_sectors ();
  my $selected_titles  = title_list_items -> selected_titles ();

  if (@{$selected_titles})
  {
    # loop over all selected titles
    foreach my $title_nr (@{$selected_titles})
    {
      my @chapter_table;
      my $any_program_selected;
      my $tsn = dvd -> title_set_nr ($title_nr);
      my $ttn = dvd -> title_nr_in_title_set ($title_nr);

      # loop over all chapters of title
      foreach my $ptt (1 .. dvd -> nr_of_ptts ($tsn, $ttn))
      {
	my $pgcn = dvd -> program_chain_nr ($tsn, $ttn, $ptt);
	my $pgn  = dvd -> program_nr ($tsn, $ttn, $ptt);

	unless ($chapter_table [$pgcn])
	{
	  # first chapter in this PGC
	  $chapter_table [$pgcn] = [];

	  # if we did not find any selected programs so far, check this PGC
	  $any_program_selected = 1
	    unless ($any_program_selected ||
		    !vts_list_items -> any_program_selected ($tsn, $pgcn));
	}

	$chapter_table [$pgcn] [$pgn] = $ptt;
      }

      # all programs of this title may have
      # been de-selected (is this an error?)
      next unless ($any_program_selected);

      my $selected_angle            = title_list_items ->
	                                selected_angle ($title_nr);
      my $selected_audio_streams    = title_list_items ->
                                        selected_audio_streams ($title_nr);
      my $selected_subtitle_streams = title_list_items ->
	                                selected_subtitle_streams ($title_nr);

      # loop over all program chains of title's VTS
      foreach my $pgcn (1 .. dvd -> nr_of_program_chains ($tsn))
      {
	my $chapter;
	my $chapter_list = $chapter_table [$pgcn];

	# skip program chains without any chapters of this title
	next unless ($chapter_list);

	# skip program chains with all programs deselected
	next unless (vts_list_items -> any_program_selected ($tsn, $pgcn));

	# loop over all programs of PGC
	foreach my $pgn (1 .. dvd -> nr_of_programs ($tsn, $pgcn))
	{
	  my $item = vts_list_items -> find_item ($tsn, $pgcn, $pgn);
	  $chapter = $chapter_list -> [$pgn] if ($chapter_list -> [$pgn]);

	  # user may have de-selected this program
	  next unless ($item -> is_selected ());

	  # mark program item as copied
	  $item -> set_copied (1);

	  my @pgm_cells   = dvd -> program_cells ($tsn, $pgcn, $pgn);
	  my @angle_cells = dvd -> refine_angle_cells ($selected_angle, $tsn,
						       $pgcn, \@pgm_cells);

	  # loop over all cells of selected angle in program
	  foreach my $cell (@angle_cells)
	  {
	    # dump single cell
	    return -1 if (dump_cell ($tsn, $pgcn, $cell,
				     $selected_audio_streams,
				     $selected_subtitle_streams,
				     $title_nr, $chapter, $selected_sectors));
	  }
	}
      }
    }
  }

  if (vts_list_items -> any_program_selected ())
  {
    # some programs not allocated to a (selected) title remaining
    # loop over all VTSs
    foreach my $tsn (1 .. dvd -> nr_of_title_sets ())
    {
      # skip VTSs without any selected programs
      next unless (vts_list_items -> any_program_selected ($tsn));

      my $nr_of_audio_streams    = dvd -> nr_of_audio_streams ($tsn);
      my $nr_of_subtitle_streams = dvd -> nr_of_subtitle_streams ($tsn);

      # loop over all program chains of VTS
      foreach my $pgcn (1 .. dvd -> nr_of_program_chains ($tsn))
      {
	# skip program chains without any selected programs
	next unless (vts_list_items -> any_program_selected ($tsn, $pgcn));

	# loop over all programs of PGC
	foreach my $pgn (1 .. dvd -> nr_of_programs ($tsn, $pgcn))
	{
	  # search corresponding item in VTS list
	  my $item = vts_list_items -> find_item ($tsn, $pgcn, $pgn);

	  # continue if item not selected or already copied
	  next unless ($item -> is_selected () && !$item -> is_copied ());

	  # loop over all cells in program
	  foreach my $cell (dvd -> program_cells ($tsn, $pgcn, $pgn))
	  {
	    # dump single cell
	    return -1 if (dump_cell ($tsn, $pgcn, $cell,
				     [ 1 .. $nr_of_audio_streams ],
				     [ 1 .. $nr_of_subtitle_streams ],
				     0, 0, $selected_sectors));
	  }
	}
      }
    }
  }

  unlink ("$base_path/vamps.inj") if ($rm_temps);

  return 0;
}


sub dump_cell
{
  my $tsn              = shift;
  my $pgcn             = shift;
  my $cell             = shift;
  my @audio_streams    = @{shift ()};
  my @subtitle_streams = @{shift ()};
  my $title_nr         = shift;
  my $chapter          = shift;
  my $selected_sectors = shift;

  $cell      = sprintf "%2d", $cell;
  my $astrms = join (",", @audio_streams);
  my $sopt   = "";

  if (@subtitle_streams)
  {
    my $sstrms = join (",", @subtitle_streams);
    $sopt      = " ";
    $sopt     .= read_setting ("/Vamps/select_subtitles");
    $sopt      = replace ($sopt, { "s" => $sstrms });
  }

  my $iopt = "";
  $iopt    = " " . read_setting ("/Vamps/ignore_read_errors")
    if (ignore_errors);

  my $base_path        = base_path;
  my $max_read_retries = read_setting ("/General/DVD_max_read_retries");

  my $cmd = read_setting ("/Vamps/evaporate_cell");
  $cmd    = replace ($cmd, { "d" => dev,     "v" => $tsn,
			     "g" => $pgcn,   "c" => $cell,
			     "a" => $astrms, "o" => $sopt,
			     "I" => $iopt,   "r" => $max_read_retries,
			     "S" => $selected_sectors * 2048,
			     "E" => (sprintf "%.4f", vap_fact),
			     "i" => (sprintf "%s/vamps.inj", base_path) });

  $title_nr    = $title_nr ? sprintf "%02d", $title_nr : "xx";
  $chapter     = $chapter  ? sprintf "%02d", $chapter  : "xx";
  my $mpg_name =
    sprintf "title_%s-chapter_%s-tsn_%02d-pgcn_%03d-cell_%03d.mpg",
    $title_nr, $chapter, $tsn, $pgcn, $cell;

  $cmd .= " > $base_path/mpegs/$mpg_name";

  my $rc = dvd_maker -> launch_sh_cmd (0, $cmd);

  return $rc ? -1 : 0;
}


1;
