# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package XmlWriter;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (dev dvd title_list_items vts_list_items fh fpc_mgr
		       mtable_items vap_fact base_path filename ignore_errors);
use DvdIfoRead;
use QVamps qw (read_setting replace error tr);
use TitleListItems;
use VTSListItems;


# XmlWriter (dev, dvd, title_list_items, vts_list_items, fpc_mgr,
#            mtable_items, vap_fact, base_path, ignore_errors, parent, name)
sub NEW
{
  my $this             = shift;
  my $dev              = shift;
  my $dvd              = shift;
  my $title_list_items = shift;
  my $vts_list_items   = shift;
  my $fpc_mgr          = shift;
  my $mtable_items     = shift;
  my $vap_fact         = shift;
  my $base_path        = shift;
  my $ignore_errors    = shift;

  $this -> SUPER::NEW (@_);

  my $fh;
  my $filename = "$base_path/dvdauthor.xml";

  unless (open $fh, ">", $filename)
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed creating dvdauthor XML file"), $filename, $!);

    return;
  }

  dev              = $dev;
  dvd              = $dvd;
  title_list_items = $title_list_items;
  vts_list_items   = $vts_list_items;
  fpc_mgr          = $fpc_mgr;
  mtable_items     = $mtable_items;
  vap_fact         = $vap_fact;
  base_path        = $base_path;
  ignore_errors    = $ignore_errors;
  filename         = $filename;
  fh               = $fh;
}


sub DESTROY
{
  this -> close ();
  parent () -> removeChild (this);

  dev              = undef;
  dvd              = undef;
  title_list_items = undef;
  vts_list_items   = undef;
  fpc_mgr          = undef;
  mtable_items     = undef;
  vap_fact         = undef;
  base_path        = undef;
  filename         = undef;

  SUPER -> DESTROY ();
}


sub close
{
  return -1 unless (fh);

  unless (CORE::close &fh)
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed writing dvdauthor XML file"), filename, $!);

    return -1;
  }

  fh = undef;

  return 0;
}


sub write
{
  my $fpc_box_checked  = shift;
  my $menu_box_checked = shift;

  return -1 unless (fh);

  # clear program items' "copied" flag
  vts_list_items -> clear_all_copied_flags ();

  # all programs may have been de-selected
  unless (vts_list_items -> any_program_selected ())
  {
    error (&tr ("Neither any title nor any program selected for copy"));

    return -1;
  }

  # write dvdauthor begin tag
  dvdauthor_begin_tag ();

  # write vmgm begin tag
  vmgm_begin_tag ();

  my $fpc_mgr      = fpc_mgr;
  my $fpc_title_nr = $fpc_mgr -> title_nr ();

  if ($fpc_box_checked && $fpc_title_nr)
  {
    fpc_begin_tag ();
    set_audio (3, $fpc_mgr -> audio_stream ());
    set_subtitle (3, $fpc_mgr -> subtitle_stream ());
    jump_title (3, $fpc_title_nr);
    fpc_end_tag ();
  }

  my $mtable_items = mtable_items;
  my $nr_of_menu_items = $mtable_items -> nr_of_items ();

  if ($menu_box_checked && $nr_of_menu_items)
  {
    menus_begin_tag ();
    menu_pgc_begin_tag ();
    menu_vob_tag ();

    for (my $i = 0; $i < $nr_of_menu_items; $i++)
    {
      button_begin_tag ($mtable_items -> label ($i));
      set_audio (5, $mtable_items -> audio_stream ($i));
      set_subtitle (5, $mtable_items -> subtitle_stream ($i));
      jump_title (5, $mtable_items -> title_nr ($i));
      button_end_tag ();
    }

    menu_pgc_end_tag ();
    menus_end_tag ();
  }

  # write vmgm end tag
  vmgm_end_tag ();

  my $selected_sectors = vts_list_items -> nr_of_selected_sectors ();
  my $selected_titles  = title_list_items -> selected_titles ();

  if (@{$selected_titles})
  {
    # loop over all selected titles
    foreach my $title_nr (@{$selected_titles})
    {
      my @chapter_marks;
      my $any_program_selected;
      my $tsn = dvd -> title_set_nr ($title_nr);
      my $ttn = dvd -> title_nr_in_title_set ($title_nr);

      # loop over all chapters of title
      foreach my $ptt (1 .. dvd -> nr_of_ptts ($tsn, $ttn))
      {
	my $pgcn = dvd -> program_chain_nr ($tsn, $ttn, $ptt);
	my $pgn  = dvd -> program_nr ($tsn, $ttn, $ptt);

	unless ($chapter_marks [$pgcn])
	{
	  # first chapter in this PGC
	  $chapter_marks [$pgcn] = [];

	  # if we did not find any selected programs so far, check this PGC
	  $any_program_selected = 1
	    unless ($any_program_selected ||
		    !vts_list_items -> any_program_selected ($tsn, $pgcn));
	}

	$chapter_marks [$pgcn] [$pgn] = 1;
      }

      # all programs of this title may have
      # been de-selected (is this an error?)
      next unless ($any_program_selected);

      my $selected_angle            = title_list_items ->
	                                selected_angle ($title_nr);
      my $selected_audio_streams    = title_list_items ->
                                        selected_audio_streams ($title_nr);
      my $selected_subtitle_streams = title_list_items ->
	                                selected_subtitle_streams ($title_nr);

      # write tags for start of title
      titleset_begin_tag ();
      titles_begin_tag ();
      video_tag ($tsn);
      audio_tags ($tsn, $selected_audio_streams);
      subpicture_tags ($tsn, $selected_subtitle_streams)
	if (@{$selected_subtitle_streams});

      # loop over all program chains of title's VTS
      foreach my $pgcn (1 .. dvd -> nr_of_program_chains ($tsn))
      {
	my $marks = $chapter_marks [$pgcn];

	# skip program chains without any chapters of this title
	next unless ($marks);

	# skip program chains with all programs deselected
	next unless (vts_list_items -> any_program_selected ($tsn, $pgcn));

	# write pgc begin tag
	return -1 if (title_pgc_begin_tag ($tsn, $pgcn));

	# loop over all programs of PGC
	foreach my $pgn (1 .. dvd -> nr_of_programs ($tsn, $pgcn))
	{
	  my $chapter_mark = 1 if ($marks -> [$pgn]);
	  my $item         = vts_list_items -> find_item ($tsn, $pgcn, $pgn);

	  # user may have de-selected this program
	  next unless ($item -> is_selected ());

	  # mark program item as copied
	  $item -> set_copied (1);

	  my @pgm_cells   = dvd -> program_cells ($tsn, $pgcn, $pgn);
	  my @angle_cells = dvd -> refine_angle_cells ($selected_angle, $tsn,
						       $pgcn, \@pgm_cells);

	  # loop over all cells of selected angle in program
	  foreach my $cell (@angle_cells)
	  {
	    # write vob tag
	    title_vob_tag ($tsn, $pgcn, $cell,
			   $selected_audio_streams,
			   $selected_subtitle_streams,
			   $chapter_mark, $selected_sectors);
	    $chapter_mark = 0;
	  }
	}

	# write pgc end tag
	title_pgc_end_tag ();
      }

      # write tags for end of title
      titles_end_tag ();
      titleset_end_tag ();
    }
  }

  if (vts_list_items -> any_program_selected ())
  {
    # some programs not allocated to a (selected) title remaining
    # loop over all VTSs
    foreach my $tsn (1 .. dvd -> nr_of_title_sets ())
    {
      # skip VTSs without any selected programs
      next unless (vts_list_items -> any_program_selected ($tsn));

      my $nr_of_audio_streams    = dvd -> nr_of_audio_streams ($tsn);
      my $nr_of_subtitle_streams = dvd -> nr_of_subtitle_streams ($tsn);

      # write tags for start of title
      titleset_begin_tag ();
      titles_begin_tag ();
      video_tag ($tsn);
      audio_tags ($tsn, [ 1 .. $nr_of_audio_streams ]);
      subpicture_tags ($tsn, [ 1 .. $nr_of_subtitle_streams ])
	if ($nr_of_subtitle_streams);

      # loop over all program chains of VTS
      foreach my $pgcn (1 .. dvd -> nr_of_program_chains ($tsn))
      {
	# skip program chains without any selected programs
	next unless (vts_list_items -> any_program_selected ($tsn, $pgcn));

	# write pgc begin tag
	pgc_begin_tag ($tsn, $pgcn);

	# loop over all programs of PGC
	foreach my $pgn (1 .. dvd -> nr_of_programs ($tsn, $pgcn))
	{
	  # search corresponding item in VTS list
	  my $item = vts_list_items -> find_item ($tsn, $pgcn, $pgn);

	  # continue if item not selected or already copied
	  next unless ($item -> is_selected () && !$item -> is_copied ());

	  # first cell goes into new program
	  my $chapter_mark = 1;

	  # loop over all cells in program
	  foreach my $cell (dvd -> program_cells ($tsn, $pgcn, $pgn))
	  {
	    # write vob tag
	    title_vob_tag ($tsn, $pgcn, $cell,
			   [ 1 .. $nr_of_audio_streams ],
			   [ 1 .. $nr_of_subtitle_streams ],
			   $chapter_mark, $selected_sectors);
	    $chapter_mark = 0;
	  }
	}

	# write pgc end tag
	pgc_end_tag ();
      }

      # write tags for end of title
      titles_end_tag ();
      titleset_end_tag ();
    }
  }

  # write dvdauthor end tag
  dvdauthor_end_tag ();

  return 0;
}


sub dvdauthor_begin_tag
{
  printf {&fh} "<dvdauthor dest=\"%s/image\">\n", base_path;
}


sub dvdauthor_end_tag
{
  print {&fh} "</dvdauthor>\n";
}


sub vmgm_begin_tag
{
  print {&fh} "  <vmgm>\n";
}


sub vmgm_end_tag
{
  print {&fh} "  </vmgm>\n";
}


sub fpc_begin_tag
{
  print {&fh} "    <fpc>\n";
}


sub fpc_end_tag
{
  print {&fh} "    </fpc>\n";
}


sub set_audio
{
  my $indent = shift;
  my $stream = shift;

  printf {&fh} "%saudio=%d;\n", "  " x $indent, $stream;
}


sub set_subtitle
{
  my $indent = shift;
  my $stream = shift;

  printf {&fh} "%ssubtitle=%d;\n",
               "  " x $indent, $stream < 0 ? 62 : $stream + 64;
}


sub jump_title
{
  my $indent = shift;
  my $ttn = shift;

  printf {&fh} "%sjump title %d;\n", "  " x $indent, $ttn;
}


sub menus_begin_tag
{
  print {&fh} "    <menus>\n";
}


sub menus_end_tag
{
  print {&fh} "    </menus>\n";
}


sub menu_pgc_begin_tag
{
  print {&fh} "      <pgc entry=\"title\" pause=\"inf\">\n";
}


sub menu_pgc_end_tag
{
  print {&fh} "      </pgc>\n";
}


sub menu_vob_tag
{
  printf {&fh} "        <vob file=\"%s/menu.mpg\"/>\n", base_path;
}


sub button_begin_tag
{
  my $name = shift;

  printf {&fh} "        <button name=\"%s\">\n", $name;
}


sub button_end_tag
{
  print {&fh} "        </button>\n";
}


sub titleset_begin_tag
{
  print {&fh} "  <titleset>\n";
  print {&fh} "    <menus>\n";
  print {&fh} "      <pgc entry=\"root\">\n";
  print {&fh} "        <pre>\n";
  print {&fh} "          jump vmgm menu;\n";
  print {&fh} "        </pre>\n";
  print {&fh} "      </pgc>\n";
  print {&fh} "    </menus>\n";
}


sub titleset_end_tag
{
  print {&fh} "  </titleset>\n";
}


sub titles_begin_tag
{
  print {&fh} "    <titles>\n";
}


sub titles_end_tag
{
  print {&fh} "    </titles>\n";
}


sub video_tag
{
  my $tsn = shift;

  my @attr;
  my $video_attr = dvd -> vts_video_attr ($tsn);
  my $format     = ( "ntsc", "pal" ) [$video_attr -> {video_format}];
  my $aspect     = ( "4:3", undef,
		     undef, "16:9" ) [$video_attr -> {display_aspect_ratio}];
  my $resolution = join ("x", dvd -> picture_size ($tsn));
  my $caption    = (dvd -> ntsc_cc ($tsn)) [0];

  push @attr, sprintf "format=\"%s\"",       $format  if ($format);
  push @attr, sprintf "aspect=\"%s\"",       $aspect  if ($aspect);
  push @attr, sprintf "resolution=\"%s\"",   $resolution;
  push @attr, sprintf "caption=\"field%d\"", $caption if ($caption);

  printf {&fh} "      <video %s />\n", join (" ", @attr);
}


sub audio_tags
{
  my $tsn     = shift;
  my @streams = @{shift ()};

  foreach my $stream (@streams)
  {
    my @attr;
    my $audio_attr = dvd -> vts_audio_attr ($tsn, $stream);
    my $fmt_code   = $audio_attr -> {audio_format};
    my $format     = ( "ac3", undef, undef, "mp2",
		       "pcm", undef, "dts" ) [$fmt_code];
    my $channels   = $audio_attr -> {channels} + 1;
    my $quant      = ( "16bps", "20bps",
		       "24bps", "drc") [$audio_attr -> {quantization}];
    my $dolby      = $audio_attr -> {application_mode} == 2 &&
                     $audio_attr -> {app_info} {surround} {dolby_encoded};
    my $samplerate = ( "48khz", "96khz" ) [$audio_attr -> {sample_frequency}];
    my $lang_code  = $audio_attr -> {lang_code};
    my $lang       = chr ($lang_code >> 8) . chr ($lang_code & 0xff)
      if ($audio_attr -> {lang_type} == 1);

    push @attr, sprintf "format=\"%s\"", $format if ($format);
    push @attr, sprintf "lang=\"%s\"",   $lang   if ($lang);

    # Since AC3 streams carry all attributes by themselves and some DVDs
    # seem to reflect these attributes definitely wrong, we leave it to
    # dvdauthor to detect them directly from the embedded stream. Dvdauthor
    # denies to accept mismatches anyway, so this seems quite reasonable.
    unless ($fmt_code == 0)
    {
      push @attr, sprintf "channels=\"%d\"",   $channels;
      push @attr, sprintf "quant=\"%s\"",      $quant;
      push @attr, sprintf "dolby=\"surround\""             if ($dolby);
      push @attr, sprintf "samplerate=\"%s\"", $samplerate if ($samplerate);
    }

    printf {&fh} "      <audio %s />\n", join (" ", @attr) if (@attr);
  }
}


sub subpicture_tags
{
  my $tsn     = shift;
  my @streams = @{shift ()};

  foreach my $stream (@streams)
  {
    my $subtitle_attr = dvd -> vts_subp_attr ($tsn, $stream);
    my $lang_code     = $subtitle_attr -> {lang_code};
    my $lang          = chr ($lang_code >> 8) . chr ($lang_code & 0xff)
      if ($subtitle_attr -> {type} == 1);

    printf {&fh} "      <subpicture lang=\"%s\" />\n", $lang if ($lang);
  }
}


sub title_pgc_begin_tag
{
  my $tsn  = shift;
  my $pgcn = shift;

  my $filename = sprintf "%s/vts%02d-pgc%03d.yuv", base_path, $tsn, $pgcn;
  my $rc       = open YUV, ">", $filename;

  foreach my $color (dvd -> palette ($tsn, $pgcn))
  {
    $rc &&= printf YUV "%06x\n", $color & 0xffffff;
  }

  $rc &&= CORE::close YUV;

  unless ($rc)
  {
    error (sprintf "%s: %s: %s",
	   &tr ("Failed creating palette file"), $filename, $!);

    return -1;
  }

  printf {&fh} "      <pgc palette=\"%s\">\n", $filename;

  return 0;
}


sub title_pgc_end_tag
{
  print {&fh} "      </pgc>\n";
}


sub title_vob_tag
{
  my $tsn              = shift;
  my $pgcn             = shift;
  my $cell             = shift;
  my @audio_streams    = @{shift ()};
  my @subtitle_streams = @{shift ()};
  my $chapter_mark     = shift;
  my $selected_sectors = shift;

  $cell      = sprintf "%2d", $cell;
  my $astrms = join (",", @audio_streams);
  my $sopt   = "";

  if (@subtitle_streams)
  {
    my $sstrms = join (",", @subtitle_streams);
    $sopt      = " ";
    $sopt     .= read_setting ("/Vamps/select_subtitles");
    $sopt      = replace ($sopt, { "s" => $sstrms });
  }

  my $iopt = "";
  $iopt    = " " . read_setting ("/Vamps/ignore_read_errors")
    if (ignore_errors);

  my $max_read_retries = read_setting ("/General/DVD_max_read_retries");

  my $cmd = read_setting ("/Vamps/evaporate_cell");
  $cmd    = replace ($cmd, { "d" => dev,     "v" => $tsn,
			     "g" => $pgcn,   "c" => $cell,
			     "a" => $astrms, "o" => $sopt,
			     "I" => $iopt,   "r" => $max_read_retries,
			     "S" => $selected_sectors * 2048,
			     "E" => (sprintf "%.4f", vap_fact),
			     "i" => (sprintf "%s/vamps.inj", base_path) });

  my @attr;
  push @attr, "file=\"$cmd |\"";
  push @attr, "chapters=\"0\"" if ($chapter_mark);

  printf {&fh} "        <vob %s />\n", join (" ", @attr);
}


1;
