# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


package QVamps;
use Qt;
use File::Basename;

require Exporter;
our (@ISA, @EXPORT_OK);
@ISA       = qw (Exporter);
@EXPORT_OK = qw (get_default_setting read_setting
		 get_default_list_setting read_list_setting
		 write_setting write_list_setting close_settings
		 compare_lists replace search info error trUtf8 tr);


# qVamps version
our $version = "0.97";

# places to look for global settings file
our @global_settings_paths =
(
  "/usr/share/qvamps/etc",
  "/usr/local/share/qvamps/etc"
);

# compute program's load prefix
our $load_prefix = $1 if ($0 =~ m|(.*)/bin/qvamps$|);

# place to look for user's private settings file
our @user_settings_paths =
(
  $ENV {"HOME"} . "/.qvamps"
);

# "hard coded" defaults for settings
our %default_settings =
(
  "/General/version"                 => $version,
  "/General/DVD_read_devices"        => [ '/dev/dvd', '/dev/dvdrecorder' ],
  "/General/DVD_burner"              => '/dev/dvdrecorder',
  "/General/DVD_burn_speed"          => 4,
  "/General/disk_change_cmd"         => '',
  "/General/burn_done_cmd"           => '',
  "/General/burn_failed_cmd"         => '',
  "/General/DVD_max_read_retries"    => 0,

  "/Installation/prefix"             => '/usr/local',
  "/Installation/messages_file"      => '%p/share/qvamps/translations/%l',

  "/MediaPlayer/play_dvd_title"      => 'mplayer -dvd-device %d %o dvd://%T',
  "/MediaPlayer/play_dvd_cells"      => 'play_cell %d %S %G %C | mplayer%o -',
  "/MediaPlayer/select_angle"        => '-dvdangle %N',
  "/MediaPlayer/select_audio"        => '-aid %i',
  "/MediaPlayer/select_subtitle"     => '-sid %n',
  "/MediaPlayer/play_dvd"            => 'kaffeine -d %i DVD',

  "/Dvdauthor/base_path"             => '~/video/dvd/%t',
  "/Dvdauthor/command"               => 'dvdauthor -x %x',
  "/Dvdauthor/spumux_mpv"            =>
                    'mplex -v 0 -f 8 -o /dev/stdout %v %a | spumux -P %x > %o',
  "/Dvdauthor/spumux_mpg"            => 'spumux -P %x < %i > %o',
  "/Dvdauthor/burn_cmds"             =>
    [ 's=`mkisofs -dvd-video -quiet -print-size %i`; ' .
      'mkisofs -dvd-video -V "%v" %i | ' .
      'cdrecord dev=%d speed=%s driveropts=burnfree ' .
               'fs=16M tsize=${s}s -dao -v -',
      'growisofs -dvd-compat -speed=%s ' .
                '-use-the-force-luke=dao -use-the-force-luke=tty ' .
                '-use-the-force-luke=notray -Z %d -dvd-video -V "%v" %i' ],
  "/Dvdauthor/burn_cmds_item"        => 0,

  "/Vamps/evaporate_cell"            =>
       'play_cell -r %r -g%I %d %v %g %c | vamps -E %E -a %a%o -S %S -i %i -v',
  "/Vamps/select_subtitles"          => '-s %s',
  "/Vamps/ignore_read_errors"        => '-i',
  "/Vamps/dvdr_total_sectors"        => 2294922,
  "/Vamps/dvdr_spare_sectors"        => 1500,

  "/Menu/Preferences/tv_standard"    => 'NTSC',
  "/Menu/Preferences/background"     => '%p/share/qvamps/penguin.jpg',
  "/Menu/Preferences/audio_es"       => '%p/share/qvamps/silence.mpa',
  "/Menu/Preferences/audio_language" => 'en',
  "/Menu/Preferences/audio_format"   => 'Dolby AC-3',
  "/Menu/Preferences/audio_channels" => '5.1',
  "/Menu/Preferences/ppm2mpv_ntsc"   =>
            'ppmtoy4m -n 1 -F30000:1001 -A10:11 -I t -L -S 420_mpeg2 -v 0 | ' .
            'mpeg2enc -f 8 -v 0 -n n -a 2 -o %o',
  "/Menu/Preferences/ppm2mpv_pal"    =>
            'ppmtoy4m -n 1 -F25:1 -A59:54 -I t -L -S 420_mpeg2 -v 0 | ' .
            'mpeg2enc -f 8 -v 0 -n p -o %o',

  "/Menu/Spacing/top"                => 60,
  "/Menu/Spacing/heading"            => 50,
  "/Menu/Spacing/subheading"         => 30,
  "/Menu/Spacing/button"             => 20,
  "/Menu/Spacing/bottom"             => 50,

  "/Menu/Fonts/heading"              => [ "Arial", 36, 50, 0 ],
  "/Menu/Fonts/subheading"           => [ "Arial", 18, 50, 0 ],
  "/Menu/Fonts/buttons"              => [ "Arial", 22, 50, 0 ],

  "/Menu/Colors/heading"             => [  74, 228,  94 ],
  "/Menu/Colors/subheading"          => [ 219, 160,  41 ],
  "/Menu/Colors/button_base"         => [ 219, 160,  41 ],
  "/Menu/Colors/button_hilited"      => [ 219, 147, 255 ],
  "/Menu/Colors/button_selected"     => [ 255,   0,   0 ]
);

our $qvamps_home    = $ENV {"QVAMPS"};
our @settings_paths = @global_settings_paths;

# use settings from program's load prefix
push @settings_paths, "$load_prefix/share/etc" if ($load_prefix);

# environment overrides global settings default path
@settings_paths = ( "$qvamps_home/share/etc" ) if ($qvamps_home);

# always use user's private settings
push @settings_paths, @user_settings_paths;

our $settings = Qt::Settings ();
$settings -> setPath ("OpenSourceSoftware", "qVamps", &Qt::Settings::User);

# insert our particular places into settings search path
foreach my $path (@settings_paths)
{
  $settings -> insertSearchPath (&Qt::Settings::Unix, $path);
}

# this sets the settings file's name to "qvampsrc"
$settings -> beginGroup ("/qvamps");

migrate_version ();


sub get_default_setting
{
  my $key = shift;

  # special treatment of installation prefix
  if ($key eq "/Installation/prefix")
  {
    return $qvamps_home if ($qvamps_home);

    return $load_prefix if ($load_prefix);
  }

  # special treatment of UI language
  if ($key eq "/General/language")
  {
    my $lang = $ENV {"LANG"};

    return $lang if ($lang);
  }

  # use our hard coded default
  return $default_settings {$key};
}


sub read_setting
{
  my $key = shift;

  # lookup key in settings file(s)
  my $ok;
  my $value = $settings -> readEntry ($key, undef, $ok);

  # return value if found
  return $value if ($ok);

  $value = get_default_setting ($key);

  # write default value to users private settings file
  $settings -> writeEntry ($key, $value) if (defined ($value));

  return $value;
}


sub get_default_list_setting
{
  my $key = shift;

  # use our hard coded default
  return $default_settings {$key};
}


sub read_list_setting
{
  my $key = shift;

  # lookup key in settings file(s)
  my $ok;
  my $value = $settings -> readListEntry ($key, $ok);

  # return value if found
  return $value if ($ok);

  $value = get_default_list_setting ($key);

  # write default value to users private settings file
  $settings -> writeEntry ($key, $value) if (defined ($value));

  return $value;
}


sub write_setting
{
  my $key = shift;
  my $new = shift;

  # lookup key in settings file(s)
  my $ok;
  my $old = $settings -> readEntry ($key, undef, $ok);

  # write new value if changed
  $settings -> writeEntry ($key, $new) unless ($ok && $new eq $old);
}


sub write_list_setting
{
  my $key = shift;
  my $new = shift;

  # lookup key in settings file(s)
  my $ok;
  my $old = $settings -> readListEntry ($key, $ok);

  # write new value if changed
  $settings -> writeEntry ($key, $new) unless ($ok &&
					       compare_lists ($new, $old));
}


sub close_settings
{
  undef $settings;
}


sub compare_lists
{
  my $a = shift;
  my $b = shift;

  return 0 unless @$a == @$b;

  for (my $i = 0; $i < @$a; $i++)
  {
    return 0 if ($a -> [$i] ne $b -> [$i]);
  }

  return 1;
}


sub replace
{
  my $str = shift;
  my %rep = %{shift ()};

  $rep {"%"} = "%";

  $str =~ s/%(.)/$rep{$1}/g;

  return $str;
}


sub search
{
  my $lst = shift;
  my $num = shift;

  my $i = 0;
  grep { return $i if ($_ == $num); $i++; undef } @{$lst};

  return -1;
}


sub info
{
  my $msg = shift;

  Qt::MessageBox::information (undef, &tr ("Info"), $msg, &tr ("Ok"));
}


sub error
{
  my $msg = shift;

  Qt::MessageBox::warning (undef, &tr ("Error"), $msg, &tr ("Ok"));
}


sub migrate_version
{
  # get installed version - this will write
  # a version number to the user's private settings file
  my $settings_version = read_setting ("/General/version");

  # <0.18 --> 0.18
  if ($settings_version lt "0.18")
  {
    if (read_setting ("/Menu/Preferences/ppm2mpv_ntsc") eq
	"ppmtoy4m -n 1 -F30000:1001 -A10:11 -I t -L -v 0 | " .
	"mpeg2enc -f 8 -v 0 -n n -o %o")
    {
      # setting is still the default - do update
      write_setting ("/Menu/Preferences/ppm2mpv_ntsc",
		     $default_settings {"/Menu/Preferences/ppm2mpv_ntsc"});
    }
  }

  # <0.19 --> 0.19
  if ($settings_version lt "0.19")
  {
    if (read_setting ("/Menu/Preferences/ppm2mpv_ntsc") eq
	"ppmtoy4m -n 1 -F30000:1001 -A10:11 -I t -L -v 0 | " .
	"mpeg2enc -f 8 -v 0 -n n -a 2 -o %o")
    {
      # setting is still the default - do update
      write_setting ("/Menu/Preferences/ppm2mpv_ntsc",
		     $default_settings {"/Menu/Preferences/ppm2mpv_ntsc"});
    }

    if (read_setting ("/Menu/Preferences/ppm2mpv_pal") eq
	"ppmtoy4m -n 1 -F25:1 -A59:54 -I t -L -v 0 | " .
	"mpeg2enc -f 8 -v 0 -n p -o %o")
    {
      # setting is still the default - do update
      write_setting ("/Menu/Preferences/ppm2mpv_pal",
		     $default_settings {"/Menu/Preferences/ppm2mpv_pal"});
    }
  }

  # >=0.19
  # adjust ppmtoy4m command line depending on MJPEG Tools version
  if ($settings_version ge "0.19")
  {
    my $fh;

    if (open $fh, "-|", "ppmtoy4m -S 420_mpeg2 /dev/null 2>&1")
    {
      if (<$fh> =~ /Unknown subsampling mode option: +420_mpeg2/)
      {
	my $ppm2mpv_ntsc =  read_setting ("/Menu/Preferences/ppm2mpv_ntsc");
	$ppm2mpv_ntsc    =~ s/420_mpeg2/420mpeg2/;
	write_setting ("/Menu/Preferences/ppm2mpv_ntsc", $ppm2mpv_ntsc);

	my $ppm2mpv_pal =  read_setting ("/Menu/Preferences/ppm2mpv_pal");
	$ppm2mpv_pal    =~ s/420_mpeg2/420mpeg2/;
	write_setting ("/Menu/Preferences/ppm2mpv_pal", $ppm2mpv_pal);
      }

      close $fh;
    }
  }

  # <0.21 --> 0.21
  if ($settings_version lt "0.21")
  {
    while (my ($key, $value) = each %default_settings)
    {
      if (ref ($value) eq "ARRAY")
      {
	$value = read_setting ($key);

	if (defined ($value))
	{
	  my @list = split (/,/, $value);
	  write_list_setting ($key, \@list);
	}
      }
    }

    my ($burn_cmds, $burn_cmds_item);
    my $burn_cmd = read_setting ("/Dvdauthor/burn_cmd");
    $settings -> removeEntry ("/Dvdauthor/burn_cmd");

    if ($burn_cmd =~ /growisofs/)
    {
      $burn_cmds      = [ $default_settings {"/Dvdauthor/burn_cmds"} [0],
			  $burn_cmd ];
      $burn_cmds_item = 1;
    }
    else
    {
      $burn_cmds      = [ $burn_cmd,
			  $default_settings {"/Dvdauthor/burn_cmds"} [1] ];
      $burn_cmds_item = 0;
    }

    write_list_setting ("/Dvdauthor/burn_cmds", $burn_cmds);
    write_setting ("/Dvdauthor/burn_cmds_item", $burn_cmds_item);
  }

  # <0.96 --> 0.96
  if ($settings_version lt "0.96")
  {
    my $play_cells = read_setting ("/MediaPlayer/play_dvd_cells");
    write_setting ("/MediaPlayer/play_dvd_cells.bak", $play_cells);
    write_setting ("/MediaPlayer/play_dvd_cells",
		   $default_settings {"/MediaPlayer/play_dvd_cells"});

    my $evaporate_cell = read_setting ("/Vamps/evaporate_cell");
    write_setting ("/Vamps/evaporate_cell.bak", $evaporate_cell);
    write_setting ("/Vamps/evaporate_cell",
		   $default_settings {"/Vamps/evaporate_cell"});
  }

  # update version
  write_setting ("/General/version", $default_settings {"/General/version"});
}


sub trUtf8
{
  my $source  = shift;
  my $comment = shift;

  my ($class_name) = caller;

  return Qt::app () -> translate ($class_name, $source, $comment,
				  &Qt::Application::UnicodeUTF8);
}


*tr = *trUtf8;


1;
