__all__ = ['Runner', 'ERR_NOERROR', 'ERR_NOTORRENT', 'ERR_DUPLICATE']

import os, time, re, sys
from threading import Event
from thread import start_new_thread
from util import check_dir
from sha import sha
from urllib2 import urlopen, URLError
from math import ceil

from BitTorrent import BTFailure
from BitTorrent.download import Multitorrent, Feedback
from BitTorrent.bencode import bencode, bdecode
from BitTorrent.btformats import check_message
from BitTorrent.ConvertedMetainfo import ConvertedMetainfo

ERR_NOERROR = 0
ERR_BADPATH = 1
ERR_NOTORRENT = 2
ERR_DUPLICATE = 3

class Runner:
	def __init__(self, qtorrent):
		self.qtorrent = qtorrent
		self.config = qtorrent.config
		self.torrent_dir = os.path.expanduser('~/.qtorrent3/torrents')
		
		check_dir(self.torrent_dir)
		check_dir(os.path.expanduser('~/.qtorrent3/resume'))
		
		self.url_re = re.compile('^\w+://')
		self.torrents = {}
		
		self.hash_check_queue = []
		self.hash_check_current = None
		
		self.done_flag = Event()
		
		self.multitorrent = Multitorrent(self.config, self.done_flag, self.global_error)
		self.rawserver = self.multitorrent.rawserver
		self.rawserver.add_task(self.stats, 0)
		
		start_new_thread(self.start, ())
	
	def remove(self, hash):
		obj = self.torrents[hash]
		if obj['instance']:
			obj['instance'].shutdown()
		del self.torrents[hash]
		if self.hash_check_current == hash:
			self.hash_check_current = None
			self.rawserver.add_task(self.check_hash_check_queue, 0)
		os.unlink(obj['path'])
		self.stats(False)
	
	def scan(self):
		self.rawserver.add_task(self._scan, 0)
	
	def _scan(self):
		files = os.listdir(self.torrent_dir)
		for file in files:
			if not os.path.splitext(file)[1] == '.torrent':
				continue
			path = os.path.join(self.torrent_dir, file)
			self.add(path, False)
		for path in sys.argv[1:]:
			self.add(path)
	
	def start(self):
		self.rawserver.listen_forever()
		self.torrents = {}
	
	def global_error(self, level, text):
		print ' --- ', level, text
	
	def stats(self, sched = True):
		if sched:
			self.rawserver.add_task(self.stats, 1)
		data = {}
		for hash, obj in self.torrents.items():
			name = obj['name']
			size = obj['length']
			pieces = obj['pieces']
			got = 0
			d = obj['instance']
			progress = 0.0
			peers = seeds = 0
			dist = 0.0
			uprate = 0.0
			dnrate = 0.0
			uptop = 0.0
			dntop = 0.0
			upamt = 0
			dnamt = 0
			t = 0
			if d is not None:
				stats = d.get_status()
				obj['status'] = stats['activity']
				progress = stats['fractionDone']
				if d.started and not d.closed:
					s = stats
					dist = s['numCopies']
					if d.is_seed:
						seeds = 0 # s['numOldSeeds']
						seedsmsg = "s"
					else:
						if s['numSeeds'] + s['numPeers']:
							t = stats['timeEst']
							if t is None:
								t = -1
							if t == 0:  # unlikely
								t = 0.01
							obj['status'] = 'downloading'
						else:
							t = -1
							obj['status'] = 'connecting to peers'
						seeds = s['numSeeds']
						dnrate = stats['downRate']
					peers = s['numPeers']
					uprate = stats['upRate']
					upamt = s['upTotal']
					dnamt = s['downTotal']
					if obj['stats']:
						dntop = obj['stats'][7]
						uptop = obj['stats'][8]
						if dnrate > dntop:
							dntop = dnrate
						if uprate > uptop:
							uptop = uprate
					else:
						dntop = dnrate
						uptop = uprate
					got = stats['storage_numcomplete']
			obj['stats'] = data[hash] = (name, size, obj['status'], progress, t, dnrate, uprate, dntop, uptop, dnamt, upamt, seeds, peers, dist, pieces, got, obj['errors'][:])
		self.qtorrent.stats(data, sched)
	
	def add(self, path, store = True, **kwargs):
		try:
			if self.url_re.match(path):
				data = urlopen(path, 'rb').read()
			else:
				data = open(path, 'rb').read()
			d = bdecode(data)
		except BTFailure, e:
			print 'Failure:', e
			return ERR_NOTORRENT
		except URLError, e:
			print 'URL error:', e
			return ERR_BADPATH
		except IOError, e:
			print 'IO error:', e
			return ERR_BADPATH
		except Exception, e:
			print '--' + path + '--'
			print 'Error loading torrent data:', e.__class__
			return ERR_BADPATH
		check_message(d)
		info = d['info']
		h = sha(bencode(info)).digest()
#		f = open('foo', 'a')
#		for key, data in info.items():
#			f.write(key + ' = ' + `data` + '\n')
#		f.close()
		if h in self.torrents:
			return ERR_DUPLICATE
		
		if store:
			path = os.path.join(self.torrent_dir, os.path.split(path)[1])
			if os.path.splitext(path)[1] != '.torrent':
				path += '.torrent'
			if os.path.exists(path):
				base = os.path.splitext(path)[0] + '_'
				n = 1
				while os.path.exists(base + str(n) + '.torrent'):
					n += 1
				path = base + str(n) + '.torrent'
			open(path, 'wb').write(data)
		
		obj = kwargs
		obj.update({
			'infohash': h,
			'instance': None,
			'metainfo': d,
			'path': path,
			'file': os.path.basename(path),
			'length': 0,
			'numfiles': 0,
			'status': 'waiting for hash check',
			'errors': [],
			'stats': None,
			'piece length': info['piece length'],
			'pieces': 0,
		})
		obj['name'] = info.get('name', os.path.splitext(obj['file'])[0])
		
		if info.has_key('length'):
			obj['length'] = info['length']
			obj['numfiles'] = 1
		elif info.has_key('files'):
			length = 0
			for file in info['files']:
				obj['numfiles'] += 1
				obj['length'] += file.get('length', 0)
		obj['pieces'] = ceil(obj['length'] / obj['piece length'])
		for key in ('failure reason', 'warning message', 'announce-list'):
			if info.has_key(key):
				obj[key] = info[key]
		
		self.torrents[h] = obj
		self.stats(False)
		
		self.resume(h)
		
		return ERR_NOERROR
	
	def resume(self, infohash):
		self.torrents[infohash]['status'] = 'waiting for hash check'
		self.stats(False)
		self.hash_check_queue.append(infohash)
		self.check_hash_check_queue()
	
	def check_hash_check_queue(self):
		if self.hash_check_current is not None or not self.hash_check_queue:
			return
		self.hash_check_current = h = self.hash_check_queue.pop(0)
		self.qtorrent.getFilename(self.torrents[h])
	
	def setFilename(self, target):
		h = self.hash_check_current
		obj = self.torrents[h]
		if target is None:
			self.remove(h)
			return
		self.torrents[h]['status'] = 'checking existing data'
		self.torrents[h]['instance'] = self.multitorrent.start_torrent(ConvertedMetainfo(obj['metainfo']), self.config, self, target)
		self.stats(False)
	
	def exception(self, torrent, text):
		self.torrents[torrent.infohash]['errors'].append((-1, text))
		self.stats(False)
	
	def started(self, torrent):
		self.torrents[torrent.infohash]['status'] = 'connecting to peers'
		self.stats(False)
		
		self.hash_check_current = None
		self.check_hash_check_queue()
	
	def failed(self, torrent, is_external):
		self.torrents[torrent.infohash]['status'] = 'failed'
		self.stats(False)
		self.check_hash_check_queue()
	
	def finished(self, torrent):
		self.torrents[torrent.infohash]['status'] = 'seeding'
		self.stats(False)
	
	def error(self, torrent, level, text):
		self.torrents[torrent.infohash]['errors'].append((level, text))
		self.stats(False)
	
	def shutdown(self):
		for hash, obj in self.torrents.items():
			if obj['instance']:
				obj['instance'].shutdown()
		self.rawserver.add_task(self.done_flag.set, 1)
		while self.torrents:
			time.sleep(1)
	
	def stop(self, infohash):
		obj = self.torrents[infohash]
		if obj['instance']:
			obj['instance'].shutdown()
			obj['instance'] = None
		if infohash in self.hash_check_queue:
			self.hash_check_queue.remove(infohash)
		obj['status'] = 'stopped'
		self.stats(False)
