/*
 * Copyright (c) 2003 by Hewlett-Packard Company.  All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. 
 */

/*
 * prof_utils.h
 * 
 * Utility functions for profiling and the like.
 */

#include <signal.h>
#include <stdio.h>

/*
 * The functions used to report profiling errors and warnings.  The default
 * prints the message to file descriptor 2, appends a newline
 * and, for QPROF_error, aborts.
 */
extern void (*QPROF_error)(const char *msg);
extern void (*QPROF_warn)(const char *msg);

/*
 * Switch to the text color or style indicated by QPROF_COLOR by writing
 * the appropriate ANSI terminal escape sequence to f.
 * Does nothing if the environment variable is not set.
 */
void QPROF_start_color(FILE *f);

/*
 * Switch back to "normal" color/style.
 */
void QPROF_end_color(FILE *f);

/*
 * The folloing three functions operate on a shared data structure
 * consisting of a list of PC samples stored as a large, fixed size
 * array.  They generally must be used as a unit.
 * FIXME - The intent is to provide alternate sets of functions
 * manipulating on different data structures, such as hsd_maps.
 */

/*
 * Allocate the memory for the list of samples.
 */
void QPROF_pc_sample_list_init();

/*
 * Signal handler that simply adds PC to sample buffer.
 */
void QPROF_pc_sample_list_handler(int signo, siginfo_t * si,
			    /* struct sigcontext */ void *sc);

/*
 * Signal handler that adds all PCs on call stack to sample buffer.
 */
void QPROF_pc_sample_list_stack_handler(int signo, siginfo_t * si,
			    /* struct sigcontext */ void *sc);

/*
 * Write the profile gathered by the above signal handler to a stream.
 * Does some processing to convert it to a human-readable form.
 * The output is affected by the environment variable QPROF_GRANULARITY,
 * which can be "instruction", "line", or "function".
 */
void QPROF_pc_sample_list_print_profile(FILE *f);

/*
 * Write the profile gathered by the above signal handler to
 * the directory specified by Q_DIR in a format understood by other
 * q_ tools.
 */
void QPROF_pc_sample_list_write_q_profile(void);

/*
 * Set things up so that signal handler will be invoked regularly based on a
 * timer interrupt (or in the future event counter?).
 * Applies to this process/thread only.  Looks at environment
 * variables QPROF_REAL and QPROF_VIRTUAL to determine which timer.  ITIMER_PROF
 * is the default.  Looks at QPROF_INTERVAL to determine interval between
 * signals.  Currently the unit is microseconds, but it might sometimes be
 * events in the future.
 * Returns the signal to which the handler is bound.
 * FIXME- should we separate the environment
 * reading from the actual timer/signal setup?					*/
int QPROF_setup_signals(void (*handler)(int, siginfo_t *,
				        /* struct sigcontext */ void *));

/* Return the executable name for the current process.	*/
char * QPROF_get_exe_name(void);

/*
 * Write a pc value in human-readbale form into result_buf, of length ncols+1,
 * padding the total output to ncols.
 */
void QPROF_format_pc(char *result_buf, unsigned long pc, size_t ncols);

