/* Copyright (c) 2003 Hewlett-Packard Development Company, L.P.

   Permission is hereby granted, free of charge, to any person
   obtaining a copy of this software and associated documentation
   files (the "Software"), to deal in the Software without
   restriction, including without limitation the rights to use, copy,
   modify, merge, publish, distribute, sublicense, and/or sell copies
   of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be
   included in all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
   MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
   NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
   BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
   ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
   SOFTWARE.  */

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/mman.h>
#include <sys/stat.h>

#ifdef USE_MD5
# include <gcrypt.h>
#else
# include "crc32.h"
#endif

#include "q-lib.h"

enum
  {
    COPY_METHOD_ANY	     = 0x7,
    COPY_METHOD_MAY_HARDLINK = 0x1,
    COPY_METHOD_MAY_SOFTLINK = 0x2,
    COPY_METHOD_MAY_COPY     = 0x4
  };

static const char *q_dir;
static unsigned int q_copy_method = COPY_METHOD_ANY;

static void
q_get_dir (void)
{
  unsigned int index = 0;
  char *dir, *var;
  int ret;

  /* This environment variable controls how files are copied into the
     .q directory.  */
  var = getenv ("Q_COPY_METHOD");
  if (var)
    {
      if (strcmp (var, "any") == 0)
	q_copy_method = COPY_METHOD_ANY;
      else if (strcmp (var, "copy") == 0)
	q_copy_method = COPY_METHOD_MAY_COPY;
      else if (strcmp (var, "link") == 0)
	q_copy_method = COPY_METHOD_MAY_HARDLINK | COPY_METHOD_MAY_SOFTLINK;
      else if (strcmp (var, "softlink") == 0)
	q_copy_method = COPY_METHOD_MAY_SOFTLINK;
      else if (strcmp (var, "hardlink") == 0)
	q_copy_method = COPY_METHOD_MAY_HARDLINK;
    }

  dir = getenv ("Q_DIR");
  if (!dir)
    dir = ".q";

  dir = strdup (dir);
  var = strstr (dir, "$N");
  do
    {
      if (var)
	{
	  var[0] = index / 10;
	  var[1] = index % 10;
	  ++index;
	}
      ret = mkdir (dir, 0755);
    }
  while (var && (ret < 0 && errno == EEXIST) && (index < 100));

  q_dir = dir;
}

FILE *
q_create_file (char *buf, size_t buf_size)
{
  size_t prefix_len = strlen (buf);
  unsigned long version = 0;
  char *postfix = buf + prefix_len;
  size_t remaining_size = buf_size - prefix_len;
  char path[PATH_MAX];
  int fd;

  if (!q_dir)
    {
      q_get_dir ();
      if (!q_dir)
	return NULL;
    }

  while (1)
    {
      snprintf (postfix, remaining_size, "#%lu", version++);
      snprintf (path, sizeof (path), "%s/%s", q_dir, buf);
      fd = open (path, O_CREAT | O_WRONLY | O_EXCL, 0666);
      if (fd < 0)
	{
	  if (errno == EEXIST)
	    continue;
	  return NULL;
	}
      return fdopen (fd, "w");
    }
}

void
q_close (FILE *fp)
{
  fclose (fp);
}

static ssize_t
read_file (int fd, void *buffer, size_t size)
{
  char *bp = buffer;
  ssize_t nread;

  while ((nread = read (fd, bp, size)) > 0)
    {
      bp += nread;
      size -= nread;
    }
  if (nread < 0)
    return -1;
  return ((char *) bp - (char *) buffer);
}

/* Read a file of unknown size incrementally.  Returns a pointer to
   a contiguous buffer and the size in *SIZEP or NULL if something
   goes wrong.  */
static void *
slurp_file (int fd, size_t *sizep)
{
  struct chunk
    {
      struct chunk *next;
      char data[512*1024];
    }
  *chunk_list = NULL, *chunk, *prev = NULL, *next;
  ssize_t nread, nbytes, size = 0;
  char *buffer, *bp;

  while (1)
    {
      chunk = malloc (sizeof (*chunk));
      chunk->next = NULL;

      nread = read_file (fd, chunk->data, sizeof (chunk->data));
      if (nread > 0)
	size += nread;
      if (nread < sizeof (chunk->data))
	break;			/* reached EOF */
      if (chunk_list)
	prev->next = chunk;
      else
	chunk_list = chunk;
      prev = chunk;
    }

  buffer = malloc (size);
  if (!buffer)
    goto out;

  *sizep = size;

  for (bp = buffer, chunk = chunk_list; chunk; chunk = chunk->next)
    {
      nbytes = sizeof (chunk->data);
      if (size < nbytes)
	nbytes = size;
      memcpy (bp, chunk->data, nbytes);
      bp += nbytes;
      size -= nbytes;
    }

 out:
  for (chunk = chunk_list; chunk; chunk = next)
    {
      next = chunk->next;
      free (chunk);
    }
  return buffer;
}

static int
copy_file (const char *from_path, const char *to_path)
{
  ssize_t nread, nwritten;
  int from, to, ret = -1;
  char buf[8192];

  if ((from = open (from_path, O_RDONLY)) < 0)
    return -1;
  if ((to = open (to_path, O_WRONLY | O_CREAT | O_EXCL, 0644)) < 0)
    {
      if (errno == EEXIST)
	/* The file already exists (with the proper checksum), so we're
	   done.  */
	ret = 0;
      goto out2;
    }

  while ((nread = read (from, buf, sizeof (buf))) > 0)
    {
      nwritten = write (to, buf, nread);
      if (nwritten != nread)
	goto out1;
    }
  ret = 0;
 out1:
  close (to);
 out2:
  close (from);
  return ret;
}

#define HEX(i) "0123456789abcdef"[i];

/* Compute checksum of input, into sum_str.  Sum_str will contain a
   hex character representation of the sum, prefixed by a short string
   decribing the algorithm.  */
static void
chksum (char *sum_str, size_t sum_str_len, const char *input, size_t input_len)
{
#ifdef USE_MD5
  size_t i, j;
  size_t sum_len = gcry_md_get_algo_dlen (GCRY_MD_MD5);
  unsigned char * sum = alloca (sum_len);

  gcry_md_hash_buffer (GCRY_MD_MD5, sum, input, input_len);
  strcpy(sum_str, "md5.");
  for (i = 0, j = strlen(sum_str); i < sum_len && j < sum_str_len - 1; ++i)
    {
      sum_str[j++] = HEX(sum[i] >> 4);
      sum_str[j++] = HEX(sum[i] & 0xf);
    }
  sum_str[2*sum_len] = '\0';
#else
  unsigned int sum = ~ qprof_crc32 ((unsigned char *) input, input_len, startCRC32);
  snprintf (sum_str, sum_str_len, "crc32.%08x", sum);
#endif
}

int
q_checksummed_link (enum q_link_method lm,
		    char *buf, size_t buf_size, const char *name,
		    const char *path)
{
  unsigned int copy_method;
  int fd, ret, mapped = 1;
  size_t len, old_len;
  struct stat stat;
  int prefix_len;
  void *buffer;
  size_t file_sz;

  if (!q_dir)
    {
      q_get_dir ();
      if (!q_dir)
	return -1;
    }

  copy_method = q_copy_method;
  if (lm == Q_LINK_FORCE_COPY)
    copy_method = COPY_METHOD_MAY_COPY;

  fd = open (path, O_RDONLY);
  if (fd < 0)
    return -1;
  if (fstat (fd, &stat) < 0)
    return -1;
  file_sz = stat.st_size;
  buffer = mmap (0, file_sz, PROT_READ, MAP_PRIVATE, fd, 0);
  if (buffer == (void *) MAP_FAILED)
    {
      mapped = 0;
      if (stat.st_size > 0)
	{
	  buffer = malloc (file_sz);
	  if (read_file (fd, buffer, file_sz) < 0)
	    return -1;
	}
      else
	/* happens for /proc/kallsyms, for example */
	buffer = slurp_file (fd, &file_sz);
    }
  close (fd);

  if (!buffer)
    return -1;

  prefix_len = snprintf (buf, buf_size, "%s/.%s.", q_dir, name);
  if (prefix_len < 0)
    return -1;

  chksum (buf + prefix_len, buf_size - prefix_len, buffer, file_sz);

  if (mapped)
    munmap (buffer, file_sz);
  else
    free (buffer);

#if 0
  printf ("linking form `%s' to `%s'\n", path, buf);
#endif

  if ((copy_method & COPY_METHOD_MAY_HARDLINK)
      && ((ret = link (path, buf)) == 0
	  || (ret < 0 && errno == EEXIST)))
    ;
  else if ((copy_method & COPY_METHOD_MAY_SOFTLINK)
	   && ((ret = symlink (path, buf)) == 0
	       || (ret < 0 && errno == EEXIST)))
    ;
  else if ((copy_method & COPY_METHOD_MAY_COPY) && copy_file (path, buf) == 0)
    ;
  else
    return -1;

  old_len = strlen (buf);
  len = strlen (q_dir) + 1;
  memcpy (buf, buf + len, old_len - len);
  buf[old_len - len] = '\0';
  return 0;
}
