/*
 * Copyright (c) 2003 by Hewlett-Packard Company.  All rights reserved.
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. 
 */

#include <string.h>
#include <ctype.h>
#include <dlfcn.h>
  /*
   * RTLD_NEXT seems to often not be defined by default in dlfcn.h, in spite of
   * the fact that it is reserved in SUSV3, and thus presumably could be?
   * If at all possible, we would like to use RTLD_NEXT.  Currently this
   * is possible only if our client defines _GNU_SOURCE.  We can't define it,
   * since it may already be too late.
   */

/*
 * Macro to wrap a function fn defined in library lib.
 * Lib is typically a relative file name for the library, which
 * includes the version number, e.g. "libc.so.6".
 * We arrange to call prefn(args) before the call, and postfn(result) after
 * the call.  Either may be a macro, and prefn(args may start with local
 * declarations.  The function altfn is called instead of fn if we find
 * ourselves in a nested call to fn, e.g. because it was called by
 * dlopen.
 * 
 * We define different versions for different function arities and
 * for void/non-void results.
 * These macros expect different numbers of parameters, since we
 * must specify differing numbers of argument and result types.
 *
 * Any code including this header should be linked into a dynamic
 * library (typically with "gcc -shared ....").  Normal use then
 * involves setting LD_PRELOAD to the path of the resulting
 * library.  Any program run in that envirinment will be run linked
 * against this library and have prefn and postfn called around calls
 * to fn.
 *
 * This code is not name-space clean with respect to macros.  Only
 * externally visible symbols start with WRAP_ .  #defining symbols
 * like "len" before including this file will have unpleasant consequences.
 */

#ifndef WRAP_LOG
# include <unistd.h>
# define WRAP_LOG(msg) write(2, msg, strlen(msg))
#endif

typedef enum { WRAP_UNTOUCHED, WRAP_IN_PROGRESS, WRAP_INITIALIZED} WRAP_init_state;

void WRAP_error(char * msg) {
    WRAP_LOG(msg);
    abort();
}

#ifndef RTLD_NEXT

#define WRAP_MAX_LIB_LEN 100
/* Copy source to target, deleting trailing '.' and version number. */
static void WRAP_version_delete(char * dest, const char * source)
{
   int len = strlen(source);

   if (len >= WRAP_MAX_LIB_LEN) len = WRAP_MAX_LIB_LEN - 1;
   while (isdigit(source[len-1])) --len;
   if (source[len-1] == '.') --len;
   memcpy(dest, source, len);
   dest[len] = '\0';
}

#endif /* RTLD_NEXT */

/* Return a dl-handle to lib, or simply RTLD_NEXT		*/
static void * WRAP_get_dl_handle(const char * lib)
{
# ifdef RTLD_NEXT
    return RTLD_NEXT;
# else
    char namebuf[WRAP_MAX_LIB_LEN];
    void * dl_handle = dlopen(lib, RTLD_LAZY);
    if (NULL == dl_handle)
      {
        WRAP_version_delete(namebuf, lib);
        dl_handle = dlopen(namebuf, RTLD_LAZY);
      }
    if (NULL == dl_handle) WRAP_error("PW: Couldn't open library\n");
    return dl_handle;
# endif
}

#if defined(__GNUC__) || defined(__INTEL_COMPILER)
# define TYPEOF_CAST(e) (typeof(e))
#else
# define TYPEOF_CAST(e) /* Let it generate warning */
#endif

/* FIXME: library version handling leaves something to be desired.	*/
/* We should probably just look up the version in /proc/self/maps, and	*/
/* then just dlopen which ever library version we already have mapped.	*/
/* Currently we try the specified version, and the delete the last 	*/
/* version number and try again.					*/
#define PRELOAD_DEF_INIT(fn, lib) \
\
static WRAP_init_state WRAP_##fn##_initialized = WRAP_UNTOUCHED; \
\
static void WRAP_##fn##_initialize() \
{ \
    void * dl_handle; \
    \
    WRAP_##fn##_initialized = WRAP_IN_PROGRESS; \
    dl_handle = WRAP_get_dl_handle(lib); \
    if (NULL == dl_handle) WRAP_error("PW: Couldn't open library\n"); \
    WRAP_real_##fn = TYPEOF_CAST(WRAP_real_##fn) dlsym(dl_handle, #fn); \
    if (NULL == WRAP_real_##fn) WRAP_error("PW: Couldn't find symbol\n"); \
    WRAP_##fn##_initialized = WRAP_INITIALIZED; \
}

#define PRELOAD_WRAP0(res_tp, fn, lib, prefn, postfn, altfn) \
\
res_tp (* WRAP_real_##fn) (void); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
res_tp fn(void) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	res_tp WRAP_result; \
	prefn(); \
	WRAP_result = WRAP_real_##fn(); \
	postfn(WRAP_result); \
	return(WRAP_result); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	return altfn(); \
    } \
}

#define PRELOAD_WRAP0V(fn, lib, prefn, postfn, altfn) \
\
void (* WRAP_real_##fn) (void); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
void fn(void) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	prefn(); \
	WRAP_real_##fn(); \
	postfn(); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	altfn(); \
    } \
}

#define PRELOAD_WRAP1(res_tp, arg_tp, fn, lib, prefn, postfn, altfn) \
\
res_tp (* WRAP_real_##fn) (arg_tp); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
res_tp fn(arg_tp arg) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	res_tp WRAP_result; \
	prefn(arg); \
	WRAP_result = WRAP_real_##fn(arg); \
	postfn(WRAP_result); \
	return(WRAP_result); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	return altfn(arg); \
    } \
}

#define PRELOAD_WRAP1V(arg_tp, fn, lib, prefn, postfn, altfn) \
\
void (* WRAP_real_##fn) (arg_tp); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
void fn(arg_tp arg) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	prefn(arg); \
	WRAP_real_##fn(arg); \
	postfn(); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	altfn(arg); \
    } \
}


#define PRELOAD_WRAP2(res_tp, arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn) \
\
res_tp (* WRAP_real_##fn) (arg_tp1, arg_tp2); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
res_tp fn(arg_tp1 arg1, arg_tp2 arg2) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	res_tp WRAP_result; \
	prefn(arg1, arg2); \
	WRAP_result = WRAP_real_##fn(arg1, arg2); \
	postfn(WRAP_result); \
	return(WRAP_result); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	return altfn(arg1, arg2); \
    } \
}

#define PRELOAD_WRAP2V(arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn) \
\
void (* WRAP_real_##fn) (arg_tp1, arg_tp2); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
void fn(arg_tp1 arg1, arg_tp2 arg2) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	prefn(arg1, arg2); \
	WRAP_real_##fn(arg1, arg2); \
	postfn(); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	altfn(arg1, arg2); \
    } \
}


#define PRELOAD_WRAP3(res_tp, arg_tp1, arg_tp2, arg_tp3, \
		      fn, lib, prefn, postfn, altfn) \
\
res_tp (* WRAP_real_##fn) (arg_tp1, arg_tp2, arg_tp3); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
res_tp fn(arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	res_tp WRAP_result; \
	prefn(arg1, arg2, arg3); \
	WRAP_result = WRAP_real_##fn(arg1, arg2, arg3); \
	postfn(WRAP_result); \
	return(WRAP_result); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	return altfn(arg1, arg2, arg3); \
    } \
}

#define PRELOAD_WRAP3V(arg_tp1, arg_tp2, arg_tp3, fn, lib, \
		       prefn, postfn, altfn) \
\
void (* WRAP_real_##fn) (arg_tp1, arg_tp2, arg_tp3); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
void fn(arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	prefn(arg1, arg2, arg3); \
	WRAP_real_##fn(arg1, arg2, arg3); \
	postfn(); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	altfn(arg1, arg2, arg3); \
    } \
}

#define PRELOAD_WRAP4(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
		      fn, lib, prefn, postfn, altfn) \
\
res_tp (* WRAP_real_##fn) (arg_tp1, arg_tp2, arg_tp3, arg_tp4); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
res_tp fn(arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, arg_tp4 arg4) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	res_tp WRAP_result; \
	prefn(arg1, arg2, arg3, arg4); \
	WRAP_result = WRAP_real_##fn(arg1, arg2, arg3, arg4); \
	postfn(WRAP_result); \
	return(WRAP_result); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	return altfn(arg1, arg2, arg3, arg4); \
    } \
}

#define PRELOAD_WRAP4V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, fn, lib, \
		       prefn, postfn, altfn) \
\
void (* WRAP_real_##fn) (arg_tp1, arg_tp2, arg_tp3, arg_tp4); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
void fn(arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, arg_tp4 arg4) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	prefn(arg1, arg2, arg3, arg4); \
	WRAP_real_##fn(arg1, arg2, arg3, arg4); \
	postfn(); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	altfn(arg1, arg2, arg3, arg4); \
    } \
}

#define PRELOAD_WRAP5(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
		      arg_tp5, fn, lib, prefn, postfn, altfn) \
\
res_tp (* WRAP_real_##fn) (arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
res_tp fn(arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, \
	  arg_tp4 arg4, arg_tp5 arg5) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	res_tp WRAP_result; \
	prefn(arg1, arg2, arg3, arg4, arg5); \
	WRAP_result = WRAP_real_##fn(arg1, arg2, arg3, arg4, arg5); \
	postfn(WRAP_result); \
	return(WRAP_result); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	return altfn(arg1, arg2, arg3, arg4, arg5); \
    } \
}

#define PRELOAD_WRAP5V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, fn, lib, \
		       prefn, postfn, altfn) \
\
void (* WRAP_real_##fn) (arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5); \
\
PRELOAD_DEF_INIT(fn, lib) \
\
void fn(arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, arg_tp4 arg4, arg_tp5 arg5) \
{ \
  WRAP_retry: \
    if (WRAP_##fn##_initialized == WRAP_INITIALIZED) { \
	prefn(arg1, arg2, arg3, arg4, arg5); \
	WRAP_real_##fn(arg1, arg2, arg3, arg4, arg5); \
	postfn(); \
    } else if (WRAP_##fn##_initialized == WRAP_UNTOUCHED) { \
	WRAP_##fn##_initialize(); \
	goto WRAP_retry; \
    } else { \
	altfn(arg1, arg2, arg3, arg4, arg5); \
    } \
}

/* FIXME - Currently we only handle up to 5 arguments.  C99 	*/
/* variable argument macros appear to be insufficient to handle	*/
/* this.							*/

/* Some corresponding macro definitions for wrapping functions at */
/* link time instead.						  */
#define LD_WRAP1(res_tp, arg_tp, fn, lib, prefn, postfn) \
\
res_tp __real_##fn (arg_tp) \
\
res_tp __wrap_##fn (arg_tp arg) \
{ \
    res_tp WRAP_result; \
    prefn(arg); \
    WRAP_result = __real_fn(arg); \
    postfn(WRAP_result); \
    return(WRAP_result); \
}

#define LD_WRAP1V(arg_tp, fn, lib, prefn, postfn) \
\
void __real_##fn (arg_tp) \
\
void __wrap_##fn (arg_tp arg) \
{ \
    prefn(arg); \
    __real_fn(arg); \
    postfn(); \
}

#define LD_WRAP2(res_tp, arg_tp1, arg_tp2, fn, lib, prefn, postfn) \
\
res_tp __real_##fn (arg_tp1, arg_tp2) \
\
res_tp __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2) \
{ \
    res_tp WRAP_result; \
    prefn(arg1, arg2); \
    WRAP_result = __real_fn(arg1, arg2); \
    postfn(WRAP_result); \
    return(WRAP_result); \
}

#define LD_WRAP2V(arg_tp1, arg_tp2, fn, lib, prefn, postfn) \
\
void __real_##fn (arg_tp1, arg_tp2) \
\
void __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2) \
{ \
    prefn(arg1, arg2); \
    __real_fn(arg1, arg2); \
    postfn(); \
}

#define LD_WRAP3(res_tp, arg_tp1, arg_tp2, arg_tp3, fn, lib, prefn, postfn) \
\
res_tp __real_##fn (arg_tp1, arg_tp2, arg_tp3) \
\
res_tp __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3) \
{ \
    res_tp WRAP_result; \
    prefn(arg1, arg2, arg3); \
    WRAP_result = __real_fn(arg1, arg2, arg3); \
    postfn(WRAP_result); \
    return(WRAP_result); \
}

#define LD_WRAP3V(arg_tp1, arg_tp2, arg_tp3, fn, lib, prefn, postfn) \
\
void __real_##fn (arg_tp1, arg_tp2, arg_tp3) \
\
void __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3) \
{ \
    prefn(arg1, arg2, arg3); \
    __real_fn(arg1, arg2, arg3); \
    postfn(); \
}

#define LD_WRAP4(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
		 fn, lib, prefn, postfn) \
\
res_tp __real_##fn (arg_tp1, arg_tp2, arg_tp3, arg_tp4) \
\
res_tp __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, arg_tp4 arg4) \
{ \
    res_tp WRAP_result; \
    prefn(arg1, arg2, arg3, arg4); \
    WRAP_result = __real_fn(arg1, arg2, arg3, arg4); \
    postfn(WRAP_result); \
    return(WRAP_result); \
}

#define LD_WRAP4V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
		  fn, lib, prefn, postfn) \
\
void __real_##fn (arg_tp1, arg_tp2, arg_tp3, arg_tp4) \
\
void __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, arg_tp4, arg4) \
{ \
    prefn(arg1, arg2, arg3, arg4); \
    __real_fn(arg1, arg2, arg3, arg4); \
    postfn(); \
}

#define LD_WRAP5(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		 fn, lib, prefn, postfn) \
\
res_tp __real_##fn (arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5) \
\
res_tp __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, arg_tp4 arg4, \
		    arg_tp5 arg5) \
{ \
    res_tp WRAP_result; \
    prefn(arg1, arg2, arg3, arg4, arg5); \
    WRAP_result = __real_fn(arg1, arg2, arg3, arg4, arg5); \
    postfn(WRAP_result); \
    return(WRAP_result); \
}

#define LD_WRAP5V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		  fn, lib, prefn, postfn) \
\
void __real_##fn (arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5) \
\
void __wrap_##fn (arg_tp1 arg1, arg_tp2 arg2, arg_tp3 arg3, arg_tp4 arg4, \
		  arg_tp5 arg5) \
{ \
    prefn(arg1, arg2, arg3, arg4, arg5); \
    __real_fn(arg1, arg2, arg3, arg4, arg5); \
    postfn(); \
}

#if !defined(PRELOAD_WRAP) && !defined(LD_WRAP)
#   define PRELOAD_WRAP
#endif

#if defined(PRELOAD_WRAP) && defined(LD_WRAP)
#   error Please define at most one of PRELOAD_WRAP and LD_WRAP.
#endif

/* Generic macros for doing either	*/
#ifdef PRELOAD_WRAP

#define WRAP0(res_tp, fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP0(res_tp, fn, lib, prefn, postfn, altfn)

#define WRAP0V(fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP0V(fn, lib, prefn, postfn, altfn)

#define WRAP1(res_tp, arg_tp, fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP1(res_tp, arg_tp, fn, lib, prefn, postfn, altfn)

#define WRAP1V(arg_tp, fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP1V(arg_tp, fn, lib, prefn, postfn, altfn)

#define WRAP2(res_tp, arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP2(res_tp, arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn)

#define WRAP2V(arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP2V(arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn)

#define WRAP3(res_tp, arg_tp1, arg_tp2, arg_tp3, \
	      fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP3(res_tp, arg_tp1, arg_tp2, arg_tp3, \
		      fn, lib, prefn, postfn, altfn)
#define WRAP3V(arg_tp1, arg_tp2, arg_tp3, \
	       fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP3V(arg_tp1, arg_tp2, arg_tp3, \
		      fn, lib, prefn, postfn, altfn)

#define WRAP4(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
	      fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP4(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
		      fn, lib, prefn, postfn, altfn)
#define WRAP4V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
	       fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP4V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, \
		      fn, lib, prefn, postfn, altfn)

#define WRAP5(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	      fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP5V(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		      fn, lib, prefn, postfn, altfn)
#define WRAP5V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	       fn, lib, prefn, postfn, altfn) \
        PRELOAD_WRAP5V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		      fn, lib, prefn, postfn, altfn)


#endif

#ifdef LD_WRAP

#define WRAP0(res_tp, fn, lib, prefn, postfn, altfn) \
        LD_WRAP0(res_tp, fn, lib, prefn, postfn)

#define WRAP0V(fn, lib, prefn, postfn, altfn) \
        LD_WRAP0V(fn, lib, prefn, postfn)

#define WRAP1(res_tp, arg_tp, fn, lib, prefn, postfn, altfn) \
        LD_WRAP1(res_tp, arg_tp, fn, lib, prefn, postfn)

#define WRAP1V(arg_tp, fn, lib, prefn, postfn, altfn) \
        LD_WRAP1V(arg_tp, fn, lib, prefn, postfn)

#define WRAP2(res_tp, arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn) \
        LD_WRAP2(res_tp, arg_tp1, arg_tp2, fn, lib, prefn, postfn)

#define WRAP2V(arg_tp1, arg_tp2, fn, lib, prefn, postfn, altfn) \
        LD_WRAP2V(arg_tp1, arg_tp2, fn, lib, prefn, postfn)

#define WRAP3(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	      fn, lib, prefn, postfn, altfn) \
        LD_WRAP3(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		      fn, lib, prefn, postfn)
#define WRAP3V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	       fn, lib, prefn, postfn, altfn) \
        LD_WRAP3V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		  fn, lib, prefn, postfn)

#define WRAP4(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	      fn, lib, prefn, postfn, altfn) \
        LD_WRAP4(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		      fn, lib, prefn, postfn)
#define WRAP4V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	       fn, lib, prefn, postfn, altfn) \
        LD_WRAP4V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		  fn, lib, prefn, postfn)

#define WRAP5(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	      fn, lib, prefn, postfn, altfn) \
        LD_WRAP5(res_tp, arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		 fn, lib, prefn, postfn)
#define WRAP5V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
	      fn, lib, prefn, postfn, altfn) \
        LD_WRAP5V(arg_tp1, arg_tp2, arg_tp3, arg_tp4, arg_tp5, \
		 fn, lib, prefn, postfn)

#endif
