#!/usr/bin/env python
#file test_make_3d_plots.py

__author__ = "Jesse Stombaugh"
__copyright__ = "Copyright 2011, The QIIME Project" #consider project name
__credits__ = ["Jesse Stombaugh", "Dan Knights", "Antonio Gonzalez Pena", \
                "Yoshiki Vazquez Baeza"] #remember to add yourself
__license__ = "GPL"
__version__ = "1.7.0"
__maintainer__ = "Jesse Stombaugh"
__email__ = "jesse.stombaugh@colorado.edu"
__status__ = "Release"

from numpy import array, nan, diff
from StringIO import StringIO
from os.path import exists
from cogent.util.unit_test import TestCase, main
from os import remove, mkdir, system
from random import choice, randrange, random
import shutil
from qiime.colors import data_colors
from qiime.make_3d_plots import (make_3d_plots,scale_pc_data_matrix,
                                    auto_radius,make_mage_output,
                                    get_coord,natsort,process_custom_axes, 
                                    get_custom_coords,remove_nans,
                                    scale_custom_coords,remove_unmapped_samples,
                                    make_edges_output,make_ellipsoid_faces,
                                    make_mage_ellipsoids,subdivide,
                                    get_multiple_coords,validate_coord_files,
                                    make_3d_plots_invue, make_vectors_output, 
                                    make_subgroup_vectors,
                                    run_ANOVA_trajetories, generate_3d_plots,
                                    can_run_ANOVA_trajectories,
                                    avg_vector_for_group, weight_by_vector,
                                    windowed_diff)
from qiime.util import get_tmp_filename

class TopLevelTests(TestCase):
    """Tests of top-level functions"""

    def setUp(self):
        """define some top-level data"""
        self.data={}
        self.data['coord']=[['Sample1','Sample2'],array([[-0.2,0.07],\
                            [-0.04,0.2]]),array([0.7,0.6]),\
                            array([25.00,30.00])]
        self.data['map']=[['#Sample-ID','Day'],['Sample1','Day1'],['Sample2',\
                            'Day1']]
        self.coord_header=["Sample1","Sample2","Sample3"]
        self.coords=array([[-0.219044992,0.079674486,0.09233683],[-0.042258081,\
                        0.000204041,0.024837603],[0.080504323,-0.212014503,\
                        -0.088353435]])
        self.groups={}
        self.groups['Day1']=['Sample1','Sample2','Sample3']
        self.colors={}
        self.colors['Day1']='blue1'
        self.pct_var=array([25.00,30.00,35.00])
        self.coord_tups = [("1", "2"), ("3", "2"), ("1", "3")]
        self.colors={"Day1":"blue1"}
        self.filename='test_pca.txt'
        self.dir_path='/tmp/' 
        self.prefs={}
        self.prefs['Sample']={}   
        self.prefs['Sample']['column']="Day"
        self.background_color='black'
        self.label_color='white'
        self.mapping=[["Sample-ID","Day","Type"],["Sample1","Day1","Soil"],\
                    ["Sample2","Day1","Soil"],["Sample3","Day1","Soil"]]
        self.mapping2=[["Sample-ID","Day","Type","Height","Weight"],\
                        ["Sample1","Day1","Soil","10","60"],\
                        ["Sample2","Day1","Soil","20","55"],\
                        ["Sample3","Day1","Soil","30","50"]]
        self.axis_names = 'Height,Weight'
        self._paths_to_clean_up = []
        self._dir_to_clean_up = ''

        self.prefs_vectors={}
        self.prefs_vectors['Sample']={}   
        self.prefs_vectors['Sample']['column']="Type"
        self.coords2 = array([[0, -0.219044992,0.079674486,0.09233683], \
                                [1, -0.042258081, 0.000204041, 0.024837603],\
                                [2, 0.080504323, -0.212014503, -0.088353435],\
                                [3, 0.012345551, -0.124512513, -01142356135]])
        self.custom_axes = ['Height']
        self.add_vectors = {'vectors_algorithm': 'trajectory', 'vectors_axes': 3,\
                                'vectors': ['Height'], 'vectors_path': 'vectors_test',\
                                'eigvals': array([ 0, 2.44923871, 1.11678013, 1.01533255]),\
                                'vectors_output': {},\
                                'weight_by_vector' : False,\
                                'window_size': 1}
        self.filename_vectors = 'vectors_test'
        self.file_path_vectors = 'vectors_test_dir'

        self.data_vectors = {}
        self.data_vectors['coord'] = \
            [["Sample1","Sample2","Sample3","Sample4"],self.coords2, \
            self.add_vectors['eigvals'], self.pct_var, None, None]
        self.data_vectors['map'] = [["Sample-ID","Height","Type","Weight"],\
                            ["Sample1","10","Soil_A","60"],\
                            ["Sample2","20","Soil_A","55"],\
                            ["Sample3","30","Soil_B","50"],\
                            ["Sample4","40","Soil_B","55"]]

        # data for test_can_run_ANOVA_trajectories
        self.anova_dict_true = {'first group' : [1, 2, 3, 4, 5, 6, 7],\
            'second group' : [1, 2, 3, 4, 5, 6, 7],\
            'third group' : [1, 2, 3, 4, 5, 6, 7],\
            'fourth group' : [1, 2, 3, 4, 5, 6, 7],}
        self.anova_dict_false = {'first group' : [1], 'second group' : [2],\
            'third group' : [3], 'fourth group' : [4]}
            
        # data test_avg_vector_for_group
        self.in_avg_group_name = '14655'
        self.in_avg_ids = ['FL.3.16a.281887', 'FL.3.16a.281888',\
            'FL.3.16a.281890', 'FL.3.16a.281891']
        self.in_avg_coord_dict = {\
            'FL.3.16a.281887':array([1, 2, 3, 4]),\
            'FL.3.16a.281888':array([2, 4, 6, 8]),\
            'FL.3.16a.281890':array([1, 7, 2, 1]),\
            'FL.3.16a.281891':array([2, 7, 2, 1]),}
        self.in_avg_custom_axes = ['days_since_epoch']
        self.in_avg_add_vectors = {'vectors_path': 'vectors_output.txt',\
            'vectors_algorithm': None,\
            'vectors': {'FL_R17': [('14424', 'FL.3.16a.281887'),\
            ('14435', 'FL.3.16a.281888')],\
            'FL_R18': [('14424', 'FL.3.16a.281890'),\
            ('14435', 'FL.3.16a.281891')]}}
        self.out_avg_coord_dict = {\
            '14435' : array([2, 5.5, 4.0, 4.5]),\
            '14424' : array([1, 4.5, 2.5, 2.5])}
        self.out_avg_add_vectors = {'vectors_path':'vectors_output.txt',\
            'vectors_algorithm': None,\
            'vectors':\
            {'14655': [('0', '14424'),\
            ('1', '14435')]}}

    def tearDown(self):
        map(remove,self._paths_to_clean_up)
        if self._dir_to_clean_up != '':
            shutil.rmtree(self._dir_to_clean_up)

    def test_make_3d_plots(self):
        """make_3d_plots: main script to create kinemage and html file"""
        obs_kin=make_3d_plots(self.coord_header,self.coords,self.pct_var, \
                        self.mapping,self.prefs,self.background_color, \
                        self.label_color,plot_scaled=True,plot_unscaled=True)

        self.assertEqual(obs_kin,exp_kin_full)

        # test with multiple 'colorby' columns to ensure sorting
        newprefs = {}
        newprefs['Type']={}   
        newprefs['Type']['column']="Type"
        newprefs['Day']={}   
        newprefs['Day']['column']="Day"
        obs_kin=make_3d_plots(self.coord_header,self.coords,self.pct_var, \
                            self.mapping,newprefs,self.background_color, \
                            self.label_color,plot_scaled=True,plot_unscaled=True)
        text = '\n'.join(obs_kin)
        
        self.assertTrue(text.find('Day_unscaled') < text.find('Type_unscaled'))
        
        # test with unscaled only
        obs_kin=make_3d_plots(self.coord_header,self.coords,self.pct_var, \
                        self.mapping,newprefs,self.background_color, \
                        self.label_color,plot_scaled=False,plot_unscaled=True)
        text = '\n'.join(obs_kin)
        
        self.assertTrue(text.find('Day_unscaled')>0)
        self.assertFalse(text.find('Day_scaled')>0)
        
        # test with scaled only
        obs_kin=make_3d_plots(self.coord_header,self.coords,self.pct_var, \
                        self.mapping,newprefs,self.background_color, \
                        self.label_color,plot_scaled=True,plot_unscaled=False)
        text = '\n'.join(obs_kin)
        
        self.assertFalse(text.find('Day_unscaled')>0)
        self.assertTrue(text.find('Day_scaled')>0)

 
        """make_3d_plots_invue: main script to create invue files"""
        data = {'map': [
                    ['SampleID', 'Treatment'], ['PC.354', 'Control'], ['PC.355', 'Control'], 
                    ['PC.607', 'Fast']], 
                'coord': [['PC.354', 'PC.355', 'PC.607',],
                array([[ -2.93088213e-01,   4.31583200e-02,  -6.10931011e-02, 
                    -4.45457646e-02,   1.83903403e-01,   5.62150139e-02, 
                    1.05297152e-01,   2.13883515e-01,  -5.71356497e-09],
                    [-2.09986576e-01,  -2.20253966e-01,   3.20511936e-02,
                    8.55386819e-02,  -2.25272826e-01,  -4.17365121e-02,
                    1.99893660e-01,  -4.00712898e-02,  -5.71356497e-09],
                    [1.08572516e-01,   3.82643187e-01,   2.15202408e-01,
                    1.05970644e-01,  -1.22775938e-01,  -2.16486387e-02, 
                    7.14806423e-03,   5.56601903e-02,  -5.71356497e-09]]), 
                array([  4.95533900e-01,   2.89252050e-01,   1.52106359e-01]),
                array([  2.72623936e+01,   1.59135495e+01,   8.36831432e+00,]), None, None]}
        groups_and_colors = [('Treatment', {'Control': ['PC.354', 'PC.355'], 'Fast': ['PC.607']}, 
            {'Control': 'blue1', 'Fast': 'red1'}, data_colors, data_colors.keys())]
        intp_pts = 2
        
        smp_lbl_exp = {'Treatment': {
                        'coords': [array([ -2.93088213e-01,   4.31583200e-02,  -6.10931011e-02,
                                    2.55000000e+02]), array([ -2.09986576e-01,  -2.20253966e-01,   
                                    3.20511936e-02, 2.55000000e+02]), array([  1.08572516e-01,
                                    3.82643187e-01,   2.15202408e-01, 6.52800000e+04])], 
                        'headrs': ['PC.354', 'PC.355', 'PC.607']}
                      }
        smp_lbl_grp_exp = {'Treatment': {
                        'Control': {
                            'coords': [array([-0.29308821,  0.04315832, -0.0610931 ]), 
                                    array([-0.26538767, -0.04464578, -0.030045  ]),
                                    array([-0.23768712, -0.13244987,  0.0010031 ]), 
                                    array([-0.20998658, -0.22025397,  0.03205119])], 
                            'headrs': ['PC.354', 'PC.355.0', 'PC.355.1', 'PC.355.2']}, 
                        'Fast': {
                            'coords': [array([ 0.10857252,  0.38264319,  0.21520241])], 
                            'headrs': ['PC.607']
                                }
                        }}
        poly_pts_exp = array([[0.16285877,0.57396478,0.32280361],\
            [-0.31497986,-0.33038095,0.04807679],[-0.31497986,-0.33038095,0.04807679],\
            [0.16285877,0.57396478,0.32280361],[0.16285877,0.57396478,0.32280361],\
            [-0.31497986,-0.33038095,0.04807679],[-0.31497986,-0.33038095,0.04807679]])
                      
        smp_lbl, smp_lbl_grp, poly_pts = make_3d_plots_invue(data, groups_and_colors, intp_pts, \
            polyh_pts=4, offset=1.5)
        
        self.assertFloatEqual(smp_lbl['Treatment']['coords'],smp_lbl_exp['Treatment']['coords'], 16711680.0)
        self.assertFloatEqual(smp_lbl_grp['Treatment']['Control']['coords'],\
                  smp_lbl_grp_exp['Treatment']['Control']['coords'], 16711680.0)
        self.assertFloatEqual(poly_pts,poly_pts_exp)


    def test_make_3d_plots_with_vectors(self):
        """Check different cases when creating vector 3d plots"""
        # test with custom axes
        custom_axes = ['Height','Weight']
        coord_data = array([[10,60,-0.219044992,0.079674486,0.09233683],
                            [20,55,-0.042258081, 0.000204041,0.024837603],
                            [30,50,0.080504323,-0.212014503,-0.088353435]])
        coords = [self.coord_header, coord_data]
        scale_custom_coords(custom_axes,coords)
        obs_kin=make_3d_plots(self.coord_header,coords[1],self.pct_var, \
                            self.mapping2,self.prefs,self.background_color,\
                            self.label_color,custom_axes=custom_axes, \
                            plot_scaled=True,plot_unscaled=True)
        self.assertEqual(obs_kin, exp_kin_full_axes)

        # test with custom axes and vector creation
        obs_kin = make_3d_plots(self.coord_header, self.coords2, self.pct_var, \
                            self.mapping2, self.prefs, self.background_color, \
                            self.label_color, custom_axes=self.custom_axes, \
                            add_vectors=self.add_vectors)
        self.assertEqual(obs_kin, exp_kin_vectors)

        # test with custom axes, vector creation and averages per group
        coord_header = ['PC.636', 'PC.635', 'PC.356', 'PC.481', 'PC.354',
            'PC.593', 'PC.355', 'PC.607', 'PC.634']

        coords = array([
            [5.60798235e-01, -2.76542164e-01, -1.44964375e-01, 6.66467344e-02,
            -6.77109454e-02, 1.76070270e-01, 7.29693901e-02, -2.29889464e-01,
            -4.65989417e-02, -4.84141987e-09],
            [3.65768779e-01, -2.37661394e-01, 4.60527773e-02, -1.38135815e-01,
            1.59061025e-01, -2.47484699e-01, -1.15211468e-01, -1.12864033e-01,
            6.47940730e-02, -4.84141987e-09],
            [-8.92999530e-02, 2.28820400e-01, -1.30142097e-01, -2.87149448e-01,
            8.64498846e-02, 4.42951919e-02, 2.06042607e-01, 3.10003571e-02,
            7.19920437e-02, -4.84141987e-09],
            [4.30778598e-01, 4.22628481e-02, -1.39681512e-02, 6.35314616e-02,
            -3.46120552e-01, -1.27813808e-01, 1.39350721e-02, 3.00206887e-02,
            1.40147849e-01, -4.84141987e-09],
            [-3.49339228e-01, 2.80399118e-01, -6.01282860e-03, 2.34854344e-02,
            -4.68109475e-02, -1.46624450e-01, 5.66979125e-03, -3.54299634e-02,
            -2.55785794e-01, -4.84141987e-09],
            [5.60798235e-01, 2.32872767e-01, 1.39788385e-01, 3.22871080e-01,
            1.83347007e-01, 2.04661597e-02, 5.40589147e-02, -3.66250872e-02,
            9.98235721e-02, -4.84141987e-09],
            [-2.19319591e-01, 1.70517582e-01, -1.94113269e-01, -3.08965283e-02,
            1.98086159e-02, 1.55100063e-01, -2.79923942e-01, 5.76092516e-02,
            2.42481862e-02, -4.84141987e-09],
            [-3.49339228e-01, -9.13299284e-02, 4.24147148e-01, -1.35627421e-01,
            -5.75194809e-02, 1.51363491e-01, -2.53935676e-02, 5.17306152e-02,
            -3.87382176e-02, -4.84141987e-09],
            [1.05729503e-01, -3.49339228e-01, -1.20787590e-01, 1.15274502e-01,
            6.94953934e-02, -2.53722183e-02, 6.78532019e-02, 2.44447635e-01,
            -5.98827706e-02, -4.84141987e-09]])

        pct_var = [2.66887049e+01, 1.62563704e+01, 1.37754129e+01,
            1.12172158e+01, 1.00247750e+01, 8.22835130e+00, 7.55971174e+00,
            6.24945796e+00, 1.17437419e-14]

        mapping = [['SampleID', 'Subject', 'Group', 'Time', 'Description'],
            ['PC.354', 'Subject1', 'A', '-2', 'Control_mouse_I.D._354'],
            ['PC.355', 'Subject1', 'A', '-1', 'Control_mouse_I.D._355'],
            ['PC.356', 'Subject1', 'A', '0', 'Control_mouse_I.D._356'],
            ['PC.481', 'Subject1', 'A', '4', 'Control_mouse_I.D._481'],
            ['PC.593', 'Subject1', 'A', '5', 'Control_mouse_I.D._593'],
            ['PC.607', 'Subject2', 'A', '-2', 'Fasting_mouse_I.D._607'],
            ['PC.634', 'Subject2', 'A', '1.5', 'Fasting_mouse_I.D._634'],
            ['PC.635', 'Subject2', 'A', '3.5', 'Fasting_mouse_I.D._635'],
            ['PC.636', 'Subject2', 'A', '5', 'Fasting_mouse_I.D._636']]

        prefs = {'Description': {'column': 'Description'},
            'Time': {'column': 'Time'}, 'Group': {'column': 'Group'},
            'SampleID': {'column': 'SampleID'}, 'Subject':{'column': 'Subject'}}

        background_color = 'black'
        label_color = 'white'
        taxa = None
        custom_axes = ['Time']
        edges = None
        coords_low = None
        coords_high = None
        ellipsoid_prefs = {'alpha': 0.33, 'smoothness': 1}
        user_supplied_edges = False
        add_vectors = {'vectors_algorithm': None, 'weight_by_vector': False,
        'vectors': ['Subject', 'Time'], 'vectors_path': 'vectors_output.txt'}
        plot_scaled = False
        plot_unscaled = True

        observed_reslut = make_3d_plots(coord_header, coords, pct_var, mapping,
                    prefs, background_color, label_color, taxa, custom_axes,
                    edges, coords_low, coords_high, ellipsoid_prefs,
                    user_supplied_edges, ball_scale=1.0,
                    arrow_colors={'line_color': 'white', 'head_color': 'red'},
                    add_vectors=add_vectors, plot_scaled=False, plot_unscaled=True)
        self.assertEquals(observed_reslut, exp_kin_vectors_special_a)

        # test for the case with no custom axes
        custom_axes = None
        coords = array([[ -2.76542164e-01, -1.44964375e-01, 6.66467344e-02,
            -6.77109454e-02, 1.76070270e-01,  7.29693901e-02, -2.29889464e-01,
            -4.65989417e-02, -4.84141987e-09],
            [-2.37661394e-01, 4.60527773e-02, -1.38135815e-01, 1.59061025e-01,
            -2.47484699e-01, -1.15211468e-01, -1.12864033e-01, 6.47940730e-02,
            -4.84141987e-09],
            [2.28820400e-01, -1.30142097e-01, -2.87149448e-01, 8.64498846e-02,
            4.42951919e-02, 2.06042607e-01, 3.10003571e-02, 7.19920437e-02,
            -4.84141987e-09],
            [4.22628481e-02, -1.39681512e-02, 6.35314616e-02, -3.46120552e-01,
            -1.27813808e-01, 1.39350721e-02, 3.00206887e-02, 1.40147849e-01,
            -4.84141987e-09],
            [2.80399118e-01, -6.01282860e-03, 2.34854344e-02, -4.68109475e-02,
            -1.46624450e-01, 5.66979125e-03, -3.54299634e-02, -2.55785794e-01,
            -4.84141987e-09],
            [2.32872767e-01, 1.39788385e-01, 3.22871080e-01, 1.83347007e-01,
            2.04661597e-02, 5.40589147e-02, -3.66250872e-02, 9.98235721e-02,
            -4.84141987e-09],
            [1.70517582e-01, -1.94113269e-01, -3.08965283e-02, 1.98086159e-02,
            1.55100063e-01, -2.79923942e-01, 5.76092516e-02, 2.42481862e-02,
            -4.84141987e-09],
            [-9.13299284e-02, 4.24147148e-01, -1.35627421e-01, -5.75194809e-02,
            1.51363491e-01, -2.53935676e-02, 5.17306152e-02, -3.87382176e-02,
            -4.84141987e-09],
            [-3.49339228e-01, -1.20787590e-01, 1.15274502e-01, 6.94953934e-02,
            -2.53722183e-02, 6.78532019e-02, 2.44447635e-01, -5.98827706e-02,
            -4.84141987e-09]])
        add_vectors = {'vectors_algorithm': None, 'weight_by_vector': False,
            'vectors': ['Subject', 'Time'], 'vectors_path': 'vectors_output.txt'}

        observed_reslut = make_3d_plots(coord_header, coords, pct_var, mapping,
                    prefs, background_color, label_color, taxa, custom_axes,
                    edges, coords_low, coords_high, ellipsoid_prefs,
                    user_supplied_edges, ball_scale=1.0,
                    arrow_colors={'line_color': 'white', 'head_color': 'red'},
                    add_vectors=add_vectors, plot_scaled=False, plot_unscaled=True)
        self.assertEquals(observed_reslut, exp_kin_vectors_special_b)

    def test_generate_3d_plots(self):
        """generate_3d_plots: Writes the output of make_3d_plots into a file"""
        # Tests could have failed prior to this execution, therefore check if
        # any of the following dirs exist already and try to remove them
        try:
            shutil.rmtree(self.file_path_vectors)
            remove(self.filename_vectors)
            remove('vectors_raw_values.txt')
            remove("{0}_3D_PCoA_plots.html".format(self.filename_vectors))
        except OSError:
            pass

        # Create a dir for the vector files
        mkdir(self.file_path_vectors)

        generate_3d_plots(self.prefs_vectors, self.data_vectors, self.custom_axes, self.background_color, \
                        self.label_color, data_file_path=self.file_path_vectors, \
                        filename=self.filename_vectors, add_vectors=self.add_vectors)

        # Check that the files are created
        self.assertTrue(exists(self.filename_vectors))
        self.assertTrue(exists("{0}_3D_PCoA_plots.html".format(self.filename_vectors)))

        # Remove the temp directory and all the other files
        shutil.rmtree(self.file_path_vectors)
        remove(self.filename_vectors)
        remove('vectors_raw_values.txt')
        remove("{0}_3D_PCoA_plots.html".format(self.filename_vectors))

    def test_scale_pc_data_matrix(self):
        """scale_pc_data_matrix: Scales the pc data for use in the 3d plots"""
        exp=array([[-1.56460709e-01,6.82924166e-02,9.23368300e-02],\
                    [-3.01843436e-02,1.74892286e-04,2.48376030e-02],\
                    [5.75030879e-02,-1.81726717e-01,-8.83534350e-02]])

        obs=scale_pc_data_matrix(self.coords, self.pct_var)

        self.assertFloatEqual(obs,exp)
  
    def test_auto_radius(self):
        """auto_radius: determines the radius for the circles in the plot"""
        exp=array([0.00299549315])

        obs=auto_radius(self.coords)

        self.assertFloatEqual(obs,exp)
    
    def test_make_mage_output(self):
        """make_mage_output: Create kinemage string given the data"""
        # test without custom axes
        obs_kin=make_mage_output(self.groups,self.colors,self.coord_header,\
                                self.coords,self.pct_var,self.background_color,\
                                self.label_color,data_colors)
        self.assertEqual(obs_kin,exp_kin_partial)

        # test with custom axes
        custom_axes = ['Height','Weight']
        coord_data = array([[10,60,-0.219044992,0.079674486,0.09233683],
                            [20,55,-0.042258081, 0.000204041,0.024837603],
                            [30,50,0.080504323,-0.212014503,-0.088353435]])
        coords = [self.coord_header, coord_data]
        scale_custom_coords(custom_axes,coords)
        obs_kin=make_mage_output(self.groups,self.colors,self.coord_header,\
                                coords[1],self.pct_var,self.background_color, \
                                self.label_color,data_colors,
                                custom_axes=custom_axes)
        self.assertEqual(obs_kin, exp_kin_partial_axes)

    def test_make_edge_output(self):
        """make_edge_output: Create kinemage string given the data"""
        # test without custom axes
        exp_result = ['@vectorlist {edges} dimension=4 on master={edges} nobutton', '1.0 2.0 3.0 4.0 white', '1.06 2.06 3.06 4.06 white P', '1.06 2.06 3.06 4.06 hotpink', '1.1 2.1 3.1 4.1 hotpink P', '1.0 2.0 3.0 4.0 white', '1.12 2.12 3.12 4.12 white P', '1.12 2.12 3.12 4.12 hotpink', '1.2 2.2 3.2 4.2 hotpink P']
        edges = [['a_0','a_1'],['a_0','a_2']]
        coord_dict = {}
        coord_dict['a_0'] = array([ 1.0, 2.0, 3.0, 4.0])
        coord_dict['a_1'] = array([ 1.1, 2.1, 3.1, 4.1])
        coord_dict['a_2'] = array([ 1.2, 2.2, 3.2, 4.2])
        num_coords=4
        arrow_colors={'line_color': 'white', 'head_color': 'hotpink'}
        obs_result=make_edges_output(coord_dict, edges, num_coords, \
                        self.label_color, arrow_colors=arrow_colors)
        
        self.assertEqual(obs_result, exp_result)
  
    def test_make_vectors_output(self):
        """make_vectors_output: Create kinemage string given the data"""
        vectors = {'vectors': {'test': [('1', 'a_0'), ('2', 'a_1'), ('3', 'a_0')]}}
        coord_dict = {}
        coord_dict['a_0'] = array([ 1.0, 2.0, 3.0, 4.0])
        coord_dict['a_1'] = array([ 1.1, 2.1, 3.1, 4.1])
        coord_dict['a_2'] = array([ 1.2, 2.2, 3.2, 4.2])
        num_coords=4
        color = 'hotpink'
        ids = ['a_0']
        
        # test without custom axes
        exp_result = ['@vectorlist {trace} dimension=4 on', \
            '1.0 2.0 3.0 4.0 hotpink', \
            '1.05 2.05 3.05 4.05 hotpink P', \
            '1.05 2.05 3.05 4.05 hotpink', \
            '1.1 2.1 3.1 4.1 hotpink P']
            
        obs_result=make_vectors_output(coord_dict, vectors, num_coords,\
            color, ids)
            
        self.assertEqual(obs_result, exp_result)
    
    def test_make_subgroup_vectors(self):
        """make_subgroup_vectors: Returns the correct vector based on groups
        per algorithm"""
        coord_dict = dict(zip(self.coord_header,self.coords))
        # eigvals values should go from high to low and pct_var goes from low to high
        eigvals = [i for i in reversed(self.pct_var)]
        ids = ['Sample1', 'Sample2', 'Sample3']

        # Testing rms
        exp_result = {'calc': {'avg':5.32480386489972}, \
                        'vector': [6.995474752449276, 1.5180408981614073, 7.4608959440884766],\
                        'message': None }
        self.assertFloatEqual(make_subgroup_vectors(coord_dict, eigvals, ids), exp_result)
        
        # Testing trajectory
        exp_result = {'calc': {'trajectory':15.92846773878855}, \
                        'vector': [14.383977976653455, 6.8423140875566384],\
                        'message': None }
        self.assertFloatEqual(make_subgroup_vectors(coord_dict, eigvals, ids, 'trajectory'), exp_result)

        # Testing first difference 
        exp_result = {'vector': array([-7.54166389]) , \
                        'calc': {'mean': [-7.541663889096816], 'std': [0.0]},\
                        'message': None }
        self.assertFloatEqual(make_subgroup_vectors(coord_dict, eigvals, ids, 'diff'), exp_result)

        # this test should return a message to print in the output file
        exp_result ={'vector': [-7.5416638890968164, 0.0], \
                        'message': None,
                        'calc': {
                            'std': 3.7708319445484082, 
                            'mean': -3.7708319445484082
                        }
                    }
        calc_result = make_subgroup_vectors(coord_dict, eigvals, ids, 'wdiff', window_size=1)
        self.assertEquals(calc_result, exp_result)
                      
    def test_run_ANOVA_trajetories(self):
        """run_ANOVA_trajetories: should return the same value
        """
        data = dict(zip(self.coord_header,self.coords))
        labels, groups_means, prob = run_ANOVA_trajetories(data)
        exp_labels = ['Sample1', 'Sample3', 'Sample2']
        exp_groups_means = [-0.015677892000000002, -0.073287871666666657, -0.0057388123333333334]
        exp_prob = 0.80674568762274179
                                     
        self.assertEqual(labels, exp_labels)
        self.assertFloatEqual(groups_means, exp_groups_means)
        self.assertFloatEqual(prob, exp_prob)        
    
    def test_process_custom_axes(self):
        """process_custom_axes: Parses the custom_axes \
        option from the command line"""
        exp = ['Height','Weight']
        obs=process_custom_axes(self.axis_names)
        self.assertEqual(obs,exp)

    def test_get_custom_coords(self):
        """get_custom_coords: Gets custom axis coords from the mapping file."""
        exp = 1
        custom_axes = ['Height','Weight']
        coords = [self.coord_header, self.coords]
        get_custom_coords(custom_axes, self.mapping2, coords)
        exp = array([[10,60,-0.219044992,0.079674486,0.09233683],
                           [20,55,-0.042258081, 0.000204041,0.024837603],
                           [30,50,0.080504323,-0.212014503,-0.088353435]])
        self.assertEqual(coords[1],exp)

    def test_scale_custom_coords(self):
        """scale_custom_coords: \
        Scales custom coordinates to match min/max of PC1"""
        custom_axes = ['Height','Weight']
        coord_data = array([[10,60,-0.219044992,0.079674486,0.09233683],
                            [20,55,-0.042258081, 0.000204041,0.024837603],
                            [30,50,0.080504323,-0.212014503,-0.088353435]])
        coords = [self.coord_header, coord_data]
        scale_custom_coords(custom_axes,coords)
        # calculate results
        mn = coord_data[2,].min()
        mx = coord_data[2,].max()
        h = array([10.0,20.0,30.0])
        h = (h-min(h))/(max(h)-min(h))
        h = h * (mx-mn) + mn
        w = array([60.0,55.0,50.0])
        w = (w-min(w))/(max(w)-min(w))
        w = w * (mx-mn) + mn
        exp = array([[h[0],w[0],-0.219044992,0.079674486,0.09233683],
                            [h[1],w[1],-0.042258081, 0.000204041,0.024837603],
                            [h[2],w[2],0.080504323,-0.212014503,-0.088353435]])
        self.assertEqual(coords[1],exp)

    def test_remove_nans(self):
        """remove_nans: Deletes any samples with NANs in their coordinates"""
        coord_data = array([[10,60,-0.219044992,0.079674486,0.09233683],
                            [20,55,-0.042258081, nan,0.024837603],
                            [30,50,0.080504323,-0.212014503,-0.088353435]])
        coords = [self.coord_header, coord_data]
        remove_nans(coords)

        exp_header = ["Sample1","Sample3"]
        exp_coords = array([[10,60,-0.219044992,0.079674486,0.09233683],
                            [30,50,0.080504323,-0.212014503,-0.088353435]])
        self.assertEqual(coords[0],exp_header)
        self.assertEqual(coords[1],exp_coords)


    def test_remove_unmapped_samples(self):
        """remove_unmapped_samples: \
        Removes any samples not present in mapping file"""
        coord_data = array([[10,60,-0.219044992,0.079674486,0.09233683],
                            [20,55,-0.042258081, nan,0.024837603],
                            [30,50,0.080504323,-0.212014503,-0.088353435]])
        coords = [self.coord_header, coord_data]
        # mapping without sample2
        mapping=[["Sample-ID","Day","Type","Height","Weight"],\
                               ["Sample1","Day1","Soil","10","60"],\
                               ["Sample3","Day1","Soil","30","50"]]
        remove_unmapped_samples(mapping, coords)
        exp_header = ["Sample1","Sample3"]
        exp_coords = array([[10,60,-0.219044992,0.079674486,0.09233683],
                           [30,50,0.080504323,-0.212014503,-0.088353435]])
        self.assertEqual(coords[0],exp_header)
        self.assertEqual(coords[1],exp_coords)
    
    def test_make_ellipsoid_faces(self):
        # test without scaling
        res =  make_ellipsoid_faces([0,0,0],[1,1,1],nsubdivs=0)
        res = [[tuple(["%.5f"%(aijk) for aijk in aij]) for aij in ai] \
                   for ai in res]
        exp = [[tuple(["%.5f"%(aijk) for aijk in aij]) for aij in ai] \
                   for ai in exp_make_ellipsoid_faces]
        self.assertEqual(res,exp)

        # test without scaling
        res =  make_ellipsoid_faces([-1,2,3],[.2,2,20],nsubdivs=0)
        res = [[tuple(["%.5f"%(aijk) for aijk in aij]) for aij in ai] \
                   for ai in res]
        exp = [[tuple(["%.5f"%(aijk) for aijk in aij]) for aij in ai] \
                   for ai in exp_make_ellipsoid_faces_transformed]
        self.assertEqual(res,exp)

    def test_make_mage_ellipsoids(self):
        ids = ['A','B']
        coord_dict = {'A':array([0,0,0]),'B':array([-1,2,3])}
        coord_low_dict = {'A':array([-.5,-.5,-.5]),'B':array([-.1,-1,-10])}
        coord_high_dict = {'A':array([.5,.5,.5]),'B':array([.1,1,10])}
        ellipsoid_prefs = {"smoothness":0,"alpha":.25}
        res = make_mage_ellipsoids(ids, coord_dict, coord_low_dict,
                                   coord_high_dict,"red1", ellipsoid_prefs)
        self.assertEqual(res,exp_make_mage_ellipsoids)

    def test_subdivide(self):
        res = subdivide((1,0,0),(0,1,0),(0,0,1))
        res = [tuple([tuple(["%.5f"%(aijk) for aijk in aij]) for aij in ai]) \
                   for ai in res]
        exp = [tuple([tuple(["%.5f"%(aijk) for aijk in aij]) for aij in ai]) \
                   for ai in exp_subdivide]
        self.assertEqual(res,exp)

    def test_get_multiple_coords(self):
        # create the temporary pc files
        pc_file_1 = '\n'.join(['pc vector number\t1\t2',
                               'A\t1.1\t2.2',
                               'B\t4.1\t4.2',
                               'C\t-.1\t-.2',
                               'eigvals\t0.52\t0.24',
                               '% variation explained\t25.12\t13.29'])
        pc_file_2 = '\n'.join(['pc vector number\t1\t2',
                               'A\t2.1\t3.2',
                               'B\t5.1\t6.2',
                               'C\t-1.1\t-2.2',
                               'eigvals\t0.32\t0.14',
                               '% variation explained\t20.11\t12.28'])

        edges_file = '\n'.join(['B A','B\t\t\t   \t\tC'])

        fp1 = get_tmp_filename()
        fp2 = get_tmp_filename()
        fp3 = get_tmp_filename()
        try:
            f1 = open(fp1,'w')
            f2 = open(fp2,'w')
            f3 = open(fp3,'w')
        except IOError, e:
            raise e,"Could not create temporary files: %s, %s" %(f1,f2, f3)
        
        f1.write(pc_file_1)
        f1.close()
        f2.write(pc_file_2)
        f2.close()
        f3.write(edges_file)
        f3.close()
        
        # test without custom edges
        exp_edges = [('A_0', 'A_1'), ('B_0', 'B_1'), ('C_0', 'C_1')]
        exp_coords = [['A_0', 'B_0', 'C_0', 'A_1', 'B_1', 'C_1'], 
                      array([[ 1.1,  2.2],
                             [ 4.1,  4.2],
                             [-0.1, -0.2],
                             [ 2.1,  3.2],
                             [ 5.1,  6.2],
                             [-1.1, -2.2]]), 
                      array([ 0.52,  0.24]), array([ 25.12,  13.29]),
                      None, None]
        edges, coords = get_multiple_coords([fp1,fp2])
        self.assertEqual(edges, exp_edges)
        self.assertEqual(coords, exp_coords)

        # test with custom edges
        exp_edges = [['B', 'A'], ['B', 'C']]
        exp_coords = [['A', 'B', 'C'],
                      array([[ 1.1,  2.2],
                             [ 4.1,  4.2],
                             [-0.1, -0.2]]), 
                      array([ 0.52,  0.24]), array([ 25.12,  13.29]),
                      None, None]
        edges, coords = get_multiple_coords([fp1], fp3)
        self.assertEqual(edges, exp_edges)
        self.assertEqual(coords, exp_coords)
        
        # clean up
        remove(fp1)
        remove(fp2)
        remove(fp3)

    def test_get_multiple_coords_serial(self):
        # create the temporary pc files
        pc_file_1 = '\n'.join(['pc vector number\t1\t2',
                                'A\t1.1\t2.2',
                                'B\t4.1\t4.2',
                                'C\t-.1\t-.2',
                                'eigvals\t0.52\t0.24',
                                '% variation explained\t25.12\t13.29'])
        pc_file_2 = '\n'.join(['pc vector number\t1\t2',
                                'A\t2.1\t3.2',
                                'B\t5.1\t6.2',
                                'C\t-1.1\t-2.2',
                                'eigvals\t0.32\t0.14',
                                '% variation explained\t20.11\t12.28'])
        pc_file_3 = '\n'.join(['pc vector number\t1\t2',
                                'A\t2.2\t3.3',
                                'B\t5.2\t6.3',
                                'C\t-1.2\t-2.3',
                                'eigvals\t0.34\t0.15',
                                '% variation explained\t10.11\t2.28'])

        fp1 = get_tmp_filename()
        fp2 = get_tmp_filename()
        fp3 = get_tmp_filename()
        try:
            f1 = open(fp1,'w')
            f2 = open(fp2,'w')
            f3 = open(fp3,'w')
        except IOError, e:
            raise e,"Could not create temporary files: %s, %s" %(f1,f2, f3)
        
        f1.write(pc_file_1)
        f1.close()
        f2.write(pc_file_2)
        f2.close()
        f3.write(pc_file_3)
        f3.close()
        
        # test without serial
        exp_edges = [('A_0', 'A_1'), ('A_0', 'A_2'), 
                    ('B_0', 'B_1'), ('B_0', 'B_2'),
                    ('C_0', 'C_1'),('C_0', 'C_2')
                    ]
        exp_coords = [['A_0', 'B_0', 'C_0',
                       'A_1', 'B_1', 'C_1',
                       'A_2', 'B_2', 'C_2'], 
                        array([[ 1.1,  2.2],
                            [ 4.1,  4.2],
                            [-0.1, -0.2],
                            [ 2.1,  3.2],
                            [ 5.1,  6.2],
                            [-1.1, -2.2],
                            [ 2.2,  3.3],
                            [ 5.2,  6.3],
                            [-1.2, -2.3]]), 
                        array([ 0.52,  0.24]), array([ 25.12,  13.29]),
                        None, None]
        edges, coords = get_multiple_coords([fp1,fp2,fp3],serial=False)
        self.assertEqual(edges, exp_edges)
        self.assertEqual(coords, exp_coords)

        # test with serial
        exp_edges = [('A_0', 'A_1'), ('A_1', 'A_2'),
                    ('B_0', 'B_1'), ('B_1', 'B_2'),
                    ('C_0', 'C_1'),('C_1', 'C_2')
                    ]
        edges, coords = get_multiple_coords([fp1,fp2,fp3],serial=True)
        self.assertEqual(edges, exp_edges)
        self.assertEqual(coords, exp_coords)

        # clean up
        remove(fp1)
        remove(fp2)
        remove(fp3)

    def test_validate_coord_files(self):
        """Verifies that validate_coord_files works correctly"""
        # create the temporary pc files
        # one line has 4 columns
        pc_file_1 = '\n'.join(['pc vector number\t1\t2',
                                'A\t1.1\t2.2',
                                'B\t4.1\t4.2',
                                'C\t-.1\t-.2',
                                'eigvals\t0.52\t0.24\t0.11',
                                '% variation explained\t25.12\t13.29'])
        # all lines have 3 columns
        pc_file_2 = '\n'.join(['pc vector number\t1\t2',
                                'A\t2.1\t3.2',
                                'B\t5.1\t6.2',
                                'C\t-1.1\t-2.2',
                                'eigvals\t0.32\t0.14',
                                '% variation explained\t20.11\t12.28'])
        # all lines have two columns
        pc_file_3 = '\n'.join(['pc vector number\t1',
                                'A\t2.1',
                                'B\t5.1',
                                'C\t-1.1',
                                'eigvals\t0.32',
                                '% variation explained\t20.11'])
        # all lines have 3 columns
        pc_file_4 = '\n'.join(['pc vector number\t1\t2',
                                'A\t1.1\t2.2',
                                'B\t4.1\t4.2',
                                'C\t-.1\t-.2',
                                'eigvals\t0.52\t0.24',
                                '% variation explained\t25.12\t13.29'])

        fp1 = get_tmp_filename()
        fp2 = get_tmp_filename()
        fp3 = get_tmp_filename()
        fp4 = get_tmp_filename()

        try:
            f1 = open(fp1,'w')
            f2 = open(fp2,'w')
            f3 = open(fp3,'w')
            f4 = open(fp4,'w')
        except IOError, e:
            raise e,"Could not create temporary files: %s, %s" %(f1,f2,f3,f4)
        
        f1.write(pc_file_1)
        f1.close()
        f2.write(pc_file_2)
        f2.close()
        f3.write(pc_file_3)
        f3.close()
        f4.write(pc_file_4)
        f4.close()
         
        # one file with internal inconsistency
        result = validate_coord_files(fp1)
        self.assertEqual(result, False)
        # one file has two columns, one has 3 
        result = validate_coord_files([fp2,fp3])
        self.assertEqual(result, False)
        # first file consistent, second file not
        result = validate_coord_files([fp2, fp1])
        self.assertEqual(result, False)
        # both files consistent
        result = validate_coord_files([fp2, fp4])
        self.assertEqual(result, True)
        
        
        # clean up
        remove(fp1)
        remove(fp2)
        remove(fp3)
        remove(fp4)


    def test_can_run_ANOVA_trajectories(self):
        """ Test for valid dicts to be passed"""
        self.assertTrue(can_run_ANOVA_trajectories(self.anova_dict_true))
        self.assertFalse(can_run_ANOVA_trajectories(self.anova_dict_false))

    def test_avg_vector_for_group(self):
        """ Create an average set of coordinates """

        test_avg_coord_dict, test_avg_add_vectors = \
            avg_vector_for_group(self.in_avg_group_name, self.in_avg_ids,\
                self.in_avg_coord_dict, self.in_avg_custom_axes,\
                self.in_avg_add_vectors)
        self.assertEqual(test_avg_coord_dict, self.out_avg_coord_dict)
        self.assertEqual(test_avg_add_vectors, self.out_avg_add_vectors)

    def test_weight_by_vector(self):
        """ Test that the vector is weighted by the given vector"""
        # data for test_weight_by_vector
        self.in_vector_to_weight = array([1, 2, 3, 4, 5, 6, 7, 8])
        self.in_weighting_vector = array([1, 5, 8, 12, 45, 80, 85, 90])
        self.out_weighted_vector = array([1, 6.3571428571428568,\
            12.714285714285714, 12.714285714285714, 1.9264069264069263,\
            2.1795918367346938, 17.800000000000001, 20.342857142857142])

        self.in_flat_vector_to_weight = array([1, 2, 3, 4, 5, 6, 7, 8])
        self.in_flat_weighting_vector = array([1, 2, 3, 4, 5, 6, 7, 8])
        self.out_flat_weighted_vector = array([1, 2, 3, 4, 5, 6, 7, 8])

        self.in_second_flat_vector_to_weight = array([2, 3, 4, 5, 6])
        self.in_second_flat_weighting_vector = array([25, 30, 35, 40, 45])
        self.out_second_flat_weighted_vector = array([2, 3, 4, 5, 6])

        self.in_nd_array_vector_to_weight = array([[1, 2, 3],[2, 3, 4],\
            [5, 6, 7],[8, 9, 10]])
        self.in_nd_array_weighting_vector = array([1, 2, 3, 4])
        self.out_nd_array_flat_vector = array([[1, 2, 3],[2, 3, 4],\
            [5, 6, 7],[8, 9, 10]])


        test_weighted_output = weight_by_vector(self.in_vector_to_weight,\
            self.in_weighting_vector)
        self.assertEqual(test_weighted_output, self.out_weighted_vector)

        flat_test_weighted_output = weight_by_vector(\
            self.in_flat_vector_to_weight,self.in_flat_weighting_vector)
        self.assertEqual(flat_test_weighted_output, self.out_flat_weighted_vector)

        second_flat_test_weighted_output = weight_by_vector(\
            self.in_second_flat_vector_to_weight,self.in_second_flat_weighting_vector)
        self.assertEqual(second_flat_test_weighted_output, self.out_second_flat_weighted_vector)

    def test_windowed_diff(self):
        """Test correct functioning of the modified first difference """

        in_array_windowed_diff = array([24, 15, 28, 16, 28, 43, 12, 53])

        # check all the disallowed beahviours
        with self.assertRaises(ValueError):
            windowed_diff(in_array_windowed_diff, -10)
        with self.assertRaises(ValueError):
            windowed_diff(in_array_windowed_diff, 8)
        with self.assertRaises(ValueError):
            windowed_diff(in_array_windowed_diff, 50)

        # checks for correct behaviour with a 1-d vector
        exp_resutl = array([-4.3333333333333321, 9.0, 1.0, \
            11.666666666666668, 8.0, -3.6666666666666643, 41.0, 0.0])
        calc_result = windowed_diff(in_array_windowed_diff, 3)
        self.assertEqual(calc_result, exp_resutl)

        exp_resutl = [-9.0, 13.0, -12.0, 12.0, 15.0, -31.0,\
            41.0, 0.0]
        calc_result = windowed_diff(in_array_windowed_diff, 1)
        self.assertEqual(calc_result, exp_resutl)

        # checks for correct behaviour with an n-d vector
        in_nd_array_windowed_diff = array([array([2, 5, 1, 6]),\
            array([7, 1, 6, 3]), array([1, 8, 3, 5]), array([7, 3, 2, 6]),\
            array([8, 1, 6, 1]), array([15, 5, 4, 1]), array([1, 5, 1, 2]),\
            array([33, 5, 67, 12])])
        exp_result = [array([ 2. , -0.5,  3.5, -2. ]),\
            array([-3. ,  4.5, -3.5,  2.5]), array([ 6.5, -6. ,  1. , -1.5]),\
            array([ 4.5,  0. ,  3. , -5. ]), array([ 0. ,  4. , -3.5,  0.5]),\
            array([  2.,   0.,  30.,   6.]), array([ 32.,   0.,  66.,  10.]),\
            array([ 0.,  0.,  0.,  0.])]
        calc_result = windowed_diff(in_nd_array_windowed_diff, 2)
        self.assertEqual(calc_result, exp_result)

        return


exp_kin_full=\
['@kinemage {Day_unscaled}', '@dimension {PC1} {PC2} {PC3}', '@dimminmax -0.219044992 0.080504323 -0.212014503 0.079674486 -0.088353435 0.09233683', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Day1 (n=3)} collapsible', '@balllist color=red1 radius=0.00299549315 alpha=0.75 dimension=3 master={points} nobutton', '{Sample1} -0.219044992 0.079674486 0.09233683\n{Sample2} -0.042258081 0.000204041 0.024837603\n{Sample3} 0.080504323 -0.212014503 -0.088353435', '@labellist color=red1 radius=0.00299549315 alpha=0.75 dimension=3 master={labels} nobutton', '{Sample1} -0.219044992 0.079674486 0.09233683\n{Sample2} -0.042258081 0.000204041 0.024837603\n{Sample3} 0.080504323 -0.212014503 -0.088353435', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=3 on', '-0.2299972416 -0.22261522815 -0.09277110675 white', '0.08452953915 -0.22261522815 -0.09277110675 white', '@labellist {PC1 (25%)} dimension=3 on', '{PC1 (25%)}0.0887560161075 -0.22261522815 -0.09277110675 white', '@vectorlist {PC2 line} dimension=3 on', '-0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 0.0836582103 -0.09277110675 white', '@labellist {PC2 (30%)} dimension=3 on', '{PC2 (30%)}-0.2299972416 0.087841120815 -0.09277110675 white', '@vectorlist {PC3 line} dimension=3 on', '-0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.22261522815 0.0969536715 white', '@labellist {PC3 (35%)} dimension=3 on', '{PC3 (35%)}-0.2299972416 -0.22261522815 0.101801355075 white', '@kinemage {Day_scaled}', '@dimension {PC1} {PC2} {PC3}', '@dimminmax -0.156460708571 0.0575030878571 -0.181726716857 0.0682924165714 -0.088353435 0.09233683', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Day1 (n=3)} collapsible', '@balllist color=red1 radius=0.00213963796429 alpha=0.75 dimension=3 master={points} nobutton', '{Sample1} -0.156460708571 0.0682924165714 0.09233683\n{Sample2} -0.0301843435714 0.000174892285714 0.024837603\n{Sample3} 0.0575030878571 -0.181726716857 -0.088353435', '@labellist color=red1 radius=0.00213963796429 alpha=0.75 dimension=3 master={labels} nobutton', '{Sample1} -0.156460708571 0.0682924165714 0.09233683\n{Sample2} -0.0301843435714 0.000174892285714 0.024837603\n{Sample3} 0.0575030878571 -0.181726716857 -0.088353435', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=3 on', '-0.164283744 -0.1908130527 -0.09277110675 white', '0.06037824225 -0.1908130527 -0.09277110675 white', '@labellist {PC1 (25%)} dimension=3 on', '{PC1 (25%)}0.0633971543625 -0.1908130527 -0.09277110675 white', '@vectorlist {PC2 line} dimension=3 on', '-0.164283744 -0.1908130527 -0.09277110675 white', '-0.164283744 0.0717070374 -0.09277110675 white', '@labellist {PC2 (30%)} dimension=3 on', '{PC2 (30%)}-0.164283744 0.07529238927 -0.09277110675 white', '@vectorlist {PC3 line} dimension=3 on', '-0.164283744 -0.1908130527 -0.09277110675 white', '-0.164283744 -0.1908130527 0.0969536715 white', '@labellist {PC3 (35%)} dimension=3 on', '{PC3 (35%)}-0.164283744 -0.1908130527 0.101801355075 white']

exp_kin_full_axes =\
['@kinemage {Day_unscaled}', '@dimension {Height} {Weight} {PC1} {PC2} {PC3}', '@dimminmax -0.219044992 0.161008646 -0.219044992 0.161008646 -0.219044992 0.080504323 -0.212014503 0.079674486 -0.088353435 0.09233683', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Day1 (n=3)} collapsible', '@balllist color=red1 radius=0.00380053638 alpha=0.75 dimension=5 master={points} nobutton', '{Sample1} -0.219044992 0.161008646 -0.219044992 0.079674486 0.09233683\n{Sample2} -0.029018173 -0.029018173 -0.042258081 0.000204041 0.024837603\n{Sample3} 0.161008646 -0.219044992 0.080504323 -0.212014503 -0.088353435', '@labellist color=red1 radius=0.00380053638 alpha=0.75 dimension=5 master={labels} nobutton', '{Sample1} -0.219044992 0.161008646 -0.219044992 0.079674486 0.09233683\n{Sample2} -0.029018173 -0.029018173 -0.042258081 0.000204041 0.024837603\n{Sample3} 0.161008646 -0.219044992 0.080504323 -0.212014503 -0.088353435', '@group {axes} collapsible', '@vectorlist {Height line} dimension=5 on', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '0.1690590783 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '@labellist {Height} dimension=5 on', '{Height}0.177512032215 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '@vectorlist {Weight line} dimension=5 on', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 0.1690590783 -0.2299972416 -0.22261522815 -0.09277110675 white', '@labellist {Weight} dimension=5 on', '{Weight}-0.2299972416 0.177512032215 -0.2299972416 -0.22261522815 -0.09277110675 white', '@vectorlist {PC1 line} dimension=5 on', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.2299972416 0.08452953915 -0.22261522815 -0.09277110675 white', '@labellist {PC1 (25%)} dimension=5 on', '{PC1 (25%)}-0.2299972416 -0.2299972416 0.0887560161075 -0.22261522815 -0.09277110675 white', '@vectorlist {PC2 line} dimension=5 off', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.2299972416 -0.2299972416 0.0836582103 -0.09277110675 white', '@labellist {PC2 (30%)} dimension=5 off', '{PC2 (30%)}-0.2299972416 -0.2299972416 -0.2299972416 0.087841120815 -0.09277110675 white', '@vectorlist {PC3 line} dimension=5 off', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 0.0969536715 white', '@labellist {PC3 (35%)} dimension=5 off', '{PC3 (35%)}-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 0.101801355075 white', '@kinemage {Day_scaled}', '@dimension {Height} {Weight} {PC1} {PC2} {PC3}', '@dimminmax -0.156460708571 0.115006175714 -0.156460708571 0.115006175714 -0.156460708571 0.0575030878571 -0.181726716857 0.0682924165714 -0.088353435 0.09233683', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Day1 (n=3)} collapsible', '@balllist color=red1 radius=0.00271466884286 alpha=0.75 dimension=5 master={points} nobutton', '{Sample1} -0.156460708571 0.115006175714 -0.156460708571 0.0682924165714 0.09233683\n{Sample2} -0.0207272664286 -0.0207272664286 -0.0301843435714 0.000174892285714 0.024837603\n{Sample3} 0.115006175714 -0.156460708571 0.0575030878571 -0.181726716857 -0.088353435', '@labellist color=red1 radius=0.00271466884286 alpha=0.75 dimension=5 master={labels} nobutton', '{Sample1} -0.156460708571 0.115006175714 -0.156460708571 0.0682924165714 0.09233683\n{Sample2} -0.0207272664286 -0.0207272664286 -0.0301843435714 0.000174892285714 0.024837603\n{Sample3} 0.115006175714 -0.156460708571 0.0575030878571 -0.181726716857 -0.088353435', '@group {axes} collapsible', '@vectorlist {Height line} dimension=5 on', '-0.164283744 -0.164283744 -0.164283744 -0.1908130527 -0.09277110675 white', '0.1207564845 -0.164283744 -0.164283744 -0.1908130527 -0.09277110675 white', '@labellist {Height} dimension=5 on', '{Height}0.126794308725 -0.164283744 -0.164283744 -0.1908130527 -0.09277110675 white', '@vectorlist {Weight line} dimension=5 on', '-0.164283744 -0.164283744 -0.164283744 -0.1908130527 -0.09277110675 white', '-0.164283744 0.1207564845 -0.164283744 -0.1908130527 -0.09277110675 white', '@labellist {Weight} dimension=5 on', '{Weight}-0.164283744 0.126794308725 -0.164283744 -0.1908130527 -0.09277110675 white', '@vectorlist {PC1 line} dimension=5 on', '-0.164283744 -0.164283744 -0.164283744 -0.1908130527 -0.09277110675 white', '-0.164283744 -0.164283744 0.06037824225 -0.1908130527 -0.09277110675 white', '@labellist {PC1 (25%)} dimension=5 on', '{PC1 (25%)}-0.164283744 -0.164283744 0.0633971543625 -0.1908130527 -0.09277110675 white', '@vectorlist {PC2 line} dimension=5 off', '-0.164283744 -0.164283744 -0.164283744 -0.1908130527 -0.09277110675 white', '-0.164283744 -0.164283744 -0.164283744 0.0717070374 -0.09277110675 white', '@labellist {PC2 (30%)} dimension=5 off', '{PC2 (30%)}-0.164283744 -0.164283744 -0.164283744 0.07529238927 -0.09277110675 white', '@vectorlist {PC3 line} dimension=5 off', '-0.164283744 -0.164283744 -0.164283744 -0.1908130527 -0.09277110675 white', '-0.164283744 -0.164283744 -0.164283744 -0.1908130527 0.0969536715 white', '@labellist {PC3 (35%)} dimension=5 off', '{PC3 (35%)}-0.164283744 -0.164283744 -0.164283744 -0.1908130527 0.101801355075 white']

exp_kin_vectors=\
['@kinemage {Day_unscaled}', '@dimension {Height} {PC1} {PC2} {PC3}', '@dimminmax 0.0 3.0 -0.219044992 0.080504323 -0.212014503 0.079674486 -160029789.0 0.09233683', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Day1 (n=3)} collapsible', '@balllist color=red1 radius=0.03 alpha=0.75 dimension=4 master={points} nobutton', '{Sample1} 0.0 -0.219044992 0.079674486 0.09233683\n{Sample2} 1.0 -0.042258081 0.000204041 0.024837603\n{Sample3} 2.0 0.080504323 -0.212014503 -0.088353435', '@labellist color=red1 radius=0.03 alpha=0.75 dimension=4 master={labels} nobutton', '{Sample1} 0.0 -0.219044992 0.079674486 0.09233683\n{Sample2} 1.0 -0.042258081 0.000204041 0.024837603\n{Sample3} 2.0 0.080504323 -0.212014503 -0.088353435', '@vectorlist {trace} dimension=4 on', '@vectorlist {average} dimension=4 off', '0.0 -0.219044992 0.079674486 0.09233683 red1', '0.5 -0.1306515365 0.0399392635 0.0585872165 red1 P', '0.5 -0.1306515365 0.0399392635 0.0585872165 red1', '1.0 -0.042258081 0.000204041 0.024837603 red1 P', '1.0 -0.042258081 0.000204041 0.024837603 red1', '1.5 0.019123121 -0.105905231 -0.031757916 red1 P', '1.5 0.019123121 -0.105905231 -0.031757916 red1', '2.0 0.080504323 -0.212014503 -0.088353435 red1 P', '@group {axes} collapsible', '@vectorlist {Height line} dimension=4 on', '0.0 -0.2299972416 -0.22261522815 -168031278.45 white', '3.15 -0.2299972416 -0.22261522815 -168031278.45 white', '@labellist {Height} dimension=4 on', '{Height}3.3075 -0.2299972416 -0.22261522815 -168031278.45 white', '@vectorlist {PC1 line} dimension=4 on', '0.0 -0.2299972416 -0.22261522815 -168031278.45 white', '0.0 0.08452953915 -0.22261522815 -168031278.45 white', '@labellist {PC1 (25%)} dimension=4 on', '{PC1 (25%)}0.0 0.0887560161075 -0.22261522815 -168031278.45 white', '@vectorlist {PC2 line} dimension=4 on', '0.0 -0.2299972416 -0.22261522815 -168031278.45 white', '0.0 -0.2299972416 0.0836582103 -168031278.45 white', '@labellist {PC2 (30%)} dimension=4 on', '{PC2 (30%)}0.0 -0.2299972416 0.087841120815 -168031278.45 white', '@vectorlist {PC3 line} dimension=4 off', '0.0 -0.2299972416 -0.22261522815 -168031278.45 white', '0.0 -0.2299972416 -0.22261522815 0.0969536715 white', '@labellist {PC3 (35%)} dimension=4 off', '{PC3 (35%)}0.0 -0.2299972416 -0.22261522815 0.101801355075 white']

exp_kin_partial=\
['@kinemage {_unscaled}', '@dimension {PC1} {PC2} {PC3}', '@dimminmax -0.219044992 0.080504323 -0.212014503 0.079674486 -0.088353435 0.09233683', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Day1 (n=3)} collapsible', '@balllist color=blue1 radius=0.00299549315 alpha=0.75 dimension=3 master={points} nobutton', '{Sample1} -0.219044992 0.079674486 0.09233683\n{Sample2} -0.042258081 0.000204041 0.024837603\n{Sample3} 0.080504323 -0.212014503 -0.088353435', '@labellist color=blue1 radius=0.00299549315 alpha=0.75 dimension=3 master={labels} nobutton', '{Sample1} -0.219044992 0.079674486 0.09233683\n{Sample2} -0.042258081 0.000204041 0.024837603\n{Sample3} 0.080504323 -0.212014503 -0.088353435', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=3 on', '-0.2299972416 -0.22261522815 -0.09277110675 white', '0.08452953915 -0.22261522815 -0.09277110675 white', '@labellist {PC1 (25%)} dimension=3 on', '{PC1 (25%)}0.0887560161075 -0.22261522815 -0.09277110675 white', '@vectorlist {PC2 line} dimension=3 on', '-0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 0.0836582103 -0.09277110675 white', '@labellist {PC2 (30%)} dimension=3 on', '{PC2 (30%)}-0.2299972416 0.087841120815 -0.09277110675 white', '@vectorlist {PC3 line} dimension=3 on', '-0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.22261522815 0.0969536715 white', '@labellist {PC3 (35%)} dimension=3 on', '{PC3 (35%)}-0.2299972416 -0.22261522815 0.101801355075 white']

exp_kin_vectors_special_a =\
['@kinemage {Description_unscaled}', '@dimension {Time} {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.560798235 -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Control_mouse_I.D._354 (n=1)} collapsible', '@balllist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@labellist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=10 off', '@group {Control_mouse_I.D._355 (n=1)} collapsible', '@balllist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@labellist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 blue1', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1 P', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=10 off', '@group {Control_mouse_I.D._356 (n=1)} collapsible', '@balllist color=orange1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@labellist color=orange1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 orange1', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1 P', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 orange1 P', '@vectorlist {average} dimension=10 off', '@group {Control_mouse_I.D._481 (n=1)} collapsible', '@balllist color=green1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@labellist color=green1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 green1', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1 P', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1', '0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 green1 P', '@vectorlist {average} dimension=10 off', '@group {Control_mouse_I.D._593 (n=1)} collapsible', '@balllist color=purple1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@labellist color=purple1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '@vectorlist {average} dimension=10 off', '@group {Fasting_mouse_I.D._607 (n=1)} collapsible', '@balllist color=yellow1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@labellist color=yellow1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 yellow1', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1 P', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 yellow1 P', '@vectorlist {average} dimension=10 off', '@group {Fasting_mouse_I.D._634 (n=1)} collapsible', '@balllist color=cyan1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@labellist color=cyan1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 cyan1', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1 P', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 cyan1 P', '@vectorlist {average} dimension=10 off', '@group {Fasting_mouse_I.D._635 (n=1)} collapsible', '@balllist color=pink1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@labellist color=pink1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 pink1', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1 P', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1', '0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 pink1 P', '@vectorlist {average} dimension=10 off', '@group {Fasting_mouse_I.D._636 (n=1)} collapsible', '@balllist color=teal1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=teal1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '@vectorlist {average} dimension=10 off', '@group {axes} collapsible', '@vectorlist {Time line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.58883814675 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {Time} dimension=10 on', '{Time}0.618280054088 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC1 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=10 on', '{PC1 (27%)}-0.3668061894 0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=10 on', '{PC2 (16%)}-0.3668061894 -0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=10 off', '{PC3 (14%)}-0.3668061894 -0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=10 off', '{PC4 (11%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=10 off', '{PC5 (10%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=10 off', '{PC6 (8.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=10 off', '{PC7 (7.6%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=10 off', '{PC8 (6.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=10 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {Group_unscaled}', '@dimension {Time} {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.560798235 -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {A (n=9)} collapsible', '@balllist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1 P', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1', '0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 red1 P', '-0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 red1', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 red1 P', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 red1', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1 P', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1 P', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1 P', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 red1 P', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 red1', '0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=10 off', '-0.349339228 0.0945345948 0.2090671597 -0.0560709933 -0.0521652142 0.0023695205 -0.009861888175 0.0081503259 -0.1472620058 -4.84141987e-09 red1', '-0.2843294095 0.1325260884 0.00747694535 -0.0434837608 -0.01617829915 0.07873479175 -0.144892915087 0.03287978875 -0.0615069098 -4.84141987e-09 red1 P', '-0.2843294095 0.1325260884 0.00747694535 -0.0434837608 -0.01617829915 0.07873479175 -0.144892915087 0.03287978875 -0.0615069098 -4.84141987e-09 red1', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.008214775 -0.060259414 -0.1254648435 -0.085937473 0.077972639 0.0094614868 0.13694790445 0.13772399605 0.00605463655 -4.84141987e-09 red1 P', '0.008214775 -0.060259414 -0.1254648435 -0.085937473 0.077972639 0.0094614868 0.13694790445 0.13772399605 0.00605463655 -4.84141987e-09 red1', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1 P', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1 P', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1 P', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1', '0.3982736885 -0.09769927295 0.01604231305 -0.0373021767 -0.0935297635 -0.1876492535 -0.05063819795 -0.04142167215 0.102470961 -4.84141987e-09 red1 P', '0.3982736885 -0.09769927295 0.01604231305 -0.0373021767 -0.0935297635 -0.1876492535 -0.05063819795 -0.04142167215 0.102470961 -4.84141987e-09 red1', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.4957884165 0.0102140748 -0.0082780731 0.1291451844 -0.1441512606 -0.014772796575 0.03872461225 -0.05161829345 0.0833800821 -4.84141987e-09 red1 P', '0.4957884165 0.0102140748 -0.0082780731 0.1291451844 -0.1441512606 -0.014772796575 0.03872461225 -0.05161829345 0.0833800821 -4.84141987e-09 red1', '0.560798235 -0.0218346985 -0.002587995 0.1947589072 0.0578180308 0.09826821485 0.0635141524 -0.1332572756 0.0266123152 -4.84141987e-09 red1 P', '@group {axes} collapsible', '@vectorlist {Time line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.58883814675 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {Time} dimension=10 on', '{Time}0.618280054088 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC1 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=10 on', '{PC1 (27%)}-0.3668061894 0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=10 on', '{PC2 (16%)}-0.3668061894 -0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=10 off', '{PC3 (14%)}-0.3668061894 -0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=10 off', '{PC4 (11%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=10 off', '{PC5 (10%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=10 off', '{PC6 (8.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=10 off', '{PC7 (7.6%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=10 off', '{PC8 (6.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=10 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {SampleID_unscaled}', '@dimension {Time} {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.560798235 -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {PC.354 (n=1)} collapsible', '@balllist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@labellist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=10 off', '@group {PC.355 (n=1)} collapsible', '@balllist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@labellist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 blue1', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1 P', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=10 off', '@group {PC.356 (n=1)} collapsible', '@balllist color=orange1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@labellist color=orange1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 orange1', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1 P', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 orange1 P', '@vectorlist {average} dimension=10 off', '@group {PC.481 (n=1)} collapsible', '@balllist color=green1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@labellist color=green1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 green1', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1 P', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1', '0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 green1 P', '@vectorlist {average} dimension=10 off', '@group {PC.593 (n=1)} collapsible', '@balllist color=purple1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@labellist color=purple1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '@vectorlist {average} dimension=10 off', '@group {PC.607 (n=1)} collapsible', '@balllist color=yellow1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@labellist color=yellow1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 yellow1', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1 P', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 yellow1 P', '@vectorlist {average} dimension=10 off', '@group {PC.634 (n=1)} collapsible', '@balllist color=cyan1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@labellist color=cyan1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 cyan1', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1 P', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 cyan1 P', '@vectorlist {average} dimension=10 off', '@group {PC.635 (n=1)} collapsible', '@balllist color=pink1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@labellist color=pink1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 pink1', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1 P', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1', '0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 pink1 P', '@vectorlist {average} dimension=10 off', '@group {PC.636 (n=1)} collapsible', '@balllist color=teal1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=teal1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '@vectorlist {average} dimension=10 off', '@group {axes} collapsible', '@vectorlist {Time line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.58883814675 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {Time} dimension=10 on', '{Time}0.618280054088 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC1 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=10 on', '{PC1 (27%)}-0.3668061894 0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=10 on', '{PC2 (16%)}-0.3668061894 -0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=10 off', '{PC3 (14%)}-0.3668061894 -0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=10 off', '{PC4 (11%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=10 off', '{PC5 (10%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=10 off', '{PC6 (8.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=10 off', '{PC7 (7.6%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=10 off', '{PC8 (6.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=10 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {Subject_unscaled}', '@dimension {Time} {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.560798235 -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Subject1 (n=5)} collapsible', '@balllist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@labellist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1 P', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1', '0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=10 off', '-0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1 P', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1', '0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 red1 P', '@group {Subject2 (n=4)} collapsible', '@balllist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 blue1', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1 P', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1 P', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1 P', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1 P', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1 P', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1', '0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=10 off', '-0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 blue1', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1 P', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1 P', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1 P', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1 P', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1 P', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1', '0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 blue1 P', '@group {axes} collapsible', '@vectorlist {Time line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.58883814675 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {Time} dimension=10 on', '{Time}0.618280054088 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC1 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=10 on', '{PC1 (27%)}-0.3668061894 0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=10 on', '{PC2 (16%)}-0.3668061894 -0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=10 off', '{PC3 (14%)}-0.3668061894 -0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=10 off', '{PC4 (11%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=10 off', '{PC5 (10%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=10 off', '{PC6 (8.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=10 off', '{PC7 (7.6%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=10 off', '{PC8 (6.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=10 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {Time_unscaled}', '@dimension {Time} {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.560798235 -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {0 (n=1)} collapsible', '@balllist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@labellist color=red1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.356} -0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.1707393225 0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=10 off', '@group {1.5 (n=1)} collapsible', '@balllist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@labellist color=blue1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.634} 0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1 P', '0.235749141 -0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=10 off', '@group {3.5 (n=1)} collapsible', '@balllist color=orange1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@labellist color=orange1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.635} 0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.365768779 -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 orange1', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 orange1 P', '0.463283507 -0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 orange1', '0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 orange1 P', '@vectorlist {average} dimension=10 off', '@group {4 (n=1)} collapsible', '@balllist color=green1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@labellist color=green1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.481} 0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '0.430778598 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 green1', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1 P', '0.4957884165 0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1', '0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 green1 P', '@vectorlist {average} dimension=10 off', '@group {5 (n=2)} collapsible', '@balllist color=purple1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=purple1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.593} 0.560798235 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.636} 0.560798235 -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '@vectorlist {average} dimension=10 off', '@group {-1 (n=1)} collapsible', '@balllist color=yellow1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@labellist color=yellow1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.355} -0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 yellow1', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 yellow1 P', '-0.154309772 0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 yellow1', '-0.089299953 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 yellow1 P', '@vectorlist {average} dimension=10 off', '@group {-2 (n=2)} collapsible', '@balllist color=cyan1 radius=0.00910137463 alpha=0.75 dimension=10 master={points} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@labellist color=cyan1 radius=0.00910137463 alpha=0.75 dimension=10 master={labels} nobutton', '{PC.354} -0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.607} -0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@vectorlist {trace} dimension=10 on', '-0.349339228 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 cyan1', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 cyan1 P', '-0.2843294095 0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 cyan1', '-0.219319591 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 cyan1 P', '-0.349339228 -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 cyan1', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 cyan1 P', '-0.1218048625 -0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 cyan1', '0.105729503 -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 cyan1 P', '@vectorlist {average} dimension=10 off', '@group {axes} collapsible', '@vectorlist {Time line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.58883814675 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {Time} dimension=10 on', '{Time}0.618280054088 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC1 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=10 on', '{PC1 (27%)}-0.3668061894 0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=10 on', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=10 on', '{PC2 (16%)}-0.3668061894 -0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=10 off', '{PC3 (14%)}-0.3668061894 -0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=10 off', '{PC4 (11%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=10 off', '{PC5 (10%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=10 off', '{PC6 (8.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=10 off', '{PC7 (7.6%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=10 off', '{PC8 (6.2%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=10 off', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=10 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white']

exp_kin_vectors_special_b =\
['@kinemage {Description_unscaled}', '@dimension {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Control_mouse_I.D._354 (n=1)} collapsible', '@balllist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@labellist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=9 off', '@group {Control_mouse_I.D._355 (n=1)} collapsible', '@balllist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@labellist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 blue1', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1 P', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=9 off', '@group {Control_mouse_I.D._356 (n=1)} collapsible', '@balllist color=orange1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@labellist color=orange1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 orange1', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1 P', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 orange1 P', '@vectorlist {average} dimension=9 off', '@group {Control_mouse_I.D._481 (n=1)} collapsible', '@balllist color=green1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@labellist color=green1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 green1', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1 P', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1', '0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 green1 P', '@vectorlist {average} dimension=9 off', '@group {Control_mouse_I.D._593 (n=1)} collapsible', '@balllist color=purple1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@labellist color=purple1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '@vectorlist {average} dimension=9 off', '@group {Fasting_mouse_I.D._607 (n=1)} collapsible', '@balllist color=yellow1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@labellist color=yellow1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 yellow1', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1 P', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 yellow1 P', '@vectorlist {average} dimension=9 off', '@group {Fasting_mouse_I.D._634 (n=1)} collapsible', '@balllist color=cyan1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@labellist color=cyan1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 cyan1', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1 P', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 cyan1 P', '@vectorlist {average} dimension=9 off', '@group {Fasting_mouse_I.D._635 (n=1)} collapsible', '@balllist color=pink1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@labellist color=pink1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 pink1', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1 P', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1', '-0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 pink1 P', '@vectorlist {average} dimension=9 off', '@group {Fasting_mouse_I.D._636 (n=1)} collapsible', '@balllist color=teal1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=teal1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '@vectorlist {average} dimension=9 off', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=9 on', '{PC1 (27%)}0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=9 on', '{PC2 (16%)}-0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=9 on', '{PC3 (14%)}-0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=9 off', '{PC4 (11%)}-0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=9 off', '{PC5 (10%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=9 off', '{PC6 (8.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=9 off', '{PC7 (7.6%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=9 off', '{PC8 (6.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=9 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {Group_unscaled}', '@dimension {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {A (n=9)} collapsible', '@balllist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1 P', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1', '0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 red1 P', '-0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 red1', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 red1 P', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 red1', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1 P', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1 P', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1 P', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 red1 P', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 red1', '-0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=9 off', '0.0945345948 0.2090671597 -0.0560709933 -0.0521652142 0.0023695205 -0.009861888175 0.0081503259 -0.1472620058 -4.84141987e-09 red1', '0.1325260884 0.00747694535 -0.0434837608 -0.01617829915 0.07873479175 -0.144892915087 0.03287978875 -0.0615069098 -4.84141987e-09 red1 P', '0.1325260884 0.00747694535 -0.0434837608 -0.01617829915 0.07873479175 -0.144892915087 0.03287978875 -0.0615069098 -4.84141987e-09 red1', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '-0.060259414 -0.1254648435 -0.085937473 0.077972639 0.0094614868 0.13694790445 0.13772399605 0.00605463655 -4.84141987e-09 red1 P', '-0.060259414 -0.1254648435 -0.085937473 0.077972639 0.0094614868 0.13694790445 0.13772399605 0.00605463655 -4.84141987e-09 red1', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1 P', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 red1', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1 P', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 red1', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1 P', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 red1', '-0.09769927295 0.01604231305 -0.0373021767 -0.0935297635 -0.1876492535 -0.05063819795 -0.04142167215 0.102470961 -4.84141987e-09 red1 P', '-0.09769927295 0.01604231305 -0.0373021767 -0.0935297635 -0.1876492535 -0.05063819795 -0.04142167215 0.102470961 -4.84141987e-09 red1', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.0102140748 -0.0082780731 0.1291451844 -0.1441512606 -0.014772796575 0.03872461225 -0.05161829345 0.0833800821 -4.84141987e-09 red1 P', '0.0102140748 -0.0082780731 0.1291451844 -0.1441512606 -0.014772796575 0.03872461225 -0.05161829345 0.0833800821 -4.84141987e-09 red1', '-0.0218346985 -0.002587995 0.1947589072 0.0578180308 0.09826821485 0.0635141524 -0.1332572756 0.0266123152 -4.84141987e-09 red1 P', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=9 on', '{PC1 (27%)}0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=9 on', '{PC2 (16%)}-0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=9 on', '{PC3 (14%)}-0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=9 off', '{PC4 (11%)}-0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=9 off', '{PC5 (10%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=9 off', '{PC6 (8.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=9 off', '{PC7 (7.6%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=9 off', '{PC8 (6.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=9 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {SampleID_unscaled}', '@dimension {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {PC.354 (n=1)} collapsible', '@balllist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@labellist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=9 off', '@group {PC.355 (n=1)} collapsible', '@balllist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@labellist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 blue1', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1 P', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 blue1', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=9 off', '@group {PC.356 (n=1)} collapsible', '@balllist color=orange1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@labellist color=orange1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 orange1', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1 P', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 orange1', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 orange1 P', '@vectorlist {average} dimension=9 off', '@group {PC.481 (n=1)} collapsible', '@balllist color=green1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@labellist color=green1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 green1', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1 P', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1', '0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 green1 P', '@vectorlist {average} dimension=9 off', '@group {PC.593 (n=1)} collapsible', '@balllist color=purple1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@labellist color=purple1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '@vectorlist {average} dimension=9 off', '@group {PC.607 (n=1)} collapsible', '@balllist color=yellow1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@labellist color=yellow1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 yellow1', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1 P', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 yellow1', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 yellow1 P', '@vectorlist {average} dimension=9 off', '@group {PC.634 (n=1)} collapsible', '@balllist color=cyan1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@labellist color=cyan1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 cyan1', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1 P', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 cyan1', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 cyan1 P', '@vectorlist {average} dimension=9 off', '@group {PC.635 (n=1)} collapsible', '@balllist color=pink1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@labellist color=pink1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 pink1', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1 P', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 pink1', '-0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 pink1 P', '@vectorlist {average} dimension=9 off', '@group {PC.636 (n=1)} collapsible', '@balllist color=teal1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=teal1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '@vectorlist {average} dimension=9 off', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=9 on', '{PC1 (27%)}0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=9 on', '{PC2 (16%)}-0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=9 on', '{PC3 (14%)}-0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=9 off', '{PC4 (11%)}-0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=9 off', '{PC5 (10%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=9 off', '{PC6 (8.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=9 off', '{PC7 (7.6%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=9 off', '{PC8 (6.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=9 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {Subject_unscaled}', '@dimension {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Subject1 (n=5)} collapsible', '@balllist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@labellist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09\n{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09\n{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09\n{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1 P', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1', '0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=9 off', '0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 red1', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1 P', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 red1', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1 P', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 red1', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1 P', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 red1', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1 P', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1 P', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 red1', '0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 red1 P', '@group {Subject2 (n=4)} collapsible', '@balllist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09\n{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09\n{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09\n{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 blue1', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1 P', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1 P', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1 P', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1 P', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1 P', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1', '-0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=9 off', '-0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 blue1', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1 P', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 blue1', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1 P', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1 P', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1 P', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1 P', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 blue1', '-0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 blue1 P', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=9 on', '{PC1 (27%)}0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=9 on', '{PC2 (16%)}-0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=9 on', '{PC3 (14%)}-0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=9 off', '{PC4 (11%)}-0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=9 off', '{PC5 (10%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=9 off', '{PC6 (8.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=9 off', '{PC7 (7.6%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=9 off', '{PC8 (6.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=9 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white', '@kinemage {Time_unscaled}', '@dimension {PC1} {PC2} {PC3} {PC4} {PC5} {PC6} {PC7} {PC8} {PC9}', '@dimminmax -0.349339228 0.280399118 -0.194113269 0.424147148 -0.287149448 0.32287108 -0.346120552 0.183347007 -0.247484699 0.17607027 -0.279923942 0.206042607 -0.229889464 0.244447635 -0.255785794 0.140147849 -4.84141987e-09 -4.84141987e-09', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {0 (n=1)} collapsible', '@balllist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@labellist color=red1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.356} 0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 red1', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1 P', '0.13554162405 -0.0720551241 -0.1118089932 -0.1298353337 -0.04175930805 0.10998883955 0.0305105229 0.10606994635 -4.84141987e-09 red1', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 red1 P', '@vectorlist {average} dimension=9 off', '@group {1.5 (n=1)} collapsible', '@balllist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@labellist color=blue1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.634} -0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 blue1', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1 P', '-0.293500311 -0.03736740635 -0.0114306565 0.1142782092 -0.13642845865 -0.02367913305 0.065791801 0.0024556512 -4.84141987e-09 blue1', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 blue1 P', '@vectorlist {average} dimension=9 off', '@group {3.5 (n=1)} collapsible', '@balllist color=orange1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@labellist color=orange1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.635} -0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '-0.237661394 0.0460527773 -0.138135815 0.159061025 -0.247484699 -0.115211468 -0.112864033 0.064794073 -4.84141987e-09 orange1', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 orange1 P', '-0.257101779 -0.04945579885 -0.0357445403 0.0456750398 -0.0357072145 -0.02112103895 -0.1713767485 0.00909756565 -4.84141987e-09 orange1', '-0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09 orange1 P', '@vectorlist {average} dimension=9 off', '@group {4 (n=1)} collapsible', '@balllist color=green1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@labellist color=green1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.481} 0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.0422628481 -0.0139681512 0.0635314616 -0.346120552 -0.127813808 0.0139350721 0.0300206887 0.140147849 -4.84141987e-09 green1', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1 P', '0.13756780755 0.0629101169 0.1932012708 -0.0813867725 -0.05367382415 0.0339969934 -0.00330219925 0.11998571055 -4.84141987e-09 green1', '0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09 green1 P', '@vectorlist {average} dimension=9 off', '@group {5 (n=2)} collapsible', '@balllist color=purple1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@labellist color=purple1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.593} 0.232872767 0.139788385 0.32287108 0.183347007 0.0204661597 0.0540589147 -0.0366250872 0.0998235721 -4.84141987e-09\n{PC.636} -0.276542164 -0.144964375 0.0666467344 -0.0677109454 0.17607027 0.0729693901 -0.229889464 -0.0465989417 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '@vectorlist {average} dimension=9 off', '@group {-1 (n=1)} collapsible', '@balllist color=yellow1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@labellist color=yellow1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.355} 0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 yellow1', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 yellow1 P', '0.199668991 -0.162127683 -0.15902298815 0.05312925025 0.09969762745 -0.0369406675 0.04430480435 0.04812011495 -4.84141987e-09 yellow1', '0.2288204 -0.130142097 -0.287149448 0.0864498846 0.0442951919 0.206042607 0.0310003571 0.0719920437 -4.84141987e-09 yellow1 P', '@vectorlist {average} dimension=9 off', '@group {-2 (n=2)} collapsible', '@balllist color=cyan1 radius=0.00629738346 alpha=0.75 dimension=9 master={points} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@labellist color=cyan1 radius=0.00629738346 alpha=0.75 dimension=9 master={labels} nobutton', '{PC.354} 0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09\n{PC.607} -0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09', '@vectorlist {trace} dimension=9 on', '0.280399118 -0.0060128286 0.0234854344 -0.0468109475 -0.14662445 0.00566979125 -0.0354299634 -0.255785794 -4.84141987e-09 cyan1', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 cyan1 P', '0.22545835 -0.1000630488 -0.00370554695 -0.0135011658 0.0042378065 -0.137127075375 0.0110896441 -0.1157688039 -4.84141987e-09 cyan1', '0.170517582 -0.194113269 -0.0308965283 0.0198086159 0.155100063 -0.279923942 0.0576092516 0.0242481862 -4.84141987e-09 cyan1 P', '-0.0913299284 0.424147148 -0.135627421 -0.0575194809 0.151363491 -0.0253935676 0.0517306152 -0.0387382176 -4.84141987e-09 cyan1', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 cyan1 P', '-0.2203345782 0.151679779 -0.0101764595 0.00598795625 0.06299563635 0.02122981715 0.1480891251 -0.0493104941 -4.84141987e-09 cyan1', '-0.349339228 -0.12078759 0.115274502 0.0694953934 -0.0253722183 0.0678532019 0.244447635 -0.0598827706 -4.84141987e-09 cyan1 P', '@vectorlist {average} dimension=9 off', '@group {axes} collapsible', '@vectorlist {PC1 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '0.2944190739 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC1 (27%)} dimension=9 on', '{PC1 (27%)}0.309140027595 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC2 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 0.4453545054 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC2 (16%)} dimension=9 on', '{PC2 (16%)}-0.3668061894 0.46762223067 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC3 line} dimension=9 on', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 0.339014634 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC3 (14%)} dimension=9 on', '{PC3 (14%)}-0.3668061894 -0.20381893245 0.3559653657 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC4 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 0.19251435735 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC4 (11%)} dimension=9 off', '{PC4 (11%)}-0.3668061894 -0.20381893245 -0.3015069204 0.202140075218 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC5 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.1848737835 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC5 (10%)} dimension=9 off', '{PC5 (10%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 0.194117472675 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC6 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.21634473735 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC6 (8.2%)} dimension=9 off', '{PC6 (8.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 0.227161974218 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC7 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.25667001675 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC7 (7.6%)} dimension=9 off', '{PC7 (7.6%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 0.269503517588 -0.2685750837 -5.0834908635e-09 white', '@vectorlist {PC8 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.14715524145 -5.0834908635e-09 white', '@labellist {PC8 (6.2%)} dimension=9 off', '{PC8 (6.2%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 0.154513003522 -5.0834908635e-09 white', '@vectorlist {PC9 line} dimension=9 off', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.0834908635e-09 white', '@labellist {PC9 (1.2e-14%)} dimension=9 off', '{PC9 (1.2e-14%)}-0.3668061894 -0.20381893245 -0.3015069204 -0.3634265796 -0.25985893395 -0.2939201391 -0.2413839372 -0.2685750837 -5.33766540667e-09 white']

exp_kin_partial_axes=\
['@kinemage {_unscaled}', '@dimension {Height} {Weight} {PC1} {PC2} {PC3}', '@dimminmax -0.219044992 0.161008646 -0.219044992 0.161008646 -0.219044992 0.080504323 -0.212014503 0.079674486 -0.088353435 0.09233683', '@master {points}', '@master {labels}', '@hsvcolor {blue1} 240.0 100.0 100.0', '@hsvcolor {blue2} 211.0 42.0 85.0', '@hsvcolor {blue3} 197.0 100.0 100.0', '@hsvcolor {brown1} 36.0 89.0 42.0', '@hsvcolor {brown2} 33.0 45.0 77.0', '@hsvcolor {cyan1} 184.0 49.0 96.0', '@hsvcolor {gray1} 0.0 0.0 50.2', '@hsvcolor {gray2} 0.0 0.0 75.3', '@hsvcolor {green1} 120.0 100.0 50.2', '@hsvcolor {green2} 142.0 36.0 79.0', '@hsvcolor {green3} 60.0 100.0 50.2', '@hsvcolor {green4} 81.0 100.0 26.0', '@hsvcolor {lime} 123.0 99.0 96.0', '@hsvcolor {orange1} 28.0 98.0 95.0', '@hsvcolor {orange2} 32.0 46.0 99.0', '@hsvcolor {orange3} 26.0 100.0 65.0', '@hsvcolor {pink1} 333.0 37.0 96.0', '@hsvcolor {purple1} 302.0 73.0 57.0', '@hsvcolor {purple2} 269.0 29.0 75.0', '@hsvcolor {purple4} 264.0 75.0 100.0', '@hsvcolor {red1} 0.0 100.0 100.0', '@hsvcolor {red2} 14.0 51.0 97.0', '@hsvcolor {red3} 325.0 100.0 93.0', '@hsvcolor {red4} 348.0 31.0 74.0', '@hsvcolor {red5} 0.0 100.0 50.2', '@hsvcolor {teal1} 178.0 42.0 63.0', '@hsvcolor {teal3} 180.0 100.0 50.2', '@hsvcolor {yellow1} 60.0 100.0 100.0', '@hsvcolor {yellow2} 56.0 40.0 100.0', '@hsvcolor {white} 180.0 0.0 100.0', '@group {Day1 (n=3)} collapsible', '@balllist color=blue1 radius=0.00380053638 alpha=0.75 dimension=5 master={points} nobutton', '{Sample1} -0.219044992 0.161008646 -0.219044992 0.079674486 0.09233683\n{Sample2} -0.029018173 -0.029018173 -0.042258081 0.000204041 0.024837603\n{Sample3} 0.161008646 -0.219044992 0.080504323 -0.212014503 -0.088353435', '@labellist color=blue1 radius=0.00380053638 alpha=0.75 dimension=5 master={labels} nobutton', '{Sample1} -0.219044992 0.161008646 -0.219044992 0.079674486 0.09233683\n{Sample2} -0.029018173 -0.029018173 -0.042258081 0.000204041 0.024837603\n{Sample3} 0.161008646 -0.219044992 0.080504323 -0.212014503 -0.088353435', '@group {axes} collapsible', '@vectorlist {Height line} dimension=5 on', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '0.1690590783 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '@labellist {Height} dimension=5 on', '{Height}0.177512032215 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '@vectorlist {Weight line} dimension=5 on', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 0.1690590783 -0.2299972416 -0.22261522815 -0.09277110675 white', '@labellist {Weight} dimension=5 on', '{Weight}-0.2299972416 0.177512032215 -0.2299972416 -0.22261522815 -0.09277110675 white', '@vectorlist {PC1 line} dimension=5 on', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.2299972416 0.08452953915 -0.22261522815 -0.09277110675 white', '@labellist {PC1 (25%)} dimension=5 on', '{PC1 (25%)}-0.2299972416 -0.2299972416 0.0887560161075 -0.22261522815 -0.09277110675 white', '@vectorlist {PC2 line} dimension=5 off', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.2299972416 -0.2299972416 0.0836582103 -0.09277110675 white', '@labellist {PC2 (30%)} dimension=5 off', '{PC2 (30%)}-0.2299972416 -0.2299972416 -0.2299972416 0.087841120815 -0.09277110675 white', '@vectorlist {PC3 line} dimension=5 off', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 -0.09277110675 white', '-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 0.0969536715 white', '@labellist {PC3 (35%)} dimension=5 off', '{PC3 (35%)}-0.2299972416 -0.2299972416 -0.2299972416 -0.22261522815 0.101801355075 white']
#expected ellipsoid faces with no translation/scaling
exp_make_ellipsoid_faces = \
[[(0.85065000000000002, 0.52573000000000003, 0.0), (0.0, 0.85065000000000002, 0.52573000000000003), (0.52573000000000003, 0.0, 0.85065000000000002)], [(-0.85065000000000002, 0.52573000000000003, 0.0), (0.0, 0.85065000000000002, -0.52573000000000003), (-0.52573000000000003, 0.0, -0.85065000000000002)], [(0.85065000000000002, -0.52573000000000003, 0.0), (0.0, -0.85065000000000002, 0.52573000000000003), (0.0, -0.85065000000000002, -0.52573000000000003)], [(-0.52573000000000003, 0.0, -0.85065000000000002), (-0.85065000000000002, -0.52573000000000003, 0.0), (-0.85065000000000002, 0.52573000000000003, 0.0)], [(0.85065000000000002, 0.52573000000000003, 0.0), (0.52573000000000003, 0.0, -0.85065000000000002), (0.0, 0.85065000000000002, -0.52573000000000003)], [(-0.85065000000000002, -0.52573000000000003, 0.0), (0.0, -0.85065000000000002, 0.52573000000000003), (-0.52573000000000003, 0.0, 0.85065000000000002)], [(-0.85065000000000002, -0.52573000000000003, 0.0), (0.0, -0.85065000000000002, -0.52573000000000003), (0.0, -0.85065000000000002, 0.52573000000000003)], [(0.0, 0.85065000000000002, 0.52573000000000003), (-0.52573000000000003, 0.0, 0.85065000000000002), (0.52573000000000003, 0.0, 0.85065000000000002)], [(0.85065000000000002, -0.52573000000000003, 0.0), (0.52573000000000003, 0.0, 0.85065000000000002), (0.0, -0.85065000000000002, 0.52573000000000003)], [(-0.85065000000000002, -0.52573000000000003, 0.0), (-0.52573000000000003, 0.0, -0.85065000000000002), (0.0, -0.85065000000000002, -0.52573000000000003)], [(0.52573000000000003, 0.0, 0.85065000000000002), (0.85065000000000002, -0.52573000000000003, 0.0), (0.85065000000000002, 0.52573000000000003, 0.0)], [(0.0, -0.85065000000000002, 0.52573000000000003), (0.52573000000000003, 0.0, 0.85065000000000002), (-0.52573000000000003, 0.0, 0.85065000000000002)], [(0.85065000000000002, -0.52573000000000003, 0.0), (0.0, -0.85065000000000002, -0.52573000000000003), (0.52573000000000003, 0.0, -0.85065000000000002)], [(0.85065000000000002, 0.52573000000000003, 0.0), (0.0, 0.85065000000000002, -0.52573000000000003), (0.0, 0.85065000000000002, 0.52573000000000003)], [(0.52573000000000003, 0.0, -0.85065000000000002), (0.85065000000000002, 0.52573000000000003, 0.0), (0.85065000000000002, -0.52573000000000003, 0.0)], [(0.0, 0.85065000000000002, -0.52573000000000003), (0.52573000000000003, 0.0, -0.85065000000000002), (-0.52573000000000003, 0.0, -0.85065000000000002)], [(-0.85065000000000002, 0.52573000000000003, 0.0), (-0.52573000000000003, 0.0, 0.85065000000000002), (0.0, 0.85065000000000002, 0.52573000000000003)], [(-0.85065000000000002, 0.52573000000000003, 0.0), (0.0, 0.85065000000000002, 0.52573000000000003), (0.0, 0.85065000000000002, -0.52573000000000003)], [(-0.52573000000000003, 0.0, 0.85065000000000002), (-0.85065000000000002, 0.52573000000000003, 0.0), (-0.85065000000000002, -0.52573000000000003, 0.0)], [(0.0, -0.85065000000000002, -0.52573000000000003), (-0.52573000000000003, 0.0, -0.85065000000000002), (0.52573000000000003, 0.0, -0.85065000000000002)]]

#expected ellipsoid faces with translation by (-1,2,3), scaling by (.2,2,20)
exp_make_ellipsoid_faces_transformed = \
[[(-0.82986983832959194, 3.051462224238267, 3.0), (-1.0, 3.7013016167040798, 13.514622242382671), (-0.89485377757617324, 2.0, 20.013016167040799)], [(-1.1701301616704081, 3.051462224238267, 3.0), (-1.0, 3.7013016167040798, -7.5146222423826714), (-1.1051462224238267, 2.0, -14.013016167040799)], [(-0.82986983832959194, 0.94853777576173282, 3.0), (-1.0, 0.29869838329592002, 13.514622242382671), (-1.0, 0.29869838329592002, -7.5146222423826714)], [(-1.1051462224238267, 2.0, -14.013016167040799), (-1.1701301616704081, 0.94853777576173282, 3.0), (-1.1701301616704081, 3.051462224238267, 3.0)], [(-0.82986983832959194, 3.051462224238267, 3.0), (-0.89485377757617324, 2.0, -14.013016167040799), (-1.0, 3.7013016167040798, -7.5146222423826714)], [(-1.1701301616704081, 0.94853777576173282, 3.0), (-1.0, 0.29869838329592002, 13.514622242382671), (-1.1051462224238267, 2.0, 20.013016167040799)], [(-1.1701301616704081, 0.94853777576173282, 3.0), (-1.0, 0.29869838329592002, -7.5146222423826714), (-1.0, 0.29869838329592002, 13.514622242382671)], [(-1.0, 3.7013016167040798, 13.514622242382671), (-1.1051462224238267, 2.0, 20.013016167040799), (-0.89485377757617324, 2.0, 20.013016167040799)], [(-0.82986983832959194, 0.94853777576173282, 3.0), (-0.89485377757617324, 2.0, 20.013016167040799), (-1.0, 0.29869838329592002, 13.514622242382671)], [(-1.1701301616704081, 0.94853777576173282, 3.0), (-1.1051462224238267, 2.0, -14.013016167040799), (-1.0, 0.29869838329592002, -7.5146222423826714)], [(-0.89485377757617324, 2.0, 20.013016167040799), (-0.82986983832959194, 0.94853777576173282, 3.0), (-0.82986983832959194, 3.051462224238267, 3.0)], [(-1.0, 0.29869838329592002, 13.514622242382671), (-0.89485377757617324, 2.0, 20.013016167040799), (-1.1051462224238267, 2.0, 20.013016167040799)], [(-0.82986983832959194, 0.94853777576173282, 3.0), (-1.0, 0.29869838329592002, -7.5146222423826714), (-0.89485377757617324, 2.0, -14.013016167040799)], [(-0.82986983832959194, 3.051462224238267, 3.0), (-1.0, 3.7013016167040798, -7.5146222423826714), (-1.0, 3.7013016167040798, 13.514622242382671)], [(-0.89485377757617324, 2.0, -14.013016167040799), (-0.82986983832959194, 3.051462224238267, 3.0), (-0.82986983832959194, 0.94853777576173282, 3.0)], [(-1.0, 3.7013016167040798, -7.5146222423826714), (-0.89485377757617324, 2.0, -14.013016167040799), (-1.1051462224238267, 2.0, -14.013016167040799)], [(-1.1701301616704081, 3.051462224238267, 3.0), (-1.1051462224238267, 2.0, 20.013016167040799), (-1.0, 3.7013016167040798, 13.514622242382671)], [(-1.1701301616704081, 3.051462224238267, 3.0), (-1.0, 3.7013016167040798, 13.514622242382671), (-1.0, 3.7013016167040798, -7.5146222423826714)], [(-1.1051462224238267, 2.0, 20.013016167040799), (-1.1701301616704081, 3.051462224238267, 3.0), (-1.1701301616704081, 0.94853777576173282, 3.0)], [(-1.0, 0.29869838329592002, -7.5146222423826714), (-1.1051462224238267, 2.0, -14.013016167040799), (-0.89485377757617324, 2.0, -14.013016167040799)]]

exp_make_mage_ellipsoids = \
['@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.850650808352 0.525731112119 0.0', '0.0 0.850650808352 0.525731112119', '0.525731112119 0.0 0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.850650808352 0.525731112119 0.0', '0.0 0.850650808352 -0.525731112119', '-0.525731112119 0.0 -0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.850650808352 -0.525731112119 0.0', '0.0 -0.850650808352 0.525731112119', '0.0 -0.850650808352 -0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.525731112119 0.0 -0.850650808352', '-0.850650808352 -0.525731112119 0.0', '-0.850650808352 0.525731112119 0.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.850650808352 0.525731112119 0.0', '0.525731112119 0.0 -0.850650808352', '0.0 0.850650808352 -0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.850650808352 -0.525731112119 0.0', '0.0 -0.850650808352 0.525731112119', '-0.525731112119 0.0 0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.850650808352 -0.525731112119 0.0', '0.0 -0.850650808352 -0.525731112119', '0.0 -0.850650808352 0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.0 0.850650808352 0.525731112119', '-0.525731112119 0.0 0.850650808352', '0.525731112119 0.0 0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.850650808352 -0.525731112119 0.0', '0.525731112119 0.0 0.850650808352', '0.0 -0.850650808352 0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.850650808352 -0.525731112119 0.0', '-0.525731112119 0.0 -0.850650808352', '0.0 -0.850650808352 -0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.525731112119 0.0 0.850650808352', '0.850650808352 -0.525731112119 0.0', '0.850650808352 0.525731112119 0.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.0 -0.850650808352 0.525731112119', '0.525731112119 0.0 0.850650808352', '-0.525731112119 0.0 0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.850650808352 -0.525731112119 0.0', '0.0 -0.850650808352 -0.525731112119', '0.525731112119 0.0 -0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.850650808352 0.525731112119 0.0', '0.0 0.850650808352 -0.525731112119', '0.0 0.850650808352 0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.525731112119 0.0 -0.850650808352', '0.850650808352 0.525731112119 0.0', '0.850650808352 -0.525731112119 0.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.0 0.850650808352 -0.525731112119', '0.525731112119 0.0 -0.850650808352', '-0.525731112119 0.0 -0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.850650808352 0.525731112119 0.0', '-0.525731112119 0.0 0.850650808352', '0.0 0.850650808352 0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.850650808352 0.525731112119 0.0', '0.0 0.850650808352 0.525731112119', '0.0 0.850650808352 -0.525731112119', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.525731112119 0.0 0.850650808352', '-0.850650808352 0.525731112119 0.0', '-0.850650808352 -0.525731112119 0.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '0.0 -0.850650808352 -0.525731112119', '-0.525731112119 0.0 -0.850650808352', '0.525731112119 0.0 -0.850650808352', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.82986983833 3.05146222424 3.0', '-1.0 3.7013016167 13.5146222424', '-0.894853777576 2.0 20.013016167', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.17013016167 3.05146222424 3.0', '-1.0 3.7013016167 -7.51462224238', '-1.10514622242 2.0 -14.013016167', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.82986983833 0.948537775762 3.0', '-1.0 0.298698383296 13.5146222424', '-1.0 0.298698383296 -7.51462224238', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.10514622242 2.0 -14.013016167', '-1.17013016167 0.948537775762 3.0', '-1.17013016167 3.05146222424 3.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.82986983833 3.05146222424 3.0', '-0.894853777576 2.0 -14.013016167', '-1.0 3.7013016167 -7.51462224238', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.17013016167 0.948537775762 3.0', '-1.0 0.298698383296 13.5146222424', '-1.10514622242 2.0 20.013016167', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.17013016167 0.948537775762 3.0', '-1.0 0.298698383296 -7.51462224238', '-1.0 0.298698383296 13.5146222424', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.0 3.7013016167 13.5146222424', '-1.10514622242 2.0 20.013016167', '-0.894853777576 2.0 20.013016167', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.82986983833 0.948537775762 3.0', '-0.894853777576 2.0 20.013016167', '-1.0 0.298698383296 13.5146222424', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.17013016167 0.948537775762 3.0', '-1.10514622242 2.0 -14.013016167', '-1.0 0.298698383296 -7.51462224238', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.894853777576 2.0 20.013016167', '-0.82986983833 0.948537775762 3.0', '-0.82986983833 3.05146222424 3.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.0 0.298698383296 13.5146222424', '-0.894853777576 2.0 20.013016167', '-1.10514622242 2.0 20.013016167', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.82986983833 0.948537775762 3.0', '-1.0 0.298698383296 -7.51462224238', '-0.894853777576 2.0 -14.013016167', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.82986983833 3.05146222424 3.0', '-1.0 3.7013016167 -7.51462224238', '-1.0 3.7013016167 13.5146222424', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-0.894853777576 2.0 -14.013016167', '-0.82986983833 3.05146222424 3.0', '-0.82986983833 0.948537775762 3.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.0 3.7013016167 -7.51462224238', '-0.894853777576 2.0 -14.013016167', '-1.10514622242 2.0 -14.013016167', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.17013016167 3.05146222424 3.0', '-1.10514622242 2.0 20.013016167', '-1.0 3.7013016167 13.5146222424', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.17013016167 3.05146222424 3.0', '-1.0 3.7013016167 13.5146222424', '-1.0 3.7013016167 -7.51462224238', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.10514622242 2.0 20.013016167', '-1.17013016167 3.05146222424 3.0', '-1.17013016167 0.948537775762 3.0', '@trianglelist color=red1 alpha=0.250000 master={points} nobutton', '-1.0 0.298698383296 -7.51462224238', '-1.10514622242 2.0 -14.013016167', '-0.894853777576 2.0 -14.013016167']
exp_subdivide = \
[((1, 0, 0), (0.89442719099991586, 0.0, 0.44721359549995793), (0.89442719099991586, 0.44721359549995793, 0.0)), ((0.89442719099991586, 0.0, 0.44721359549995793), (0.89442719099991586, 0.44721359549995793, 0.0), (0.57735026918962573, 0.57735026918962573, 0.57735026918962573)), ((0.89442719099991586, 0.0, 0.44721359549995793), (0.44721359549995793, 0.0, 0.89442719099991586), (0.57735026918962573, 0.57735026918962573, 0.57735026918962573)), ((0.44721359549995793, 0.0, 0.89442719099991586), (0.0, 0.44721359549995793, 0.89442719099991586), (0.57735026918962573, 0.57735026918962573, 0.57735026918962573)), ((0.44721359549995793, 0.0, 0.89442719099991586), (0, 0, 1), (0.0, 0.44721359549995793, 0.89442719099991586)), ((0.89442719099991586, 0.44721359549995793, 0.0), (0.57735026918962573, 0.57735026918962573, 0.57735026918962573), (0.44721359549995793, 0.89442719099991586, 0.0)), ((0.44721359549995793, 0.89442719099991586, 0.0), (0, 1, 0), (0.0, 0.89442719099991586, 0.44721359549995793)), ((0.44721359549995793, 0.89442719099991586, 0.0), (0.0, 0.89442719099991586, 0.44721359549995793), (0.57735026918962573, 0.57735026918962573, 0.57735026918962573)), ((0.0, 0.89442719099991586, 0.44721359549995793), (0.0, 0.44721359549995793, 0.89442719099991586), (0.57735026918962573, 0.57735026918962573, 0.57735026918962573))]

#run tests if called from command line
if __name__ == "__main__":
    main()
