/*
	Description: interface to git programs

	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <unistd.h> // usleep()
#include <stdlib.h> // used by getenv(), setenv()
#include <qapplication.h>
#include <qdatetime.h>
#include <qtextcodec.h>
#include <qregexp.h>
#include <qsettings.h>
#include <qfile.h>
#include <qdir.h>
#include <qeventloop.h>
#include <qprocess.h>
#include <qtextcodec.h>
#include "mainimpl.h"
#include "annotate.h"
#include "cache.h"
#include "dataloader.h"
#include "git.h"

using namespace QGit;

Git::Git(QWidget* p, const char* n) : QObject(p, n), par(p) {

	EM_INIT(exGitStopped, "Stopping connection with git");

	cacheNeedsUpdate = isMergeHead = false;
	isStGIT = isGIT = loadingUnAppliedPatches = false;
	errorReportingEnabled = true; // report errors if run() fails
	runningProcesses = 0;

	revs.setAutoDelete(true);
	revs.clear();
	revs.resize(100003); // must be a prime number see QDict docs

	cache = new Cache(this);

	// get git bins path
	QString execPath;
	if (run("git --exec-path", &execPath)) { // workDir is empty now
		execPath = execPath.section('\n', 0, 0).append(':');
		QString curPath(getenv("PATH"));
		if (!curPath.contains(execPath)) {
			curPath.prepend(execPath);
			setenv("PATH", curPath.latin1(), 1);
		}
	}
	// control git version
	QString gitVersion;
	if (run("git --version", &gitVersion)) {

		gitVersion = gitVersion.section(' ', -1, -1).section('.', 0, 2);
		if (gitVersion < GIT_VERSION) {

			// simply send information, the 'not compatible version'
			// policy should be implemented upstream
			const QString cmd("Current git version is " + gitVersion +
				" but is required " + GIT_VERSION + " or better");
			const QString errorDesc("Your installed git is too old."
				"\nPlease upgrade to avoid possible misbehaviours.");
			MainExecErrorEvent* e = new MainExecErrorEvent(cmd, errorDesc);
			QApplication::postEvent(p, e);
		}
	}
}

Git::~Git() {

	clearRevs();
}

void Git::setTextCodec(QTextCodec* tc) {

	QTextCodec::setCodecForCStrings(tc); // works also with tc == 0 (Latin1)
	QString mimeName((tc != NULL) ? tc->mimeName() : "Latin1");

	// workaround Qt issue of mime name diffrent from
	// standard http://www.iana.org/assignments/character-sets
	if (mimeName == "Big5-HKSCS")
		mimeName = "Big5";

	run("git-repo-config i18n.commitencoding " + mimeName);
}

QTextCodec* Git::getTextCodec(bool* isGitArchive) {

	*isGitArchive = isGIT;
	if (!isGIT) // can be called also when not in an archive
		return NULL;

	QString runOutput;
	if (!run("git-repo-config --get i18n.commitencoding", &runOutput))
		return NULL;

	if (runOutput.isEmpty()) // git docs says default is utf-8
		return QTextCodec::codecForName("utf8");

	return QTextCodec::codecForName(runOutput.stripWhiteSpace());
}

const QString Git::getLocalDate(SCRef gitDate) {

	QDateTime d;
	d.setTime_t(gitDate.toULong());
	return d.toString(Qt::LocalDate);
}

const QString Git::getRevInfo(SCRef sha, bool isHistory) {

	const Rev* c = revLookup(sha, isHistory);
	if (c == NULL)
		return "";

	QString refsInfo;
	if (c->isBranch) {
		if (c->isCurrentBranch)
			refsInfo = "Head: " + c->branch;
		else
			refsInfo = "Branch: " + c->branch;
	}
	if (c->isTag)
		refsInfo.append("   Tag: " + c->tag);
	if (c->isRef)
		refsInfo.append("   Ref: " + c->ref);
	if (c->isApplied || c->isUnApplied)
		refsInfo.append("   Patch: " + getPatchName(sha));

	refsInfo = refsInfo.stripWhiteSpace();

	if (!refsInfo.isEmpty()) {
		SCRef msg(getTagMsg(sha));
		if (!msg.isEmpty())
			refsInfo.append("  [" + msg + "]");
	}
	return refsInfo;
}

const QString Git::getRefSha(SCRef refName) {

	if (tags.contains(refName))
		return tagsSHA[tags.findIndex(refName)];

	if (heads.contains(refName))
		return headsSHA[heads.findIndex(refName)];

	if (refs.contains(refName))
		return refsSHA[refs.findIndex(refName)];

	// if a ref was not found perhaps is an abbreviated form
	QString runOutput;
	if (!run("git-rev-parse " + refName, &runOutput))
		return "";

	return runOutput.stripWhiteSpace();
}

bool Git::isPatchName(SCRef patchName) {

	return patchNames.values().contains(patchName);
}

bool Git::isTagName(SCRef tag) { return tags.contains(tag); }

bool Git::isCommittingMerge() { return isMergeHead; }

bool Git::isStGITStack() { return isStGIT; }

bool Git::isUnapplied(SCRef sha) { return unAppliedSHA.contains(sha); }

Annotate* Git::startAnnotate(QObject* guiObj) { // non blocking

	Annotate* ann = new Annotate(this, guiObj);
	ann->start(histFileName, histRevOrder); // non  blocking call
	return ann;
}

void Git::cancelAnnotate(Annotate* ann) {

	if (ann)
		if (ann->stop()) // if still running will be deleted by annotateExited()
			delete ann;
}

void Git::annotateExited(Annotate* ann) {

	if (ann->isCanceled()) {
		ann->deleteLater();
		return; // do not emit anything
	}
	// Annotate object will be deleted by cancelAnnotate()
	if (ann->isValid()) {
		const QString msg(QString("Annotated %1 files in %2 ms")
				.arg(histRevOrder.count()).arg(ann->elapsed()));
		emit annotateReady(ann, ann->file(), ann->isValid(), msg);
	} else
		emit annotateReady(ann, ann->file(), ann->isValid(), "");
}

const FileAnnotation* Git::lookupAnnotation(Annotate* ann, SCRef fileName, SCRef sha) {

	if (!ann)
		return NULL;

	return ann->lookupAnnotation(sha, fileName);
}

void Git::cancelFileHistoryLoading() {
// normally called when closing file viewer

	emit cancelHistLoading(); // non blocking
}

const Rev* Git::revLookup(SCRef sha, bool isHistoryView) {

	const RevMap& r = (isHistoryView) ? histRevs : revs;
	return r[sha];
}

bool Git::run(SCRef runCmd, QString* runOutput, SCRef buf) {

	MyProcess p(par, this, workDir, errorReportingEnabled);
	return p.runSync(runCmd, runOutput, buf);
}

MyProcess* Git::runAsync(SCRef runCmd, QObject* receiver, SCRef buf) {

	MyProcess* p = new MyProcess(par, this, workDir, errorReportingEnabled);
	if (!p->runAsync(runCmd, receiver, buf)) {
		delete p;
		p = NULL;
	}
	return p; // auto-deleted when done
}

void Git::cancelProcess(MyProcess* p) {

	if (p)
		p->cancel(); // non blocking call
}

int Git::findFileIndex(const RevFile& rf, SCRef name) {

	if (name.isEmpty())
		return -1;

	int idx = name.findRev('/') + 1;
	SCRef dr = name.left(idx);
	SCRef nm = name.right(name.length() - idx);

	for (uint i = 0; i < rf.names.count(); ++i) {
		if (fileNamesVec[rf.names[i]] == nm && dirNamesVec[rf.dirs[i]] == dr)
			return i;
	}
	return -1;
}

const QString Git::getLaneParent(SCRef fromSHA, uint laneNum) {

	QValueVector<QString>::const_iterator it;
	it = qFind(revOrder.constBegin(), revOrder.constEnd(), fromSHA);
	if (it == revOrder.constEnd())
		return QString();

	while (it != revOrder.constBegin()) {
		--it;
		const Rev* r = revLookup(*it);
		if (laneNum >= r->lanes.count())
			return QString();

		if (!isFreeLane(r->lanes[laneNum])) {
			int type = r->lanes[laneNum], parNum = 0;
			while (!isMerge(type) && type != ACTIVE) {
				if (isHead(type))
					parNum++;
				type = r->lanes[--laneNum];
			}
			return r->parent(parNum);
		}
	}
	return QString();
}

const QStringList Git::getChilds(SCRef parent) {

	QStringList childs;
	QDictIterator<Rev> it(revs);
	for ( ; it.current(); ++it) { // traverse the whole map!
		if (it.current()->parents().findIndex(parent) != -1)
			childs.append(it.currentKey());
	}
	// reorder childs by id
	QStringList::iterator itC(childs.begin());
	for ( ; itC != childs.end(); ++itC) {
		const Rev* r = revLookup(*itC);
		(*itC).prepend(QString("%1 ").arg(r->id, 5));
	}
	childs.sort();
	for (itC = childs.begin(); itC != childs.end(); ++itC)
		(*itC) = (*itC).section(' ', -1, -1);

	return childs;
}

const QString Git::getShortLog(SCRef sha) {

	const Rev* r = revLookup(sha);
	return (r == NULL) ? "" : r->shortLog();
}

const QStringList Git::getTagNames(bool onlyLoaded) {

	return (onlyLoaded) ? loadedTagNames : tags;
}

const QStringList Git::getBranchNames() {

	return loadedBranchNames;
}

const QString Git::getTagMsg(SCRef sha) {

	Rev* c = const_cast<Rev*>(revLookup(sha));
	if (!c)
		return QString();

	if (!c->tagMsg.isEmpty())
		return c->tagMsg;

	QRegExp pgp = QRegExp("-----BEGIN PGP SIGNATURE*END PGP SIGNATURE-----", true, true);

	if (tagsObj.contains(sha)) {
		QString runOutput;
		if (run("git-cat-file tag " + tagsObj[sha], &runOutput)) {
			c->tagMsg = runOutput.section("\n\n", 1);
			if (!c->tagMsg.isEmpty())
				c->tagMsg = c->tagMsg.remove(pgp).stripWhiteSpace();
		}
		tagsObj.remove(sha);
	}
	return c->tagMsg;
}

MyProcess* Git::getDiff(SCRef sha, QObject* receiver, SCRef diffToSha, bool combined) {

	QString runCmd;
	if (sha == ZERO_SHA) {
		runCmd = "git-diff-index -r -m --patch-with-stat HEAD";
		return runAsync(runCmd, receiver);
	}
	// workaround: --patch-with-stat is still not working well with merges
	const Rev* c = revLookup(sha);
	bool isMerge = (c ? c->parentsCount() > 1 : false);
	const QString opt(isMerge ? "-p " : "--patch-with-stat ");
	runCmd = (combined) ? "git-diff-tree -r -c " + opt : "git-diff-tree -r -m " + opt;

	if (diffToSha.isEmpty())
		runCmd.append(sha);
	else
		runCmd.append(diffToSha + " " + sha);

	return runAsync(runCmd, receiver);
}

const QString Git::getFileSha(SCRef file, SCRef revSha) {

	const QString sha(revSha == ZERO_SHA ? "HEAD" : revSha);
	QString runCmd("git-ls-tree -r " + sha + " " + file), runOutput;
	run(runCmd, &runOutput);
	if (runOutput.isEmpty()) // deleted file case
		return "";

	return runOutput.mid(12, 40);
}

MyProcess* Git::getFile(SCRef file, SCRef revSha, QObject* receiver, QString* result) {

	QString runCmd;
	if (revSha == ZERO_SHA)
		runCmd = "cat " + file;
	else {
		SCRef fileSha(getFileSha(file, revSha));
		if (!fileSha.isEmpty())
			runCmd = "git-cat-file blob " + fileSha;
		else
			runCmd = "git-diff-tree HEAD HEAD"; // fake an empty file reading
	}
	if (receiver == NULL) {
		run(runCmd, result);
		return NULL; // in case of sync call we ignore run() return value
	}
	return runAsync(runCmd, receiver);
}

void Git::getTree(SCRef treeSha, SList names, SList shas,
			   SList types, bool isWorkingDir, SCRef treePath) {

	QStringList newFiles, delFiles, dummy, tmp;
	if (isWorkingDir) { // retrieve unknown and deleted files under treePath

		getWorkDirFiles(UNKNOWN, tmp, dummy);
		loopList(it, tmp) { // don't add unknown files under other directories
			QFileInfo f(*it);
			SCRef d(f.dirPath(false));
			if (d == treePath || (treePath.isEmpty() && d == "."))
				newFiles.append(f.fileName());
		}
		getWorkDirFiles(DELETED, delFiles, dummy);
	}
	// if needed fake a working directory tree starting from HEAD tree
	const QString tree(treeSha == ZERO_SHA ? "HEAD" : treeSha);
	QString runOutput;
	if (!run("git-ls-tree " + tree, &runOutput))
		return;

	const QStringList sl(QStringList::split('\n', runOutput));
	loopList(it, sl) {
		// insert in order any good unknown file to the list,
		// newFiles must be already sorted
		SCRef fn((*it).section('\t', 1, 1));
		while (!newFiles.empty() && newFiles.first() < fn) {
			names.append(newFiles.first());
			shas.append("");
			types.append(NEW);
			newFiles.pop_front();
		}
		// append any not deleted file
		SCRef fp(treePath.isEmpty() ? fn : treePath + '/' + fn);
		if (delFiles.empty() || (delFiles.findIndex(fp) == -1)) {
			names.append(fn);
			shas.append((*it).mid(12, 40));
			types.append((*it).mid(7, 4));
		}
	}
	while (!newFiles.empty()) { // append any remaining unknown file
		names.append(newFiles.first());
		shas.append("");
		types.append(NEW);
		newFiles.pop_front();
	}
}

void Git::getWorkDirFiles(SCRef status, SList files, SList dirs) {

	files.clear();
	dirs.clear();
	const RevFile* f = getFiles(ZERO_SHA);
	if (!f)
		return;

	for (uint i = 0; i < f->names.count(); i++) {
		if (f->status[i].endsWith(status)) {

			SCRef fp(filePath(*f, i));
			files.append(fp);
			for (int j = 0; j < fp.contains('/'); j++) {

				SCRef dir(fp.section('/', 0, j));
				if (dirs.findIndex(dir) == -1)
					dirs.append(dir);
			}
		}
	}
}

bool Git::isSameTree(SCRef sha) {

	const RevFile* files = getFiles(sha);
	if (files) {
		for (uint i = 0; i < files->names.count(); ++i) {
			SCRef st(files->status[i].right(1));
			if (st == DELETED || st == NEW)
				return false;
		}
	}
	return true;
}

bool Git::isParentOf(SCRef par, SCRef child) {

	const Rev* c = revLookup(child);
	if (!c || c->parentsCount() != 1) // we don't handle merges
		return false;

	return (c->parent(0) == par);
}

bool Git::isSameFiles(SCRef tree1Sha, SCRef tree2Sha) {

	// early skip common case of browsing with up and down arrows, i.e.
	// going from parent(child) to child(parent). In this case we can
	// check RevFileMap and skip a costly git-diff-tree call.
	if (isParentOf(tree1Sha, tree2Sha))
		return isSameTree(tree2Sha);

	if (isParentOf(tree2Sha, tree1Sha))
		return isSameTree(tree1Sha);

	const QString runCmd("git-diff-tree -r -c " + tree1Sha + " " + tree2Sha);
	QString runOutput;
	if (!run(runCmd, &runOutput))
		return false;

	if (runOutput.find(" A\t") != -1 || runOutput.find(" D\t") != -1)
		return false;

	return true;
}

const QString Git::getDefCommitMsg() {

	SCRef sha((appliedSHA.empty()) ? ZERO_SHA : appliedSHA.last());
	const Rev* c = revLookup(sha);
	if (c == NULL) {
		dbp("ASSERT: getDefCommitMsg sha <%1> not found", sha);
		return "";
	}
	if (sha == ZERO_SHA)
		return c->longLog();

	return c->shortLog() + '\n' + c->longLog().stripWhiteSpace();
}

const QString Git::colorMatch(SCRef txt, QRegExp& regExp) {

	QString text(txt);
	if (regExp.isEmpty())
		return text;

	// we use $_1 and $_2 instead of '<' and '>' to avoid later substitutions
	SCRef startCol(QString::fromLatin1("$_1b$_2$_1font color=\"red\"$_2"));
	SCRef endCol(QString::fromLatin1("$_1/font$_2$_1/b$_2"));
	int pos = 0;
	while ((pos = text.find(regExp, pos)) != -1) {

		SCRef match(regExp.cap(0));
		const QString coloredText(startCol + match + endCol);
		text.replace(pos, match.length(), coloredText);
		pos += coloredText.length();
	}
	return text;
}

const QString Git::getDesc(SCRef sha, QRegExp& shortLogRE, QRegExp& longLogRE) {

	if (sha.isEmpty())
		return QString();

	const Rev* c = revLookup(sha);
	if (c == NULL) {
		dbp("ASSERT: getDesc sha <%1> not found", sha);
		return QString();
	}
	if (c->isDiffCache)
		return c->longLog();

	QString text("Author: " + c->author() + "\nDate:   ");
	text.append(getLocalDate(c->authorDate()));
	text.append("\nParent: ").append(c->parents().join("\nParent: "));
	text.append("\n\n    " + colorMatch(c->shortLog(), shortLogRE) +
			'\n' + colorMatch(c->longLog(), longLogRE));

	// replace '<' and '>' to avoid fooling rtf engine
	text.replace('<', QString::fromLatin1("&lt;"));
	text.replace('>', QString::fromLatin1("&gt;"));

	// hightlight SHA's
	int pos = 0;
	QRegExp reSHA("[0-9a-f]{40}", false);
	reSHA.setMinimal(true);
	while ((pos = text.find(reSHA, pos)) != -1) {

		SCRef sha(reSHA.cap(0));
		const Rev* r = revLookup(sha);
		QString slog((r == NULL) ? sha : r->shortLog());
		if (slog.isEmpty()) // very rare but possible
			slog = sha;
		if (slog.length() > 60)
			slog = slog.left(57).stripWhiteSpace().append("...");

		slog.replace('<', QString::fromLatin1("&lt;"));
		slog.replace('>', QString::fromLatin1("&gt;"));
		const QString link(QString::fromLatin1("<a href=\"") + sha +
			QString::fromLatin1("\">") + slog + QString::fromLatin1("</a>"));

		text.replace(pos, sha.length(), link);
		pos += link.length();
	}

	// convert spaces and tabs to &nbsp; to preserve multiple spaces
	pos = 0;
	QRegExp longSpaces(" ([ ]+)");
	while ((pos = longSpaces.search(text)) != -1) {
		QString cap(longSpaces.cap(1));
		cap.replace(QRegExp(" "), QString::fromLatin1("&nbsp;"));
		text.replace(pos + 1, longSpaces.matchedLength() - 1, cap);
	}
	text.replace(QRegExp("\t"), QString::fromLatin1("&nbsp;&nbsp;&nbsp;"));
	text.replace(QRegExp("\n"), QString::fromLatin1("<br>\n"));
	text.replace(QString::fromLatin1("$_1"), QString::fromLatin1("<"));
	text.replace(QString::fromLatin1("$_2"), QString::fromLatin1(">"));
	text.prepend(QString::fromLatin1("<qt><p>"));
	text.append(QString::fromLatin1("</p></qt>"));
	return text;
}

const RevFile* Git::getAllMergeFiles(const Rev* r) {

	SCRef mySha(ALL_MERGE_FILES + r->sha());
	RevFileMap::Iterator rf = revsFiles.find(mySha);
	if (rf == revsFiles.end()) {
		QString runCmd("git-diff-tree -r -m " + r->sha()), runOutput;
		if (!run(runCmd, &runOutput))
			return NULL;

		rf = revsFiles.insert(mySha, RevFile());
		parseDiffFormat(*rf, runOutput);
	}
	return &(*rf);
}

const RevFile* Git::getFiles(SCRef sha, SCRef diffToSha, bool allFiles, SCRef path) {

	const Rev* r = revLookup(sha);
	if (r == NULL)
		return NULL;

	if (r->parentsCount() == 0) // skip initial rev
		return NULL;

	if (r->parentsCount() > 1 && diffToSha.isEmpty() && allFiles)
		return getAllMergeFiles(r);

	if (!diffToSha.isEmpty() && (sha != ZERO_SHA)) {
		QString runCmd("git-diff-tree -r -m ");
		runCmd.append(diffToSha + " " + sha);
		if (!path.isEmpty())
			runCmd.append(" " + path);
		QString runOutput;
		if (!run(runCmd, &runOutput))
			return NULL;
		// we insert a dummy revision file object. It will be
		// overwritten at each request but we don't care.
		RevFileMap::Iterator rf = revsFiles.insert(CUSTOM_SHA, RevFile());
		parseDiffFormat(*rf, runOutput);
		return &(*rf);
	}
	RevFileMap::Iterator rf = revsFiles.find(sha); // ZERO_SHA search arrives here
	if (rf == revsFiles.end()) {

		if (sha == ZERO_SHA) {
			dbs("ASSERT in Git::getFiles, ZERO_SHA not found");
			return NULL;
		}
		const QString runCmd("git-diff-tree -r -c " + sha);
		QString runOutput;
		if (!run(runCmd, &runOutput))
			return false;

		if (!revsFiles.contains(sha)) { // has been created in the mean time?
			rf = revsFiles.insert(sha, RevFile());
			parseDiffFormat(*rf, runOutput);
			cacheNeedsUpdate = true;
		} else
			return &(*revsFiles.find(sha));
	}
	return &(*rf);
}

void Git::getFileHistory(SCRef name) {

	histRevs.clear();
	histRevOrder.clear();
	firstHistFreeLane = 0;
	histLns.clear();
	histFileName = name;
	startRevList(histFileName, true);
}

void Git::getFileFilter(SCRef path, QMap<QString, bool>& shaMap) {

	shaMap.clear();
	QRegExp rx(path, false, true);
	RevFileMap::const_iterator it;
	loop(StrVect, itr, revOrder) {
		it = revsFiles.find(*itr);
		if (it == revsFiles.constEnd())
			continue;
		// case insensitive, wildcard search
		for (uint i = 0; i < (*it).names.count(); ++i)
			if (rx.search(filePath(*it, i)) != -1) {
				shaMap.insert(it.key(), true);
				break;
			}
	}
}

bool Git::getPatchFilter(SCRef exp, bool isRegExp, QMap<QString, bool>& shaMap) {

	shaMap.clear();
	QString buf;
	loop(StrVect, it, revOrder)
		if (*it != ZERO_SHA)
			buf.append(*it).append('\n');

	if (buf.isEmpty())
		return true;

	QString runCmd("git-diff-tree -r -s --stdin "), runOutput;
	if (isRegExp)
		runCmd.append("--pickaxe-regex ");

	runCmd.append(QUOTE_CHAR + "-S" + exp + QUOTE_CHAR);
	if (!run(runCmd, &runOutput, buf))
		return false;

	QStringList sl(QStringList::split('\n', runOutput));
	loopList(it2, sl)
		shaMap.insert(*it2, true);

	return true;
}

bool Git::applyPatchFile(SCRef patchPath, bool dropping, bool fold) {

	SCRef quotedPath(QUOTE_CHAR + patchPath + QUOTE_CHAR);

	if (isStGIT) {
		if (fold)
			return run("stg fold " + quotedPath);

		return run("stg import --mail " + quotedPath);
	}
	QString runCmd("git-am -k -u --3way ");
	if (testFlag(SIGN_F) && !dropping)
		runCmd.append("--signoff ");
	return run(runCmd  + quotedPath);
}

bool Git::formatPatch(SCList shaList, SCRef dirPath, SCRef remoteDir) {

	bool remote = !remoteDir.isEmpty();
	QSettings settings;
	const QString FPArgs = settings.readEntry(APP_KEY + FPATCH_ARGS_KEY, FPATCH_ARGS_DEF);

	QString runCmd("git-format-patch");
	if (testFlag(NUMBERS_F) && !remote)
		runCmd.append(" -n");
	if (remote)
		runCmd.append(" --keep-subject");
	runCmd.append(" -o " + dirPath);
	if (!FPArgs.isEmpty())
		runCmd.append(" " + FPArgs);

// 	runCmd.append(" --start-number="); keep git 1.3.0 compatibility

	const QString tmp(workDir);
	if (remote)
		workDir = remoteDir;

// 	int n = 1; keep git 1.3.0 compatibility
	bool ret = false;
	loopList(it, shaList) { // shaList is ordered by newest to oldest
/*		const QString cmd(runCmd + QString::number(n) + " " +
				*it + QString::fromLatin1("^..") + *it);
		n++;*/
		const QString cmd(runCmd + " " +
				*it + QString::fromLatin1("^..") + *it);
		ret = run(cmd); // run() uses workDir value
		if (!ret)
			break;
	}
	workDir = tmp;
	return ret;
}

bool Git::updateIndex(SCList selFiles) {

	if (selFiles.empty())
		return true;
	QString runCmd("git-update-index --add --remove --replace -- ");
	runCmd.append(selFiles.join(" "));
	return run(runCmd);
}

bool Git::commitFiles(SCList selFiles, SCRef msg) {
/*
	Part of this code is taken from Fredrik Kuivinen "Gct"
	tool. I have just translated from Python to C++
*/
	const QString msgFile(gitDir + "/qgit_cmt_msg");
	if (!writeToFile(msgFile, msg)) // early skip
		return false;

	// add user selectable commit options
	QSettings settings;
	const QString CMArgs = settings.readEntry(APP_KEY + CMT_ARGS_KEY, CMT_ARGS_DEF);

	QString cmtOptions;
	if (!CMArgs.isEmpty())
		cmtOptions.append(" " + CMArgs);

	if (testFlag(SIGN_CMT_F))
		cmtOptions.append(" -s");

	if (testFlag(VERIFY_CMT_F))
		cmtOptions.append(" -v");

	// extract not selected files already updated
	// in index, we don't want to commit them so we need
	// to revert the index update with git-read-tree HEAD
	QStringList notSelCachedFiles(getOtherFiles(selFiles, true));

	// extract selected NOT to be deleted files because
	// git-commit doesn't work with them
	QStringList selNotDelFiles;
	const RevFile* files = getFiles(ZERO_SHA);
	loopList(it, selFiles) {
		int idx = findFileIndex(*files, *it);
		if (!files->status[idx].endsWith(DELETED))
			selNotDelFiles.append(*it);
	}
	// test if we need a git-read-tree to temporary
	// remove not selected file's changes from index
	if (!notSelCachedFiles.empty())
		if (!run("git-read-tree --reset HEAD"))
			return false;

	// before to commit we have to update index with selected files
	// because git-commit doesn't uses --add flag
	updateIndex(selFiles);

	// now we can commit, the script will update index with
	// selected files (only for modify) for us
	QString runCmd("git-commit -i" + cmtOptions + " -F " + msgFile);
	runCmd.append(" " + selNotDelFiles.join(" "));
	if (!run(runCmd))
		return false;

	// re-update index with not selected files
	if (!notSelCachedFiles.empty())
		if (!updateIndex(notSelCachedFiles))
			return false;
	QDir dir(workDir);
  	dir.remove(msgFile);
	return true;
}

bool Git::mkPatchFromIndex(SCRef msg, SCRef patchFile) {

	QString runOutput;
	if (!run("git-diff-index --cached -p HEAD", &runOutput))
		return false;

	SCRef patch("Subject: " + msg + "\n---\n" + runOutput);
	if (!writeToFile(patchFile, patch))
		return false;

	return true;
}

const QStringList Git::getOtherFiles(SCList selFiles, bool onlyCached) {

	const RevFile* files = getFiles(ZERO_SHA);
	QStringList notSelFiles;
	for (uint i = 0; i < files->names.count(); ++i) {
		SCRef fp = filePath(*files, i);
		if (selFiles.find(fp) == selFiles.constEnd()) // not selected...
			if (!onlyCached)
				notSelFiles.append(fp);
			else if (files->status[i].startsWith(CACHED_FILE))
				notSelFiles.append(fp);
	}
	return notSelFiles;
}

void Git::removeFiles(SCList selFiles, SCRef workDir, SCRef ext) {

	QDir d(workDir);
	loopList(it, selFiles)
		d.rename(*it, *it + ext);
}

void Git::restoreFiles(SCList selFiles, SCRef workDir, SCRef ext) {

	QDir d(workDir);
	loopList(it, selFiles)
		d.rename(*it + ext, *it); // overwrites any exsistant file
}

void Git::removeDeleted(SCList selFiles) {

	QDir dir(workDir);
	const RevFile* files = getFiles(ZERO_SHA);
	loopList(it, selFiles) {
		int idx = findFileIndex(*files, *it);
		if (files->status[idx].endsWith(DELETED))
			dir.remove(*it);
	}
}

bool Git::stgCommit(SCList selFiles, SCRef msg, SCRef patchName, bool add) {

	// here the deal is to create a patch with the diffs between the
	// updated index and HEAD, then resetting the index and working
	// dir to HEAD so to have a clean tree, then import/fold the patch
	bool retval = true;
	const QString patchFile(gitDir + "/qgit_tmp_patch");
	const QString extNS(".qgit_removed_not_selected");
	const QString extS(".qgit_removed_selected");

	// we have selected modified files in selFiles, we still need
	// to know the not selected but modified files and, among
	// theese the cached ones to proper restore state at the end.
	QStringList notSelFiles = getOtherFiles(selFiles, false);
	QStringList notSelCachedFiles = getOtherFiles(selFiles, true);

	// update index with selected files
	if (!run("git-read-tree --reset HEAD"))
		goto error;
	if (!updateIndex(selFiles))
		goto error;

	// create a patch with diffs between index and HEAD
	if (!mkPatchFromIndex(msg, patchFile))
		goto error;

	// temporary remove files according to their type
	removeFiles(selFiles, workDir, extS); // to use in case of rollback
	removeFiles(notSelFiles, workDir, extNS); // to restore at the end

	// checkout index to have a clean tree
	if (!run("git-read-tree --reset HEAD"))
		goto error;
	if (!run("git-checkout-index -q -f -u -a"))
		goto rollback;

	// finally import/fold the patch
	if (add) {
		// update patch message before to fold so to use refresh only as a rename tool
		if (!msg.isEmpty()) {
			if (!run("stg refresh --message \"" + msg.stripWhiteSpace() + "\""))
				goto rollback;
		}
		if (!run("stg fold " + patchFile))
			goto rollback;
		if (!run("stg refresh")) // refresh needed after fold
			goto rollback;
	} else {
		 if (!run("stg import --mail --name " + patchName + " " + patchFile))
			goto rollback;
	}
	goto exit;

rollback:
	restoreFiles(selFiles, workDir, extS);
	removeDeleted(selFiles); // remove files to be deleted from working tree

error:
	retval = false;

exit:
	// it is safe to call restore() also if back-up files don't
	// exsist, so we can 'goto exit' from anywhere.
	restoreFiles(notSelFiles, workDir, extNS);
	updateIndex(notSelCachedFiles);
	QDir dir(workDir);
  	dir.remove(patchFile);
	loopList(it, selFiles)
		dir.remove(*it + extS); // remove temporary backup rollback files
	return retval;
}

bool Git::makeTag(SCRef sha, SCRef tagName, SCRef msg) {

	if (msg.isEmpty())
		return run("git-tag " + tagName + " " + sha);

	return run("git-tag -m \"" + msg + "\" " + tagName + " " + sha);
}

bool Git::deleteTag(SCRef sha) {

	const Rev* r = revLookup(sha);
	if (!r || !r->isTag)
		return false;

	// r->tag can contain more then one tag, separated by '\n'
	return run("git-tag -d " + r->tag.section('\n', 0, 0));
}

bool Git::stgPush(SCRef sha) {

	const QString quotedPath(QUOTE_CHAR + patchNames[sha] + QUOTE_CHAR);
	bool ret = run("stg push " + quotedPath);
	if (!ret)
		run("stg push --undo");
	return ret;
}

bool Git::stgPop(SCRef sha) {

	if (sha == appliedSHA.last()) // top of the stack
		return run("stg pop");
	if (sha == appliedSHA.first())
		return run("stg pop --all");

	const QString quotedPath(QUOTE_CHAR + patchNames[sha] + QUOTE_CHAR);
	if (!run("stg pop --to=" + quotedPath))
		return false;

	return run("stg pop");
}

bool Git::writeToFile(SCRef fileName, SCRef data) {

	QFile file(QFile::encodeName(fileName));
	if (!file.open(IO_WriteOnly)) {
		dbp("ERROR: unable to write file %1", fileName);
		return false;
	}
	QTextStream stream(&file);
	stream << data;
	file.close();
	return true;
}

bool Git::readFromFile(SCRef fileName, QString& data) {

	data = "";
	QFile file(QFile::encodeName(fileName));
	if (!file.open(IO_ReadOnly)){
		dbp("ERROR: unable to read file %1", fileName);
		return false;
	}
	QTextStream stream(&file);
	while (!stream.atEnd())
		data.append(stream.readLine()).append('\n');

	file.close();
	return true;
}
