/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/

#ifndef PREFERENCES_H
#define PREFERENCES_H

#include <QColor>
#include <QHostAddress>
#include <QKeySequence>
#include <QDir>

/**
        @author Anistratov Oleg <ower@users.sourceforge.net>
*/
class Preferences
{
  private:
    static QString m_settingsDir;
    // Messages
    QColor m_myMsgsColor;
    QColor m_sysMsgsColor;
    QColor m_baseMsgColor;

    bool   m_colorWholeMsg;
    bool   m_colorWholeSysMsg;

    QString m_displayMsgFormat;

    // Network
    static bool    m_allowDiffPorts;
    static quint16 m_portIn;
    static quint16 m_portOut;
    static quint64 m_ip;
    static QHostAddress m_broadcast;
    static quint64 m_customIp;
    static QHostAddress m_customBroadcast;

    // Misc
    bool m_activateOnMsgIn;
    bool m_soundOnMsgIn;

    // Status
    QString m_statusDescription[6];
    QStringList m_statusDescriptions;


    // Now Listening
    /// 1 - send with message, 2 - change status, (1 | 2) - both
    int m_nlMode;
    QString m_nlFormat;
    /*static*/ QString m_nowListening;

    // Request message hystory
    /// -1 - request maximum messages
    int m_nHistoryMsgs;
    /// how long we will be wait answer for 'messages number request'
    int m_historyReqTimeout;

    // Smiles
    QString m_smilesThemePath;

    // Shortcuts
    QKeySequence m_sendMessageSeq;

    // Users List
    int m_usersListRefreshInterval;
    int m_usersListDeepRefreshInterval;

    QString m_executeCommandOnIncomingMsg;
    bool    m_isExecuteCommandOnIncomingMsg;

    // Warning Messages
    bool m_warningAboutTray;

  public:
    Preferences();
    ~Preferences(){};

    static QString defaultSmilesDir();
    static QString defaultSmilesDir(QList<QDir>);

    static void setBroadcast      (const QHostAddress & br){m_broadcast = br;}
    static void setSettingsDir    (const QString & dir){m_settingsDir = dir;}
    static void setPortIn         (quint16 port){m_portIn = port;}
    static void setPortOut        (quint16 port){m_portOut = port;}
    static void setIp             (quint64 ip)  {m_ip = ip;}
    static void setAllowDiffPorts (bool mode)   {m_allowDiffPorts = mode;}

    void setMyColor        (const QColor & color)   {m_myMsgsColor   = color;}
    void setSysColor       (const QColor & color)   {m_sysMsgsColor  = color;}
    void setBaseColor      (const QColor & color)   {m_baseMsgColor  = color;}
    void setMsgColorMode   (bool mode)      {m_colorWholeMsg    = mode;}
    void setSysMsgColorMode(bool mode)      {m_colorWholeSysMsg = mode;}
    void setActivateOnMsgIn(bool mode)      {m_activateOnMsgIn  = mode;}
    void setSoundOnMsgIn   (bool mode)      {m_soundOnMsgIn     = mode;}
    void setStatusDescription(const QString & descr, int status);

    static const QHostAddress & broadcast(){return m_broadcast;}
    static quint16 portIn                (){return m_portIn;}
    static quint16 portOut               (){return m_portOut;}
    static quint64 ip                    (){return m_ip;}
    static bool    allowDiffPorts        (){return m_allowDiffPorts;}
    static const QString & settingsDir   (){return m_settingsDir;}

    const QString & statusDescription(int status) const;
    const QColor  & sysColor () const {return m_sysMsgsColor;}
    const QColor  & myColor  () const {return m_myMsgsColor;}
    const QColor  & baseColor() const {return m_baseMsgColor;}
    bool colorWholeMsg       () const {return m_colorWholeMsg;}
    bool colorWholeSysMsg    () const {return m_colorWholeSysMsg;}
    bool activateOnMsgIn     () const {return m_activateOnMsgIn;}
    bool soundOnMsgIn        () const {return m_soundOnMsgIn;}

    void setNHistoryMsgs(int value){m_nHistoryMsgs = value;}
    int  nHistoryMsgs() const {return m_nHistoryMsgs;}

    void setHistoryReqTimeout(int value){m_historyReqTimeout = value;}
    int historyReqTimeout() const {return m_historyReqTimeout;}

    void setNlMode(int value){Q_ASSERT(value >= 0 && value <= 3); m_nlMode = value;}
    int nlMode() const{return m_nlMode;}

    void setSmilesThemePath(const QString & value)
    {m_smilesThemePath = value; if(!value.isEmpty() && value[value.size() - 1] != '/') m_smilesThemePath += "/";}
    const QString & smilesThemePath() const {return m_smilesThemePath;}

    void setExecuteCommandOnIncomingMsg(const QString& value){m_executeCommandOnIncomingMsg = value;}
    const QString & executeCommandOnIncomingMsg() const {return m_executeCommandOnIncomingMsg;}

    void setNlFormat(const QString& value){m_nlFormat = value;}
    const QString & nlFormat() const {return m_nlFormat;}

    void setDisplayMsgFormat(const QString& value){m_displayMsgFormat = value;}
    const QString & displayMsgFormat() const {return m_displayMsgFormat;}

    void setSendMessageSeq(const QKeySequence& value){m_sendMessageSeq = value;}
    const QKeySequence & sendMessageSeq() const {return m_sendMessageSeq;}

    void setUsersListRefreshInterval(int value){m_usersListRefreshInterval = value;}
    int usersListRefreshInterval() const {return m_usersListRefreshInterval;}

    void setUsersListDeepRefreshInterval(int value){m_usersListDeepRefreshInterval = value;}
    int usersListDeepRefreshInterval() const {return m_usersListDeepRefreshInterval;}

    static void setCustomIp(const quint64& value){m_customIp = value;}
    static quint64 customIp(){return m_customIp;}

    static void setCustomBroadcast(const QHostAddress& value){m_customBroadcast = value;}
    static QHostAddress customBroadcast(){return m_customBroadcast;}

    void setWarningAboutTray(bool value) {m_warningAboutTray = value;}
    bool warningAboutTray() const {return m_warningAboutTray;}

    void setStatusDescriptions(const QStringList & value){m_statusDescriptions = value;}
    void appendStatusDescription(const QString & value);
    void removeStatusDescription(const QString & value){m_statusDescriptions.removeAll(value);}
    QStringList statusDescriptions() const {return m_statusDescriptions;}

    void setNowListening(const QString& value){m_nowListening = value;}
    QString nowListening() const {return m_nowListening;}

    void setIsExecuteCommandOnIncomingMsg(bool value){m_isExecuteCommandOnIncomingMsg = value;}
    bool isExecuteCommandOnIncomingMsg() const {return m_isExecuteCommandOnIncomingMsg;}
};

#endif
