/***************************************************************************
  model.h
  -------------------
  Model class for brewing
  -------------------
  begin         12/12/2001
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright (c) 2002 David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef MODEL_H
#define MODEL_H

#include <qmap.h>
#include <qstring.h>

#include "calc.h"
#include "grain.h"
#include "hops.h"
#include "miscingredient.h"
#include "recipe.h"
#include "style.h"

/**
 * A homebrewing domain model class. This is a singleton class, since there is
 * only one brewing domain model. Currently the model can only access one
 * recipe at a time. Multiple recipes is a future trajectory.
 */
class Model {
 public:
    // Return pointer to the model.
    static Model *instance();

    // Create a new recipe
    void newRecipe();
    // Load a recipe into the model
    bool loadRecipe(const QString &filename);
    // Save the current recipe
    bool saveRecipe(const QString &filename);

    // set the default size volume
    void setDefaultSize(const Volume &v);
    // get the default size volume
    const Volume &defaultSize();
    // set the default style
    void setDefaultStyle(const Style &s);
    // get the default style
    const Style &defaultStyle();
   // set the default grain units
    void setDefaultGrainUnit(Unit &u);
    // get the default grain units
    const Unit &defaultGrainUnit();
    // set the default hops units
    void setDefaultHopsUnit(Unit &u);
    // get the default hops units
    const Unit &defaultHopsUnit();
    // set the default misc units
    void setDefaultMiscUnit(Unit &u);
    // get the default misc units
    const Unit &defaultMiscUnit();

    // return a list of styles
    QStringList stylesList();
    // return given style from name
    Style &style(const QString &name);
    // return a list of grains
    QStringList grainsList();
    // return given grain from name
    Grain* grain(const QString &name);
    // return a list of hops
    QStringList hopsList();
    // return a list of hop styles (this should be in Hops class)
    QStringList formsList();
    // return given hop from name
    Hops* hop(const QString &name);
    // return a list of misc ingredients
    QStringList miscList();
    // return given misc ingredient from name
    MiscIngredient* misc(const QString &name);

    // check validity of ingredients, adding them to db if not found
    void checkGrain(const Grain &g);
    void checkHops(const Hops &h);
    void checkMisc(const MiscIngredient &m);

    // return pointer to current recipe
    Recipe *recipe();

    // is the current recipe modified?
    bool modified();
    // set modified status
    void setModified(bool m);

    // return the recipe as an ascii string
    QString recipeText();
    // recalculate recipe
    void recalc();

    // set the data directory
    void setDataDir(const QString &datadir);

 private:
    // Private constructor.
    Model();
    // Private destructor
    ~Model();

    // load the data
    bool loadData(const QString &filename, bool quiet=false);
    // save the data
    void saveData(const QString &filename);
    // figure out the base directory for the data
    QString dataBase();

 private:
    static Model *instance_;

    QMap<QString, int> groupmap_;
    QMap<QString, int> fieldmap_;

    Volume defaultsize_;
    Style defaultstyle_;
    Unit *defaultgrainunit_;
    Unit *defaulthopsunit_;
    Unit *defaultmiscunit_;

    GrainList graindb_;
    HopsList hopdb_;
    MiscIngredientList miscdb_;
    StyleList styledb_;

    Recipe *recipe_;

    QString datadir_;
};

// Inlined Methods ///////////////////////////////////////////////////////////

inline const Volume &Model::defaultSize() { return defaultsize_; }

inline const Style &Model::defaultStyle() { return defaultstyle_; }

inline const Unit &Model::defaultGrainUnit() { return *defaultgrainunit_; }

inline const Unit &Model::defaultHopsUnit() { return *defaulthopsunit_; }

inline const Unit &Model::defaultMiscUnit() { return *defaultmiscunit_; }

inline Recipe *Model::recipe() { return recipe_; }

inline bool Model::modified() { return recipe_->modified(); }

inline void Model::setModified(bool m) { recipe_->setModified(m); }

inline void Model::recalc() { Calc::recalc(recipe_); }

inline void Model::setDataDir(const QString &datadir) { datadir_ = datadir; }

#endif // MODEL_H

/*
Local variables:
mode: c++
c-basic-offset: 4
End:
*/
