/***************************************************************************
  fixedspinbox.h
  -------------------
  A spinbox for fixed numbers
  -------------------
  begin         September 28th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this
  list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
  this list of conditions and the following disclaimer in the documentation
  and/or other materials provided with the distribution.

  3. Neither name of the copyright holders nor the names of its contributors may
  be used to endorse or promote products derived from this software without
  specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 ***************************************************************************/

#ifndef FIXEDSPINBOX_H
#define FIXEDSPINBOX_H

#include <qspinbox.h>
#include <qstring.h>

class QDoubleValidator;

class FixedSpinBox : public QSpinBox  {
    Q_OBJECT
public:
    // constructor
    FixedSpinBox(double minValue, double maxValue, double step = 0.1, unsigned fix = 2,
        QWidget * parent = 0, const char * name = 0 );
    // destructor
    ~FixedSpinBox();

    // set the minimum spinbox value
    void setMinValue (double value);
    // set the maximum spinbox value
    void setMaxValue (double value);
    // set the spinbox step value
    void setLineStep (double value);

    // return the spinbox value
    double value() const;

public slots:
    // set the value of the spinbox
    void setValue(double value);

signals:
    // send when the spinbox value changes
    void valueChanged(double value);

protected:
    // map the spinbox value to a text string
    virtual QString mapValueToText(int value);
    // map a text string to a spinbox value
    virtual int mapTextToValue(bool* ok);
    // called from validator when value has changed
    virtual void valueChange();

private:
    QDoubleValidator* validator_;
    unsigned fixed_;
    double divisor_;
    double tinyval_;
};

//////////////////////////////////////////////////////////////////////////////
// Inlined Methods

inline QString FixedSpinBox::mapValueToText(int value)
    { return QString::number(double(value) / divisor_, 'f', fixed_); }

inline int FixedSpinBox::mapTextToValue(bool*)
    { return int(cleanText().toDouble() * divisor_ + tinyval_); }

inline double FixedSpinBox::value() const
    { return QSpinBox::value() / double(divisor_); }

inline void FixedSpinBox::setMinValue(double value)
    { QSpinBox::setMinValue(int(divisor_ * value)); }

inline void FixedSpinBox::setMaxValue(double value)
    { QSpinBox::setMaxValue(int(divisor_ * value)); }

inline void FixedSpinBox::setLineStep(double value)
    { QSpinBox::setLineStep(int(divisor_ * value)); }

inline void FixedSpinBox::setValue(double value)
    { QSpinBox::setValue(int(divisor_ * value + tinyval_)); }

#endif // FIXEDSPINBOX_H
