#
# Copyright (c) 2002, 2003, 2004, 2005 Art Haas
#
# This file is part of PythonCAD.
#
# PythonCAD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# PythonCAD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PythonCAD; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# the GTK code for displaying a drawing
#

import types

import pygtk
pygtk.require('2.0')
import gtk
import gobject

from PythonCAD.Generic import image
from PythonCAD.Generic import layer
from PythonCAD.Generic import util

class LayerDisplay(object):
    def __init__(self, im):
        if not isinstance(im, image.Image):
            raise TypeError, "Invalid Image type: " + `type(im)`
        _sw = gtk.ScrolledWindow()
        _sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        _model = gtk.TreeStore(gobject.TYPE_STRING, gobject.TYPE_PYOBJECT)
        # _model.connect("rows-reordered", self._reordered)
        _iter = _model.append(None)
        _top_layer = im.getTopLayer()
        _model.set(_iter, 0, _top_layer.getName())
        _model.set(_iter, 1, _top_layer)
        _tv = gtk.TreeView(_model)
        _tv.set_reorderable(True) # drag-and-drop
        _tv.connect("button_press_event", self._treeViewButtonPress)
        _tv.connect("key_press_event", self._treeViewKeyPress)
        _select = _tv.get_selection()
        _select.set_mode(gtk.SELECTION_SINGLE)
        _select.connect("changed", self._selectionChanged)

        _renderer = gtk.CellRendererText()
        # _renderer.set_property("editable", True)
        _renderer.connect("edited", self._cellEdited)

        _column = gtk.TreeViewColumn("Layers", _renderer, text=0)
        _tv.append_column(_column)
        _sw.add(_tv)
        self.__image = im
        self.__sw = _sw
        self.__model = _model
        self.__treeview = _tv
        self.__layer = None
        im.connect('active_layer_changed', self._activeLayerChanged)
        im.connect('added_layer', self._addedLayer)
        im.connect('deleted_layer', self._deletedLayer)

    def _treeViewButtonPress(self, widget, event):
        _retval = False
        _button = event.button
        _x = int(event.x)
        _y = int(event.y)
        if event.window is widget.get_bin_window() and _button == 3:
            _menu = self._makePopupMenu(self, widget)
            _menu.popup(None, None, None, _button, event.time)
            _retval = True
        return _retval

    def _makePopupMenu(self, widget):
        _model = widget.get_model()
        _path, _col, _cx, _cy = widget.get_path_at_pos(_x, _y)
        _iter = _model.get_iter(_path)
        _layer = _model.get_value(_iter, 1)
        #
        _menu = gtk.Menu()
        _item = gtk.MenuItem("Rename")
        _item.connect("activate", self._renameLayer)
        _menu.append(_item)
        if _layer.isVisible():
            _item = gtk.MenuItem("Hide")
            _flag = False
        else:
            _item = gtk.MenuItem("Show")
            _flag = True
        _item.connect("activate", self._setLayerVisibility, _flag)
        _menu.append(_item)
        _item = gtk.MenuItem("Add Child Layer")
        _item.connect("activate", self._addChildLayer)
        _menu.append(_item)
        if _layer.hasSublayers():
            _item = gtk.MenuItem("Hide Children")
            _item.connect("activate", self._setChildrenVisibility, False)
            _menu.append(_item)
            _item = gtk.MenuItem("Show Children")
            _item.connect("activate", self._setChildrenVisibility, True)
            _menu.append(item)
        else:
            if _layer.getParentLayer() is not None:
                if not _layer.hasSublayers():
                    _item = gtk.MenuItem("Delete")
                    _item.connect("activate", self._deleteLayer)
                    _menu.append(_item)
        _item = gtk.MenuItem("Clear Layer")
        _item.connect("activate", self._clearLayer)
        _menu.append(_item)
        _menu.show_all()
        self.__layer = _layer
        return _menu

    def _treeViewKeyPress(self, widget, event):
        print "_treeViewKeyPress() ..."
        _key = event.keyval
        if (_key == gtk.keysyms.Page_Up or
            _key == gtk.keysyms.Page_Down or
            _key == gtk.keysyms.Left or
            _key == gtk.keysyms.Right or
            _key == gtk.keysyms.Up or
            _key == gtk.keysyms.Down):
            print "Got special key"
            pass # need to handle adjusting view in drawing area
        elif _key == gtk.keysyms.Tab:
            # print "got tab key ..."
            self.getDA().grab_focus()
            return True
        elif _key == gtk.keysyms.Escape:
            self.reset()
        else:
            self.__entry.grab_focus()
            return self.__entry.event(event)
        return False

    def _selectionChanged(self, selection):
        if selection is not None:
            _model, _iter = selection.get_selected()
            if _iter is not None:
                _layer = _model.get_value(_iter, 1)
                self.__image.setActiveLayer(_layer)

    def _cellEdited(self, cell_renderer, path, text):
        _model = self.__model
        _iter = _model.get_iter_from_string(path)
        _layer = _model.get_value(_iter, 1)
        _layer.setName(text)
        _model.set(_iter, 0, text)

    def _reordered(self, model, path, iter, new_order):
        print "in reordered()"
        print "model: " + `model`
        print "path: " + `path`
        print "iter: " + `iter`
        print "new_order: " + `new_order`

    def _modelFindLayer(iter=None):
        print "_modelFindLayer() ..."
        _model = self.__model
        _iter = iter
        if _iter is None:
            _iter = _model.get_iter_first()
        _layer = self.__layer
        _path = None
        _mlayer = _model.get_value(_iter, 1)
        if _mlayer is _layer:
            _path = model.get_path(_iter)
        else:
            if _model.iter_has_child(_iter):
                _child = _model.iter_children(_iter)
                while _child is not None:
                    _path = self._modelFindLayer(_child)
                    if _path is not None:
                        break
                    _child = _model.iter_next(_child)
        return _path

    def _modelAddLayer(self, model, path, iter):
        print "_modelAddLayer() ..."
        _layer = self.__layer
        _mlayer = model.get_value(iter, 1)
        _val = False
        if _mlayer is _layer.getParentLayer():
            _iter = model.append(iter)
            model.set(_iter, 0, _layer.getName())
            model.set(_iter, 1, _layer)
            _val = True
        return _val

    def _addedLayer(self, obj, *args):
        print "_addedLayer()"
        _model = self.__model
        self.__layer = obj
        _model.foreach(self._modelAddLayer)
        _path = self._modelFindLayer()
        if _path is None:
            raise ValueError, "Layer not found in model!"
        _ppath = _path[:-1]
        _tv = self.__treeview
        while len(_ppath):
            if not _tv.row_expanded(_ppath):
                _tv.expand_row(_ppath, False)
            _ppath = _ppath[:-1]
        _tv.get_selection().select_path(_path)
        obj.connect('visiblity_changed', self._layerVisibilityChanged)

    def _modelDeleteLayer(self, model, path, iter):
        print "_modelDeleteLayer() ..."
        _mlayer = model.get_value(iter, 1)
        _val = False
        if _mlayer is self.__layer:
            model.remove(iter)
            _val = True
        return _val

    def _deletedLayer(self, obj, *args):
        print "_deletedLayer()"
        _model = self.__model
        _path = self._modelFindLayer()
        if _path is None:
            raise ValueError, "Layer not found in model!"
        _ppath = _path[:-1]
        _tv = self.__treeview
        while len(_ppath):
            if not _tv.row_expanded(_ppath):
                _tv.expand_row(_ppath, False)
            _ppath = _ppath[:-1]
        _tv.get_selection().select_path(_path[:-1])
        _model.foreach(self._modelDeleteLayer)
        obj.disconnect(self)

    def _activeLayerChanged(self, obj, *args):
        print "_activeLayerChanged()"
        self.__layer = obj
        _path = self._modelFindLayer()
        if _path is None:
            raise ValueError, "Layer not found in model!"
        _ppath = _path[:-1]
        _tv = self.__treeview
        while len(_ppath):
            if not _tv.row_expanded(_ppath):
                _tv.expand_row(_ppath, False)
            _ppath = _ppath[:-1]
        _tv.get_selection().select_path(_path)

    def _layerVisibilityChanged(self, obj, *args):
        print "_layerVisibilityChanged()"
        for _obj in obj.getLayerEntities('point'):
            if _obj.isVisible():
                _obj.sendMessage('refresh')
        _ctypes = ['hcline', 'vcline', 'acline', 'cline', 'ccircle']
        for _ctype in _ctypes:
            for _obj in obj.getLayerEntities(_ctype):
                if _obj.isVisible():
                    _obj.sendMessage('refresh')
        _gtypes = ['segment', 'circle', 'arc', 'leader', 'polyline',
                   'chamfer', 'fillet', 'textblock', 'linear_dimension',
                   'horizontal_dimension', 'vertical_dimension',
                   'radial_dimension', 'angular_dimension']
        for _gtype in _gtypes:
            for _obj in obj.getLayerEntities(_gtype):
                if _obj.isVisible():
                    _obj.sendMessage('refresh')

    def _setLayerVisibility(self, menuitem, vis):
        _image = self.__image
        _image.startAction()
        try:
            self.__layer.setVisibility(vis)
        finally:
            _image.endAction()
        return False

    def _setChildrenVisibility(self, menuitem, vis):
        _image = self.__image
        _image.startAction()
        try:
            _layers = self.__layer.getSublayers()
            while len(_layers):
                _layer = _layers.pop()
                _layer.setVisibility(vis)
                _layers.extend(_layer.getSublayers())
        finally:
            _image.endAction()
        return False

    def _deleteLayer(self, menuitem):
        _image = self.__image
        _layer = self.__layer
        _image.startAction()
        try:
            _image.delLayer(_layer)
            _layer.clear()
        finally:
            _image.endAction()
        self.__layer = None
        return False

    def _clearLayer(self, menuitem):
        _image = self.__image
        _image.startAction()
        try:
            self.__layer.clear()
        finally:
            _image.endAction()
        return False

    def _addChildLayer(self, menuitem):
        _window = gtkimage.getWindow()
        _dialog = gtk.Dialog("Add New Child Layer", _window,
                             gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                             (gtk.STOCK_OK, gtk.RESPONSE_OK,
                              gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT))
        _hbox = gtk.HBox(False, 2)
        _hbox.set_border_width(2)
        _dialog.vbox.pack_start(_hbox, False, False, 0)
        _label = gtk.Label("Name:")
        _hbox.pack_start(_label, False, False, 0)
        _entry = gtk.Entry()
        _entry.set_text("NewChildLayer")
        _hbox.pack_start(_entry, True, True, 0)
        _dialog.show_all()
        _response = _dialog.run()
        if _response == gtk.RESPONSE_OK:
            _name = _entry.get_text()
            if len(_name):
                _new_layer = layer.Layer(_name)
                _image = self.__image
                _image.startAction()
                try:
                    _image.addChildLayer(_new_layer, self.__layer)
                finally:
                    _image.endAction()
        _dialog.destroy()
        return False

    def _renameLayer(self, menuitem):
        _layer = self.__layer
        _name = _layer.getName()
        _window = gtkimage.getWindow()
        _dialog = gtk.Dialog("Rename Layer", _window,
                             gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                             (gtk.STOCK_OK, gtk.RESPONSE_OK,
                              gtk.STOCK_CANCEL, gtk.RESPONSE_REJECT))
        _hbox = gtk.HBox(False, 2)
        _hbox.set_border_width(2)
        _dialog.vbox.pack_start(_hbox, False, False, 0)
        _label = gtk.Label("Name:")
        _hbox.pack_start(_label, False, False, 0)
        _entry = gtk.Entry()
        _entry.set_text(_name)
        _hbox.pack_start(_entry, True, True, 0)
        _dialog.show_all()
        _response = _dialog.run()
        if _response == gtk.RESPONSE_OK:
            _new_name = _entry.get_text()
            if len(_new_name):
                _image = self.__image
                _image.startAction()
                try:
                    _layer.setName(_new_name)
                finally:
                    _image.endAction()
        _dialog.destroy()
        return False


    def close(self):
        self.__image.disconnect(self)

    def getWindow(self):
        return self.__sw

class ImageView(object):
    def __init__(self, im, xmin=0.0, ymax=0.0, upp=1.0):
        if not isinstance(im, image.Image):
            raise TypeError, "Invalid Image type: " + `type(im)`
        _xmin = util.get_float(xmin)
        _ymax = util.get_float(ymax)
        _upp = util.get_float(upp)
        if not _upp > 0.0:
            raise ValueError, "Invalid units-per-pixel value: %g" % _upp
        self.__image = im
        self.__pixmap = None
        self.__gc = None
        self.__width = None
        self.__height = None
        _da = gtk.DrawingArea()
        _black = gtk.gdk.color_parse('black')
        _da.modify_fg(gtk.STATE_NORMAL, _black)
        _da.modify_bg(gtk.STATE_NORMAL, _black)
        _da.set_flags(gtk.CAN_FOCUS)

        _da.connect("event", self._generalEvent)
        _da.connect("expose_event", self._exposeEvent)
        _da.connect("realize", self._realizeEvent)
        _da.connect("configure_event", self._configureEvent)
        _da.connect("focus_in_event", self._focusChangedEvent, True)
        _da.connect("focus_out_event", self._focusChangedEvent, False)

        _da.set_events(gtk.gdk.EXPOSURE_MASK |
                       gtk.gdk.LEAVE_NOTIFY_MASK |
                       gtk.gdk.BUTTON_PRESS_MASK |
                       gtk.gdk.BUTTON_RELEASE_MASK |

                       gtk.gdk.ENTER_NOTIFY_MASK|
                       gtk.gdk.LEAVE_NOTIFY_MASK|
                       gtk.gdk.KEY_PRESS_MASK |
                       gtk.gdk.KEY_RELEASE_MASK |
                       gtk.gdk.FOCUS_CHANGE_MASK |
                       gtk.gdk.POINTER_MOTION_MASK)
        self.__da = _da
        self.__xmin = _xmin
        self.__ymax = _ymax
        self.__upp = _upp
        self.__view = None
        im.connect('added_object', self._addedObject)
        im.connect('deleted_object', self._deletedObject)
        im.connect('modified', self._objectModified)

    def _realizeEvent(self, widget):
        _win = widget.window
        _w, _h = _win.get_size()
        self.__width = _w
        self.__height = _h
        widget.set_size_request(100,100)
        _gc = _win.new_gc()
        _gc.set_exposures(True)
        self.__gc = _gc

    def _exposeEvent(self, widget, event):
        _pixmap = self.__pixmap
        _x, _y, _w, _h = event.area
        _gc = widget.get_style().fg_gc[gtk.STATE_NORMAL]
        # _gc.set_function(gtk.gdk.COPY)
        widget.window.draw_drawable(_gc, _pixmap, _x, _y, _x, _y, _w, _h)
        return True

    def _configureEvent(self, widget, event):
        _win = widget.window
        _w, _h = _win.get_size()
        if self.__width != _w or self.__height != _h:
            self.__view = None
            self.__width = _w
            self.__height = _h
            _pixmap = gtk.gdk.Pixmap(_win, _w, _h)
            _gc = widget.get_style().fg_gc[gtk.STATE_NORMAL]
            _pixmap.draw_rectangle(_gc, True, 0, 0, _w, _h)
            self.__pixmap = _pixmap
            # _xmin, _ymin = gtkimage.getView()[:2]
            # if _xmin is None or _ymin is None:
                # _xmin = 1.0
                # _ymin = 1.0
            # _upp = gtkimage.getUnitsPerPixel()
            # gtkimage.setView(_xmin, _ymin, _upp)
        return True

    def _focusChangedEvent(self, widget, event, flag):
        return False

    def _generalEvent(self, widget, event):
        _type = event.type
        _tool = gtkimage.getTool()
        if _type == gtk.gdk.BUTTON_PRESS:
            set_toolpoint(gtkimage, event)
            button = event.button
            if button == 1:
                if _tool is not None and _tool.hasHandler("button_press"):
                    handler = _tool.getHandler("button_press")
                    handler(gtkimage, widget, event, _tool)
            elif button == 2:
                # print "button2 press event"
                pass
            elif button == 3:
                # print "button3 press event"
                pass
        elif _type == gtk.gdk.BUTTON_RELEASE:
            set_toolpoint(gtkimage, event)
            button = event.button
            if button == 1:
                if _tool is not None and _tool.hasHandler("button_release_event"):
                    handler = _tool.getHandler("button_release_event")
                    handler(gtkimage, widget, event, _tool)
        elif _type == gtk.gdk.MOTION_NOTIFY:
            set_toolpoint(gtkimage, event)
            if _tool is not None and _tool.hasHandler("motion_notify"):
                handler = _tool.getHandler("motion_notify")
                handler(gtkimage, widget, event, _tool)
        elif _type == gtk.gdk.KEY_PRESS:
            _key = event.keyval
            if (_key == gtk.keysyms.Page_Up or
                _key == gtk.keysyms.Page_Down or
                _key == gtk.keysyms.Left or
                _key == gtk.keysyms.Right or
                _key == gtk.keysyms.Up or
                _key == gtk.keysyms.Down):
                # print "Got special key"
                pass # handle moving the drawing in some fashion ...
            elif _key == gtk.keysyms.Escape:
                gtkimage.reset()
            else:
                _entry = gtkimage.getEntry()
                _entry.grab_focus()
                if _key == gtk.keysyms.Tab:
                    return True
                else:
                    return _entry.event(event)
        elif _type == gtk.gdk.ENTER_NOTIFY:
            set_toolpoint(gtkimage, event)
        elif _type == gtk.gdk.LEAVE_NOTIFY:
            set_toolpoint(gtkimage, event)
        else:
            # print "Got type %d" % _type
            pass
        return False

    def _drawObject(self, obj, col=None):
        _col = col
        _xmin, _ymin, _xmax, _ymax = self.getView()
        if obj.inRegion(_xmin, _ymin, _xmax, _ymax):
            _image = self.__image
            if _col is None:
                if obj.getParent() is not _image.getActiveLayer():
                    _col = _image.getOption('INACTIVE_LAYER_COLOR')
            obj.draw(self, _col)

    def _addedObject(self, obj, *args):
        print "_addedObject()"
        _layer = obj.getParent()
        if _layer is None:
            raise ValueError, "Object has no parent: " + `obj`
        _image = _layer.getParent()
        if _image is not self.__image:
            raise ValueError, "Object parent not in Image: " + `_parent`
        obj.connect('modified', self._objModified)
        obj.connect('visibility_changed', self._objVisibilityChanged)
        obj.connect('refresh', self._refreshObject)
        if obj.isVisible() and _layer.isVisible():
            self._drawObject(obj)

    def _deletedObject(self, obj, *args):
        print "_deletedObject()"
        if obj.isVisible() and obj.getParent().isVisible():
            self._drawObject(obj, self.__image.getOption('BACKGROUND_COLOR'))
        obj.disconnect(self)

    def _objModified(self, obj, *args):
        if obj.isVisible() and obj.getParent().isVisible():
            self._drawObject(obj)

    def _objVisibilityChanged(self, obj, *args):
        print "_objVisibilityChanged()"
        if obj.getParent().isVisible():
            _col = None
            if not obj.isVisible():
                _col = self.__image.getOption('BACKGROUND_COLOR')
            self._drawObject(obj, _col)

    def _refreshObject(self, obj, *args):
        print "_refreshObject()"
        _col = None
        if not obj.isVisible() or not obj.getParent().isVisible():
            _col = self.__image.getOption('BACKGROUND_COLOR')
        self._drawObject(obj, _col)

    def getView(self):
        """Get the region of space the ImageView draws

getRegion()

This method returns a tuple of four values: xmin, ymin, xmax, ymax
        """
        if self.__view is None:
            _xmin = self.__xmin
            _ymax = self.__ymax
            _upp = self.__upp
            _w = self.__width
            _h = self.__height
            _xmax = _xmin + (_upp * _w)
            _ymin = _ymax - (_upp * _h)
            self.__view = (_xmin, _ymin, _xmax, _ymax)
        return self.__view

    def getPixmap(self):
        """Return the gtk.gdk.Pixmap for this ImageView.

getPixmap()
        """
        return self.__pixmap

    def getUnitsPerPixel(self):
        """Return the units-per-pixel scaling factor for the ImageView.

getUnitsPerPixel()
        """
        return self.__upp

    def setUnitsPerPixel(self, upp):
        """Set the units-per-pixel scaling factor for the ImageView.

setUnitsPerPixel(upp)

Argument 'upp' should be a float value greater
        """
        _upp = util.get_float(upp)
        if not _upp > 0.0:
            raise ValueError, "Invalid units-per-pixel value: %g" % _upp
        self.__upp = _upp
        self.__view = None
        # self.redraw()

    def pixelsToCoords(self, x, y):
        """Convert from pixel coordinates to x-y coordinates.

pixelsToCoords(x, y)

Arguments 'x' and 'y' should be positive integers. This method
returns a tuple of two float values
        """
        if not isinstance(x, int):
            raise TypeError, "Invalid 'x' type: " + `type(x)`
        if x < 0:
            raise ValueError, "Invalid 'x' value: %d" % x
        if not isinstance(y, int):
            raise TypeError, "Invalid 'y' type: " + `type(y)`
        if y < 0:
            raise ValueError, "Invalid 'y' value: %d" % y
        _upp = self.__upp
        _xc = self.__xmin + (x * _upp)
        _yc = self.__ymax - (y * _upp)
        return _xc, _yc

    def coordsToPixels(self, x, y):
        """Convert from x-y coordinates to pixel coordinates

coordsToPixels(x, y)

Arguments 'x' and 'y' should be float values. This method
returns a tuple holding two integer values
        """
        _x = util.get_float(x)
        _y = util.get_float(y)
        _upp = self.__upp
        _xp = int((x - self.__xmin)/_upp)
        _yp = int((self.__ymax - y)/_upp)
        return _xp, _yp

    def close(self):
        self.__image.disconnect(self)

    def redraw(self):
        """This method draws all the objects visible in the ImageView.

redraw()
        """
        _da = self.__da
        if (_da.flags() & gtk.MAPPED):
            _image = self.__image
            _w = self.__width
            _h = self.__height
            _gc = _da.get_style().fg_gc[gtk.STATE_NORMAL]
            self.__pixmap.draw_rectangle(_gc, True, 0, 0, _w, _h)
            _active = _image.getActiveLayer()
            _layers = [_image.getTopLayer()]
            while (len(_layers)):
                _layer = _layers.pop()
                if _layer is not _active:
                    self.drawLayer(_layer)
                _layers.extend(_layer.getSublayers())
            self.drawLayer(_active)
            _da.queue_draw() # generate an expose event

    def drawLayer(self, l):
        if not isinstance(l, layer.Layer):
            raise TypeError, "Invalid layer type: " + `type(l)`
        if l.getParent() is not self.__image:
            raise ValueError, "Layer not found in Image"
        if l.isVisible():
            _xmin = self.__xmin
            _ymax = self.__ymax
            _upp = self.__upp
            _xmax = _xmin + (_upp * _w)
            _ymin = _ymax - (_upp * _h)
            _col = self.getOption('INACTIVE_LAYER_COLOR')
            if l is self.getActiveLayer():
                _col = None
            _cobjs = []
            _objs = []
            _pts = []
            for _obj in l.objsInRegion(_xmin, _ymin, _xmax, _ymax):
                if _obj.isVisible():
                    if isinstance(_obj, Point):
                        _pts.append(_obj)
                    elif isinstance(_obj, ConstructionObject):
                        _cobjs.append(_obj)
                    else:
                        _objs.append(_obj)
            for _obj in _pts:
                _obj.draw(self, _col)
            for _obj in _cobjs:
                _obj.draw(self, _col)
            for _obj in _objs:
                _obj.draw(self, _col)


class ImageWindow(object):
    def __init__(self, im):
        if not isinstance(im, image.Image):
            raise TypeError, "Invalid Image type: " + `type(im)`
        self.__image = im
        #
        # Display window
        #
        _window = gtk.Window()
        _window.set_title("Untitled")
        _window.connect("destroy", self._closeImage)
        _width = min(1024, int(0.8 * float(gtk.gdk.screen_width())))
        _height = min(768, int(0.8 * float(gtk.gdk.screen_height())))
        _window.set_default_size(_width, _height)
        #
        _vbox = gtk.VBox(False, 2)
        _vbox.set_border_width(2)
        _window.add(_vbox)
        self.__ldisp = LayerDisplay(im)

    def getImage(self):
        """Return the Image stored in the ImageDisplay.

getImage()
        """
        return self.__image

    image = propery(getImage, None, None, "The Image used for visualizing.")

    def closeImage(self, widget):
        """Close a window containing a Image.

closeImage()
        """
        _image = self.__image
        _image.close()
        _image.disconnect(self)
        self.__ldisp(close)
        for _i in xrange(len(globals.imagelist)):
            _gimage = globals.imagelist[_i]
            if _image is _gimage:
                del globals.imagelist[_i]
                _gimage.window.destroy()
                if not len(globals.imagelist):
                    gtk.main_quit()
                break
        return False
