#
# Copyright (c) 2002, 2003, 2004 Art Haas
#
# This file is part of PythonCAD.
# 
# PythonCAD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# PythonCAD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with PythonCAD; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# classes for graphic objects
#

import warnings

import style
import color
import linetype
import baseobject
import logger

class old_GraphicObject(object):
    """A class representing an object that is drawn.

This class is meant to be a base class for things like
line segments, circles, etc. The class has one attribute
that cannot be None:

style: The Style object

There are three other attributes that override values in
the style attribute:

color: A Color object
thickness: A positive float value
linetype: A Linetype object

A GraphicObject object has the following methods:

{get/set}Style(): Get/Set the Style of a GraphicObject.
{get/set}Color(): Get/Set the Color of a GraphicObject.
{get/set}Thickness(): Get/Set the thickness of a GraphicObject.
{get/set}Linetype(): Get/Set the Linetype of a GraphicObject.
inRegion(): Returns whether or not the GraphicObject is within a bounded area.
    """

    __defstyle = style.Style(u'Default Style',
                             linetype.Linetype(u'Solid', None),
                             color.Color(0xffffff),
                             1.0)

    def __init__(self, st=None, lt=None, col=None, t=None):
        """Initialize a GraphicObject.

GraphicObject([st, lt, col, t])

Optional arguments:
st: A style object that overrides the class default
lt: A Linetype object that overrides the value in the Style
col: A Color object that overrides the value in the Style
t: A positive float that overrides the value in the Style
        """
        _st = st
        if _st is not None:
            if not isinstance(_st, style.Style):
                raise TypeError, "Invalid style: " + `_st`
        _col = col
        if _col is not None:
            if not isinstance(_col, color.Color):
                _col = color.Color(col)
        _lt = lt
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
        _t = t
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
        self.__style = _st
        self.__color = _col
        self.__linetype = lt
        self.__thickness = _t
        
        
    def getStyle(self):
        """Get the Style of the GraphicObject.

getStyle()

Returns the current Style of an object. The values in the
style may be overriden if any of the color, linetype, or
thickness attributes are not None.
        """
        _style = self.__style
        if _style is None:
            _style = old_GraphicObject.__defstyle
        return _style

    def setStyle(self, s):
        """Set the Style of the Object.

setStyle([s])

Setting the style of a GraphicObject unsets any overrides for
the object. Setting the Style to None restores the default
style.
        """
        _s = s
        if _s is not None:
            if not isinstance(_s, style.Style):
                raise TypeError, "Invalid style: " + `_s`
        self.__style = _s

    style = property(getStyle, setStyle, None, "Object Style.")

    def getColor(self):
        """Get the Color of the Object.

getColor()
        """
        _color = self.__color
        if _color is None:
            if self.__style is not None:
                _color = self.__style.getColor()
        if _color is None:
            _color = old_GraphicObject.__defstyle.getColor()
        return _color

    def setColor(self, c=None):
        """Set the color of the object.

setColor([c])

Setting the color overrides the value in the Style. Setting
the color to None or invoking this method without arguments
restores the color value defined in the Style.
        """
        _c = c
        if _c is not None:
            if not isinstance(_c, color.Color):
                _c = color.Color(c)
        self.__color = _c
        
    color = property(getColor, setColor, None, "Object color.")
    
    def getThickness(self):
        """Get the thickness of the GraphicObject.

getThickness()
        """
        _th = self.__thickness
        if _th is None:
            if self.__style is not None:
                _th = self.__style.getThickness()
        if _th is None:
            _th = old_GraphicObject.__defstyle.getThickness()
        return _th

    def setThickness(self, t=None):
        """Set the thickness of a object.

setThickness([t])

Setting the thickness overrides the value in the Style. Setting
the thickness to None, or invoking this method without an
argument, restores the thickness value defined in the Style.
        """
        _t = t
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
        self.__thickness = _t

    thickness = property(getThickness, setThickness, None, "Object thickness.")
    
    def getLinetype(self):
        """Get the Linetype of the object.

getLinetype()
        """
        _lt = self.__linetype
        if _lt is None:
            if self.__style is not None:
                _lt = self.__style.getLinetype()
        if _lt is None:
            _lt = old_GraphicObject.__defstyle.getLinetype()
        return _lt

    def setLinetype(self, lt=None):
        """Set the Linetype of the GraphicObject.

setLinetype([lt])

Setting the Linetype overrides the value in the Sytle. Setting
the Linetype to None or invoking this method without arguments
restores the linetype value defined in the Style.
        """
        _lt = lt
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
        self.__linetype = _lt

    linetype = property(getLinetype, setLinetype, None, "Object Linetype.")

    def inRegion(self, xmin, ymin, xmax, ymax, fully=False):
        """Return whether or not an object falls within a boundary.

isRegion(xmin, ymin, xmax, ymax[, fully])

The first four arguments define the boundary. The optional
fifth argument "fully" indicates whether or not the object
must be completely contained within the region or just pass
through it.

This method should be overriden in classes derived from GraphicObjects.
        """
        return False

class GraphicObject(baseobject.Subpart):
    """A class representing an object that is drawn.

This class is meant to be a base class for things like
line segments, circles, etc. The GraphicObject class
has one attribute that cannot be None:

style: The Style object

There are three other attributes that override values in
the style attribute:

color: A Color object
thickness: A positive float value
linetype: A Linetype object

A GraphicObject object has the following methods:

{get/set}Style(): Get/Set the Style of a GraphicObject.
{get/set}Color(): Get/Set the Color of a GraphicObject.
{get/set}Thickness(): Get/Set the thickness of a GraphicObject.
{get/set}Linetype(): Get/Set the Linetype of a GraphicObject.
    """
    messages = {
    'attribute_changed' : True,
    }
    __defstyle = style.Style(u'Default Style',
                             linetype.Linetype(u'Solid', None),
                             color.Color(0xffffff),
                             1.0)

    def __init__(self, st=None, lt=None, col=None, t=None, **kw):
        """Initialize a GraphicObject.

GraphicObject([st, lt, col, t])

Optional arguments:
style: A style object that overrides the class default
linetype: A Linetype object that overrides the value in the Style
color: A Color object that overrides the value in the Style
thickness: A positive float that overrides the value in the Style
        """
        baseobject.Subpart.__init__(self, **kw)
        _st = st
        if _st is None:
            _st = GraphicObject.__defstyle
        else:
            if not isinstance(_st, style.Style):
                raise TypeError, "Invalid style: " + `_st`
        _col = _lt = _t = None
        if 'color' in kw:
            _col = kw['color']
        else:
            if 'col' in kw:
                warnings.warn("keyword 'col' is deprecated;  use 'color'",
                DeprecationWarning, stacklevel=2)
                _col = kw['col']
        if _col is not None:
            if not isinstance(_col, color.Color):
                _col = color.Color(_col)
            if _col == _st.getColor():
                _col = None
        _col = col
        if _col is not None:
            if not isinstance(_col, color.Color):
                _col = color.Color(col)
            if _col == _st.getColor():
                _col = None
        if 'linetype' in kw:
            _lt = kw['linetype']
        else:
            if 'lt' in kw:
                warnings.warn("keyword 'lt' is deprecated;  use 'linetype'",
                              DeprecationWarning, stacklevel=2)
                _lt = kw['lt']
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
            if _lt == _st.getLinetype():
                _lt = None
        _lt = lt
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
            if _lt == _st.getLinetype():
                _lt = None
        if 'thickness' in kw:
            _t = kw['thickness']
        else:
            if 'th' in kw:
                warnings.warn("keyword 'th' is deprecated;  use 'thickness'",
                               DeprecationWarning, stacklevel=2)
                _t = kw['th']
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(_t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
            if abs(_t - _st.getThickness()) < 1e-10:
                _t = None
        _t = t
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
            if abs(_t - _st.getThickness()) < 1e-10:
                _t = None
        self.__style = _st
        self.__color = _col
        self.__linetype = _lt
        self.__thickness = _t
        
    def getStyle(self):
        """Get the Style of the GraphicObject.

getStyle()

Returns the current Style of an object. The values in the
style may be overriden if any of the color, linetype, or
thickness attributes are not None.
        """
        _style = self.__style
        if _style is None:
            _style = GraphicObject.__defstyle
        return _style

    def setStyle(self, s):
        """Set the Style of the Object.

setStyle([s])

Setting the style of a GraphicObject unsets any overrides for
the object. Setting the Style to None restores the default
style.
        """
        if self.isLocked():
            raise RuntimeError, "Style change not allowed - objected locked."
        _s = s
        if _s is not None:
            if not isinstance(_s, style.Style):
                raise TypeError, "Invalid style: " + `_s`
        _cs = self.getStyle()            
        if _cs != _s:
            self.__style = _s
            self.sendMessage('attribute_changed', 'style', _cs)
            self.modified()

    style = property(getStyle, setStyle, None, "Object Style.")

    def getColor(self):
        """Get the Color of the Object.

getColor()
        """
        _color = self.__color
        if _color is None:
            if self.__style is not None:
                _color = self.__style.getColor()
        if _color is None:
            _color = GraphicObject.__defstyle.getColor()
        return _color

    def setColor(self, c=None):
        """Set the color of the object.

setColor([c])

Setting the color overrides the value in the Style. Setting
the color to None or invoking this method without arguments
restores the color value defined in the Style.
        """
        if self.isLocked():
            raise RuntimeError, "Color change not allowed - object locked."
        _c = c
        if _c is not None:
            if not isinstance(_c, color.Color):
                _c = color.Color(c)
        _oc = self.getColor()                
        if _oc != _c:
            self.__color = _c
            self.sendMessage('attribute_changed', 'color', _oc)
            self.modified()
        
    color = property(getColor, setColor, None, "Object color.")
    
    def getThickness(self):
        """Get the thickness of the GraphicObject.

getThickness()
        """
        _th = self.__thickness
        if _th is None:
            if self.__style is not None:
                _th = self.__style.getThickness()
        if _th is None:
            _th = GraphicObject.__defstyle.getThickness()
        return _th

    def setThickness(self, t=None):
        """Set the thickness of a object.

setThickness([t])

Setting the thickness overrides the value in the Style. Setting
the thickness to None, or invoking this method without an
argument, restores the thickness value defined in the Style.
        """
        if self.isLocked():
            raise RuntimeError, "Thickness change not allowed - object locked."
        _t = t
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
        _ot = self.getThickness()            
        if ((_t is None and _ot is not None) or
            (_t is not None and _ot is None) or
            (_t is not None and _ot is not None and abs(_ot - _t) > 1e-10)):
            self.__thickness = _t
            self.sendMessage('attribute_changed', 'thickness', _ot)
            self.modified()

    thickness = property(getThickness, setThickness, None, "Object thickness.")
    
    def getLinetype(self):
        """Get the Linetype of the object.

getLinetype()
        """
        _lt = self.__linetype
        if _lt is None:
            if self.__style is not None:
                _lt = self.__style.getLinetype()
        if _lt is None:
            _lt = GraphicObject.__defstyle.getLinetype()
        return _lt

    def setLinetype(self, lt=None):
        """Set the Linetype of the GraphicObject.

setLinetype([lt])

Setting the Linetype overrides the value in the Sytle. Setting
the Linetype to None or invoking this method without arguments
restores the linetype value defined in the Style.
        """
        if self.isLocked():
            raise RuntimeError, "Linetype change not allowed - object locked."
        _lt = lt
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
        _ol = self.getLinetype()            
        if _ol != _lt:
            self.__linetype = _lt
            self.sendMessage('attribute_changed', 'linetype', _ol)
            self.modified()

    linetype = property(getLinetype, setLinetype, None, "Object Linetype.")

    def getGraphicValues(self, defstyle=None):
        if defstyle is not None and not isinstance(defstyle, style.Style):
            raise TypeError, "Invalid default style: " + `defstyle`
        _s = _c = _l = _t = None        
        _ss = self.__style        
        if defstyle is None or defstyle != _ss:
            _s = _ss.getStyleValues()
        _sc = self.__color
        if _sc is not None:
            _c = str(_sc)
        _sl = self.__linetype
        if _sl is not None:
            _l = (_sl.getName(), _sl.getList())
        _st = self.__thickness
        if _st is not None:
            _t = self.__thickness
        return _s, _l, _c, _t
        
    def sendsMessage(self, m):
        if m in GraphicObject.messages:
            return True
        return baseobject.Subpart.sendsMessage(self, m)

#
# GraphicObject history class
#

class GraphicObjectLog(logger.Logger):
    def __init__(self, obj):
        if not isinstance(obj, GraphicObject):
            raise TypeError, "Invalid GraphicObject: " + `obj`
        logger.Logger.__init__(self)
        self.__obj = obj
        obj.connect('attribute_changed', self, GraphicObjectLog.attrChanged)

    def detatch(self):
        self.__obj.disconnect(self)
        self.__obj = None

    def getObject(self):
        return self.__obj
    
    def attrChanged(self, obj, *args):
        _alen = len(args)
        if len(args) < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _attr = args[0]
        if not isinstance(_attr, str):
            raise TypeError, "Unexpected attribute string: " + str(_attr)
        _v = args[1]
        if _attr == 'linetype':
            if not isinstance(_v, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_v`
            _val = (_v.getName(), _v.getList())
        elif _attr == 'color':
            if not isinstance(_v, color.Color):
                raise TypeError, "Invalid color: " + `_v`
            _val = (_v.r, _v.g, _v.b)
        elif _attr == 'style':
            if not isinstance(_v, style.Style):
                raise TypeError, "Invalid style: " + `_v`
            _l = _v.getLinetype.getList()
            _c = str(_v.getColor())
            _t = _v.getThickness()
            _val = (_l, _c, _t)
        elif _attr == 'thickness':
            _val = _v
            if not isinstance(_val, float):
                _val = float(_v)
        else:
            raise ValueError, "Unexpected attribute: %s" % _attr
        self.saveUndoData('attr_changed', _attr, _val)

    def execute(self, action, *args):
        if action != 'undo' and action != 'redo':
            raise ValueError, "Unexpected action: %s" % action
        _alen = len(args)
        if len(args) == 0:
            raise ValueError, "No arguments to execute()"
        _obj = self.__obj
        _op = args[0]
        if _op == 'attr_changed':        
            if len(args) < 3:
                raise ValueError, "Invalid argument count: %d" % _alen
            _attr = args[1]
            _val = args[2]
            if _attr == 'linetype':
                _sv = (_obj.getName(), _obj.getList())
            elif _attr == 'color':
                _col = _obj.getColor()
                _sv = (_col.r, _col.g, _col.b)
            elif _attr == 'style':
                _l = _obj.getLinetype.getList()
                _c = str(_obj.getColor())
                _t = _obj.getThickness()
                _sv = (_l, _c, _t)
            elif _attr == 'thickness':
                _sv = _obj.getThickness()
            else:
                raise ValueError, "Unexpected attribute: %s" % _attr
            if action == 'undo':
                self.saveRedoData(_op, _attr, _sv)
            else:
                self.saveUndoData(_op, _attr, _sv)
            self.ignore('attribute_changed')
            try:
                if _attr == 'style':
                    pass # fixme
                elif _attr == 'linetype':
                    _name, _list = _val
                    _lt = linetype.get_linetype_by_dashes(_list)
                    if undo:
                        _obj.startUndo(True)
                        try:
                            _obj.setLinetype(_lt)
                        finally:
                            _obj.endUndo()
                    else:
                        _obj.setLinetype(_lt)
                elif _attr == 'color':
                    _r, _g, _b = _val
                    _col = color.get_color(_r, _g, _b)
                    if undo:
                        _obj.startUndo(True)
                        try:
                            _obj.setColor(_col)
                        finally:
                            _obj.endUndo()
                    else:
                        _obj.setColor(_col)
                elif _attr == 'thickness':
                    if undo:
                        _obj.startUndo(True)
                        try:
                            _obj.setThickness(_val)
                        finally:
                            _obj.endUndo()
                    else:
                        _obj.setThickness(_val)
                else:
                    raise ValueError, "Unexpected attribute: %s" % _attr
            finally:
                self.receive('attribute_changed')
        else:
            raise ValueError, "Unexpected operation: %s" % _op
