#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (C) 2010-2011 by Mike Gabriel <mike.gabriel@das-netzwerkteam.de>
#
# Python X2go is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# Python X2go is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the
# Free Software Foundation, Inc.,
# 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

"""\
Default variables and values for Python X2go.

"""
__NAME__ = 'x2godefaults-pylib'

import sys
import os
import paramiko
import platform

## X2go imports
import utils

##
## Common X2go defaults
##

X2GOCLIENT_OS = platform.system()

LOCAL_HOME = os.path.expanduser('~')
X2GO_SESSIONS_ROOTDIR = '.x2go'
X2GO_CLIENT_ROOTDIR = '.x2goclient'
X2GO_SSH_ROOTDIR = os.path.join('.x2go','.ssh')

# setting OS dependent variables
if X2GOCLIENT_OS == "Windows":
    # on Windows we will use the current directory as ,,ROOTDIR'' which 
    # will normally be the application directory
    ROOT_DIR = os.path.abspath(os.path.curdir)
    ETC_DIR = os.path.join(ROOT_DIR, 'etc')
    import win32api
    CURRENT_LOCAL_USER = win32api.GetUserName()
    X2GO_SSH_ROOTDIR = '.ssh'
    SUPPORTED_SOUND = True
    SUPPORTED_PRINTING = True
    SUPPORTED_FOLDERSHARING = True
    SUPPORTED_MIMEBOX = True

elif X2GOCLIENT_OS == "Linux":
    ROOT_DIR = '/'
    ETC_DIR = os.path.join(ROOT_DIR, 'etc', 'x2goclient')
    import getpass
    CURRENT_LOCAL_USER = getpass.getuser()
    X2GO_SSH_ROOTDIR = '.ssh'
    SUPPORTED_SOUND = True
    SUPPORTED_PRINTING = True
    SUPPORTED_FOLDERSHARING = True
    SUPPORTED_MIMEBOX = True

elif X2GOCLIENT_OS == "Mac":
    ROOT_DIR = '/'
    ETC_DIR = os.path.join(ROOT_DIR, 'etc', 'x2goclient')
    import getpass
    CURRENT_LOCAL_USER = getpass.getuser()
    X2GO_SSH_ROOTDIR = '.ssh'
    SUPPORTED_SOUND = True
    SUPPORTED_PRINTING = True
    SUPPORTED_FOLDERSHARING = True
    SUPPORTED_MIMEBOX = True

else:
    import exceptions
    class OSNotSupportedException(exceptions.StandardError): pass
    raise OSNotSupportedException('Platform %s is not supported' % platform.system())

##
## control and terminal session backend as well as session info and proxy backend defaults
##

BACKENDS_CONTROLSESSION = {
    'STDOUT': 'X2goControlSessionSTDOUT',
}
BACKENDS_TERMINALSESSION = {
    'STDOUT': 'X2goTerminalSessionSTDOUT',
}
BACKENDS_SERVERSESSIONINFO = {
    'STDOUT': 'X2goServerSessionInfoSTDOUT',
}
BACKENDS_SERVERSESSIONLIST = {
    'STDOUT': 'X2goServerSessionListSTDOUT',
}
BACKENDS_PROXY = {
    'NX3': 'X2goProxyNX3',
}

BACKEND_CONTROLSESSION_DEFAULT =  'X2goControlSessionSTDOUT'
BACKEND_TERMINALSESSION_DEFAULT = 'X2goTerminalSessionSTDOUT'
BACKEND_SERVERSESSIONINFO_DEFAULT = 'X2goServerSessionInfoSTDOUT'
BACKEND_SERVERSESSIONLIST_DEFAULT = 'X2goServerSessionListSTDOUT'
BACKEND_PROXY_DEFAULT = 'X2goProxyNX3'

##
## profile backend defaults
##

BACKENDS_SESSIONPROFILES = {
    'FILE': 'X2goSessionProfilesFILE',
    'GCONF': 'X2goSessionProfilesGCONF',
    'HTTPSBROKER': 'X2goSessionProfilesHTTPSBROKER',
    'WINREG': 'X2goSessionProfilesWINREG',
}
"""Python X2go backends for storing session profiles."""
BACKENDS_CLIENTSETTINGS = {
    'FILE': 'X2goClientSettingsFILE',
    'GCONF': 'X2goClientSettingsGCONF',
    'HTTPSBROKER': 'X2goClientSettingsHTTPSBROKER',
    'WINREG': 'X2goClientSettingsWINREG',
}
"""Python X2go backends for storing client settings."""
BACKENDS_CLIENTPRINTING = {
    'FILE': 'X2goClientPrintingFILE',
    'GCONF': 'X2goClientPrintingGCONF',
    'HTTPSBROKER': 'X2goClientPrintingHTTPSBROKER',
    'WINREG': 'X2goClientPrintingWINREG',
}
"""Python X2go backends for storing print settings."""

BACKEND_SESSIONPROFILES_DEFAULT = 'X2goSessionProfilesFILE'
BACKEND_CLIENTSETTINGS_DEFAULT = 'X2goClientSettingsFILE'
BACKEND_CLIENTPRINTING_DEFAULT = 'X2goClientPrintingFILE'

##
## X2go Printing
##

X2GO_SETTINGS_FILENAME = 'settings'
X2GO_SETTINGS_CONFIGFILES = [
    os.path.join(LOCAL_HOME, X2GO_CLIENT_ROOTDIR, 'settings'),
    os.path.join(ETC_DIR,X2GO_SETTINGS_FILENAME),
]
X2GO_PRINTING_FILENAME = 'printing'
X2GO_PRINTING_CONFIGFILES = [
    os.path.join(LOCAL_HOME, X2GO_CLIENT_ROOTDIR, 'printing'),
    os.path.join(ETC_DIR,X2GO_PRINTING_FILENAME),
]
X2GO_SESSIONPROFILES_FILENAME = 'sessions'
X2GO_SESSIONPROFILES_CONFIGFILES = [
    os.path.join(LOCAL_HOME, X2GO_CLIENT_ROOTDIR, 'sessions'),
    os.path.join(ETC_DIR,X2GO_SESSIONPROFILES_FILENAME),
]
X2GO_XCONFIG_FILENAME = 'xconfig'
X2GO_XCONFIG_CONFIGFILES = [
    os.path.join(LOCAL_HOME, X2GO_CLIENT_ROOTDIR, 'xconfig'),
    os.path.join(ETC_DIR,X2GO_XCONFIG_FILENAME),
]

X2GO_CLIENTSETTINGS_DEFAULTS = {
    'LDAP': {
        'useldap': False,
        'port': 389,
        'server': 'localhost',
        'port1': 0,
        'port2': 0,
        },
    'General': {
        # clientport is not needed for Python X2go
        'clientport': 22, 
        'autoresume': True,
        },
    'Authorization': {
        'newprofile': True,
        'suspend': True,
        'editprofile': True,
        'resume': True
        },
    'trayicon': {
        'enabled': True,
        'mintotray': True,
        'noclose': True,
        'mincon': True,
        'maxdiscon': True,
        },
    }
X2GO_CLIENTPRINTING_DEFAULTS = {
    'General': {
        # showdialog will result in a print action that allows opening a print dialog box
        'showdialog': False,
        # if true, open a PDF viewer (or save as PDF file). If false, print via CUPS or print command
        'pdfview': True,
        },
    'print': {
        # If false, print via CUPS. If true, run "command" to process the print job
        'startcmd': False,
        # print command for non-CUPS printing
        'command': 'lpr',
        # ignored in Python X2go
        'stdin': False,
        # ignored in Python X2go
        'ps': False,
        },
    'save': {
        # a path relative to the user's home directory
        'folder': 'PDF',
    },
    'view': {
        # If General->pdfview is true: 
        #   if open is true, the PDF viewer command is executed
        #   if open is false, the incoming print job is saved in ~/PDF folder 
        'open': True,
        # command to execute as PDF viewer
        'command': 'xpdf',
        },
    'CUPS': {
        # default print queue for CUPS, if print queue does not exist, the default 
        # CUPS queue is detected
        'defaultprinter': 'PDF',
        },
    }
if X2GOCLIENT_OS == 'Windows':
    X2GO_CLIENTPRINTING_DEFAULTS['print'].update({'gsprint': os.path.join(os.environ['ProgramFiles'], 'GhostGum', 'gsview', 'gsprint.exe'), })


if X2GOCLIENT_OS == 'Windows':
    X2GO_CLIENTXCONFIG_DEFAULTS = {
        'XServers': {
            'known_xservers': ['VcXsrv', 'Xming', 'Cygwin-X', ],
        },
        'Cygwin-X': {
            'display': 'localhost:40',
            'process_name': 'XWin.exe',
            'test_installed': os.path.join(os.environ['SystemDrive'], '\\', 'cygwin', 'bin', 'XWin.exe'),
            'run_command': os.path.join(os.environ['SystemDrive'], '\\', 'cygwin', 'bin', 'XWin.exe'),
            'parameters': [':40', '-clipboard', '-multiwindow', '-notrayicon', '-nowinkill', '-nounixkill', '-swcursor', ],
            },
        'VcXsrv': {
            'display': 'localhost:40',
            'process_name': 'vcxsrv.exe',
            'test_installed': os.path.join(os.environ['ProgramFiles'], 'VcXsrv', 'vcxsrv.exe'), 
            'run_command': os.path.join(os.environ['ProgramFiles'], 'VcXsrv', 'vcxsrv.exe'),
            'parameters': [':40', '-clipboard', '-multiwindow', '-notrayicon', '-nowinkill', '-nounixkill', '-swcursor', ],
            },
        'Xming': {
            'display': 'localhost:40',
            'process_name': 'Xming.exe',
            'test_installed': os.path.join(os.environ['ProgramFiles'], 'Xming', 'Xming.exe'), 
            'run_command': os.path.join(os.environ['ProgramFiles'], 'Xming', 'Xming.exe'),
            'parameters': [':40', '-clipboard', '-multiwindow', '-notrayicon', '-nowinkill', '-nounixkill', '-swcursor', ],
            },
        }
else:
    # make the variable available when building API documentation with epydoc
    X2GO_CLIENTXCONFIG_DEFAULTS = {}

X2GO_GENERIC_APPLICATIONS = [ 'WWWBROWSER', 'MAILCLIENT', 'OFFICE', 'TERMINAL', ]
"""X2go's generic applications."""

X2GO_SESSIONPROFILE_DEFAULTS = {
    'speed': 2, 'pack': '16m-jpeg', 'quality': 9,
    'iconvto': 'UTF-8', 'iconvfrom': 'UTF-8', 'useiconv': False,
    'usesshproxy': False, 'sshproxyhost': '', 'sshproxyuser': '', 'sshproxytunnel': '', 'sshproxykeyfile': '',
    'useexports': True, 'fstunnel': True, 'export': '',
    'usemimebox': False, 'mimeboxextensions': '', 'mimeboxaction': 'OPEN',
    'fullscreen': False,
    'width': 800,'height': 600,'dpi': 96,'setdpi': False,
    'usekbd':True, 'layout': 'us', 'type': 'pc105/us',
    'sound':False, 'soundsystem': 'pulse', 'startsoundsystem': False, 'soundtunnel':True, 'defsndport':True, 'sndport':4713, 
    'name': 'NEW_PROFILE', 'icon': ':icons/128x128/x2gosession.png', 
    'host': '', 'user': CURRENT_LOCAL_USER, 'key': '', 'sshport': 22,
    'rootless': True, 'applications': str(X2GO_GENERIC_APPLICATIONS), 'command':'TERMINAL',
    'rdpoptions': '-u X2GO_USER -p X2GO_PASSWORD', 'rdpserver': '',
    'print': False,
    'xdmcpserver': 'localhost',
    }
"""L{X2goSessionProfiles} default values to fill a new session profile with."""
##
## X2go Proxy defaults
##

# here is a list of NX 3.x compression methods, this is the "%"-hashed list that
# can also be used for printing in help texts, docs etc.
# The "%"-sign can be replaced by digits 0-9.
pack_methods_nx3 = ['nopack','8','64','256','512','4k','32k','64k','256k','2m','16m',
                '256-rdp','256-rdp-compressed','32k-rdp','32k-rdp-compressed','64k-rdp',
                '64k-rdp-compressed','16m-rdp','16m-rdp-compressed',
                'rfb-hextile','rfb-tight','rfb-tight-compressed',
                '8-tight','64-tight','256-tight','512-tight','4k-tight','32k-tight',
                '64k-tight','256k-tight','2m-tight','16m-tight',
                '8-jpeg-%','64-jpeg','256-jpeg','512-jpeg','4k-jpeg','32k-jpeg',
                '64k-jpeg','256k-jpeg','2m-jpeg','16m-jpeg-%',
                '8-png-jpeg-%','64-png-jpeg','256-png-jpeg','512-png-jpeg','4k-png-jpeg',
                '32k-png-jpeg','64k-png-jpeg','256k-png-jpeg','2m-png-jpeg','16m-png-jpeg-%',
                '8-png-%','64-png','256-png','512-png','4k-Apng',
                '32k-png','64k-png','256k-png','2m-png','16m-png-%',
                '16m-rgb-%','16m-rle-%',]
"""Available NX3 compression methods."""

# use for printing on screen...
pack_methods_nx3_formatted="""
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
    \'%s\'
""" % ('\', \''.join(pack_methods_nx3[0:11]), \
           '\', \''.join(pack_methods_nx3[11:16]), \
           '\', \''.join(pack_methods_nx3[16:19]), \
           '\', \''.join(pack_methods_nx3[19:22]), \
           '\', \''.join(pack_methods_nx3[22:28]), \
           '\', \''.join(pack_methods_nx3[28:32]), \
           '\', \''.join(pack_methods_nx3[32:38]), \
           '\', \''.join(pack_methods_nx3[38:42]), \
           '\', \''.join(pack_methods_nx3[42:47]), \
           '\', \''.join(pack_methods_nx3[47:52]), \
           '\', \''.join(pack_methods_nx3[52:57]), \
           '\', \''.join(pack_methods_nx3[57:62]), \
           '\', \''.join(pack_methods_nx3[62:]))

# _pack_methods_nx3 is the complete list of NX3 pack methods that can be used to check options 
# against
_pack_methods_nx3 = [ m for m in pack_methods_nx3 if "%" not in m ]
for meth in [ m for m in pack_methods_nx3 if "%" in m ]:
    _pack_methods_nx3 += [ meth.replace('%','%s' % str(i)) for i in range(0,10) ]

##
## X2go session defaults
##

X2GO_DESKTOPSESSIONS={
    'KDE': 'startkde',
    'GNOME': 'gnome-session',
    'LXDE': 'startlxde',
    'TRINITY': 'starttrinity',
    'UNITY': 'unity-2d-launcher',
}
"""A dictionary with meta-commands for X2go's window manager sessions."""

##
## X2go SFTP server defaults
##

RSAKEY_STRENGTH = 1024
RSAHostKey = paramiko.RSAKey.generate(RSAKEY_STRENGTH)
"""\
An RSA host key for this client session. Python X2go does not use the
system's host key but generates its own host key for each running
application instance.

"""

X2GO_PRINT_ACTIONS = {
    'PDFVIEW': 'X2goPrintActionPDFVIEW',
    'PDFSAVE': 'X2goPrintActionPDFSAVE',
    'PRINT': 'X2goPrintActionPRINT',
    'PRINTCMD': 'X2goPrintActionPRINTCMD',
    'DIALOG': 'X2goPrintActionDIALOG',
}
"""Relating print action names and classes."""

DEFAULT_PDFVIEW_CMD = 'xdg-open'
"""Default PDF viewer command for Linux systems (PDFVIEW print action)."""
DEFAULT_PDFSAVE_LOCATION = 'PDF'
"""Default location for saving PDF files (PDFSAVE print action)."""
DEFAULT_PRINTCMD_CMD = 'lpr'
"""Default command for the PRINTCMD print action."""

X2GO_MIMEBOX_ACTIONS = {
    'OPEN': 'X2goMIMEboxActionOPEN',
    'OPENWITH': 'X2goMIMEboxActionOPENWITH',
    'SAVEAS': 'X2goMIMEboxActionSAVEAS',
}
"""Relating MIME box action names and classes."""

X2GO_MIMEBOX_EXTENSIONS_BLACKLIST = [
    'LOCK', 'SYS', 'SWP', 
    'EXE', 'COM', 'CMD', 'PS1', 'PS2', 'BAT',
    'JS', 'PY', 'PL', 'SH',
]
"""Black-listed MIME box file extenstions."""

# X2go desktop sharing
X2GO_SHARE_VIEWONLY=0
X2GO_SHARE_FULLACCESS=1
