// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "platform.h"
#include <boost/python/detail/wrap_python.hpp>
#include <queue>
#include <process.h>

namespace visual {

void init_platform() 
{
}

// 2003-02-16 Added explicit casts to unsigned int vice implicit casts from the compiler.
// TODO: Convert time values to integral types at the Python boundary.
void threaded_sleep(double seconds) 
{
	Py_BEGIN_ALLOW_THREADS
	Sleep( (unsigned long)(seconds * 1000.0));
	Py_END_ALLOW_THREADS
}

// 2003-02-16 Added explicit casts to unsigned int vice implicit casts from the compiler.
// TODO: Convert time values to integral types at the Python boundary.
void nothread_sleep(double seconds) 
{
	Sleep( (unsigned long)(seconds * 1000.0));
}

#define liToDouble( li ) ( double(li.LowPart) + double(li.HighPart) * 4294967296.0 )

double sclock() 
{
	static LARGE_INTEGER freq;
	static int clocks_exist = QueryPerformanceFrequency(&freq);
	static double secondsPerTick = clocks_exist ? 1.0 / liToDouble(freq) : 0.0;

	if (clocks_exist) {
		LARGE_INTEGER count;
		QueryPerformanceCounter(&count);

		return liToDouble(count) * secondsPerTick;
	} 
	else {
		return GetTickCount() * 1e-3;
	}
}

/******** mutex implementation ********/

mutex::mutex(int spincount, int _count)
	: count(_count)
{
	#if 0    // requires 98 or NT
	InitializeCriticalSectionAndSpinCount(&sec, spincount);	
	#else
	InitializeCriticalSection(&sec);
	#endif
}

mutex::~mutex() 
{
	DeleteCriticalSection(&sec);
}

/*********** event handling not needed ********/

bool handle_event(PyObject*) 
{
	return false;
}

/*********** threaded_timer implementation *************/

struct callback_data 
{
	callback_data() {}
	bool (*callback)(void*); 
	void* data;
	double when, interval;

	template<class T>
	callback_data( bool (*_callback)(T*), T* _data, double _interval )
		: callback( (bool (*)(void*))_callback ),
		data(_data),
		interval(_interval),
		when(_interval + sclock())
	{
	}

	bool operator ()() { return (*callback)(data); }

	bool operator < ( const callback_data &rhs) const { return when <  rhs.when; }
	bool operator <=( const callback_data &rhs) const { return when <= rhs.when; }
	bool operator > ( const callback_data &rhs) const { return when >  rhs.when; }
	bool operator >=( const callback_data &rhs) const { return when >= rhs.when; }
	bool operator ==( const callback_data &rhs) const { return when == rhs.when; }
	bool operator !=( const callback_data &rhs) const { return when != rhs.when; }
};

struct mainloop 
{
public:
	mainloop() : th(0) 
	{
	}

	void start() 
	{
		if (!th) {
			add_timer( callback_data( &mainloop::process_messages, this, 0.025 ) );
			th=(HANDLE)_beginthread(&mainloop::_start, 0, this);
			SetThreadPriority( th, THREAD_PRIORITY_HIGHEST );
		}
	}

	void add_timer( const callback_data& cb ) 
	{
		mutex::lock L( timerLock);
		timers.push( cb);
	}

	void run() 
	{
		while (1) {
			timerLock.sync_lock();
			callback_data cb = timers.top();
			timerLock.sync_unlock();

			double next = cb.when;
			double now = sclock();
			if (next <= now) {
				timerLock.sync_lock();
				timers.pop();
				timerLock.sync_unlock();

				bool again = cb();
				if (again) 
					add_timer( callback_data(cb.callback, cb.data, cb.interval) );
			} 
			else {
				nothread_sleep( next-now);
			}
		}
	}

private:
	HANDLE th;
	mutex timerLock;
	std::priority_queue< callback_data, std::vector<callback_data>, std::greater<callback_data> > timers;

	static void _start( void* data) 
	{
		((mainloop*)data)->run();
	}

	static bool process_messages( mainloop*) 
	{
		// Process messages for all windows:
		MSG msg;
		int count = 0;
		while ( PeekMessage( &msg, 0, 0, 0, PM_REMOVE ) ) {
			TranslateMessage(&msg);
			DispatchMessage( &msg );
			if (count++ > 50) break;
		}
	return true;
	}
} main;

void _threaded_timer( double seconds, bool (*callback)(void*), void *data) 
{
	main.start();
	if (callback) 
		main.add_timer( callback_data(callback, data, seconds) );
}

void threaded_exit(int status) 
{
	exit(status);
}

} // !namespace visual
