#!/usr/bin/env python
"""SimulationTrace 1.4alpha 11 December 2003
__version__ = '$Revision: 1.5 $ $Date: 2003-12-13 12:23:52+01 $ kgm'
LICENSE:
Copyright (C) 2002  Klaus G. Muller, Tony Vignaux
mailto: kgmuller@xs4all.nl and Tony.Vignaux@vuw.ac.nz

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
END OF LICENSE

Implements a trace capability for SimPy 1.3.
Based on generators (Python 2.2 and later)



**Change history:**
    9 May 03: SimulationTrace module based on SimPy 1.3

    November 03: Brought up to Simulation 1.4alpha
    
    13 Dec 2003: Merged in Monitor and Histogram
"""


from __future__ import generators
from SimPy.Lister import *
import bisect
import types
import sys
__TESTING=False
__version__="1.3 June 22 2003"
if __TESTING: print "SimPy.SimulationTrace %s" %__version__
hold=1234
passivate=5678
request=135
release=246
_endtime=0
_t=0
_e=None
_stop=1
True=1
False=0


def initialize():
    global _e,_t,_stop,trace
    _e=__Evlist()
    _t=0
    _stop=0

def now():
    return _t

def stopSimulation():
    """Application function to stop simulation run"""
    global _stop
    _stop=1
    
class Simerror(Exception):
    def __init__(self,value):
        self.value=value

    def __str__(self):
        return `self.value`
    
class Process(Lister):
    """Superclass of classes which may use generator functions"""
    def __init__(self,name="a_process"):
        #the reference to this Process' instances single process (==generator)
        self._nextpoint=None
        self.name=name
        self._nextTime=None #next activation time
        self._remainService=0
        self._preempted=0
        self._priority={}
        self._terminated=0      
        self._inInterrupt= False    

    def active(self):
        return self._nextTime <> None and not self._inInterrupt 

    def passive(self):
        return self._nextTime == None and not self._terminated

    def terminated(self):
        return self._terminated 

    def interrupted(self):
        return self._inInterrupt and not self._terminated

    def queuing(self,resource):
        return self in resource.waitQ
           
    def cancel(self,victim): 
        """Application function to cancel all event notices for this Process instance;(should be all event
        notices for the _generator_)."""
        _e._unpost(whom=victim)
        trace.recordCancel(self,victim)
        
    def _hold(self,a):
        if len(a[0]) == 3: 
            delay=abs(a[0][2])
        else:
            delay=0
        who=a[1]
        self.interruptLeft=delay 
        self._inInterrupt=False
        self.interruptCause=None
        _e._post(what=who,at=_t+delay)

    def _passivate(self,a):
        a[0][1]._nextTime=None

    def interrupt(self,victim):
        """Application function to interrupt active processes""" 
        # can't interrupt terminated or passive process or currently interrupted process
        if victim.active():
            save=trace._comment
            trace._comment=None	
            victim.interruptCause=self  # self causes interrupt
            left=victim._nextTime-_t
            victim.interruptLeft=left   # time left in current 'hold'
            victim._inInterrupt=True    
            reactivate(victim)
            trace._comment=save
            trace.recordInterrupt(self,victim)
            return left
        else:
            return None

    def interruptReset(self):
        """
        Application function for an interrupt victim to get out of
        'interrupted' state.
        """
        self._inInterrupt= False

def showEvents():
    """Returns string with eventlist as list of tuples (eventtime,action)"""
    return `[(_e.events[i],_e.aclist[i].who.name) for i in range(len(_e.events))]`        
                       
class __Evlist:
    """Defines event list and operations on it"""
    def __init__(self):
        self.events={} #has the structure {<time1>:(event notice1, event notice2, . . ),<time2>:(event notice 3,...),..}
        self.timestamps=[] #always sorted listed of event times (keys for self.events)

    def _post(self,what,at,prior=False):
        """Post an event notice for process 'what' for time 'at'"""
        # event notices are _Action instances 
        if at < _t:
            raise SimError("Attempt to schedule event in the past")
        if at in self.events:
            if prior:
                #before all other event notices at this time
                self.events[at][0:0]= [what]
            else:
                self.events[at].append(what)
        else: # first event notice at this time
            self.events[at]=[what]
            bisect.insort(self.timestamps,at) 
        what.who._nextTime=at

    def _unpost(self,whom):
        """
        Search through all event notices at whom's event time and remove whom's
        event notice if whom is a suspended process
        """
        thistime=whom._nextTime
        if thistime == None: #check if 'whom' was actually active
            return
        else:
            thislist=self.events[thistime]
            for n in thislist:
                if n.who==whom:
                    self.events[thistime].remove(n)
                    whom._nextTime = None 
                    if not self.events[thistime]:
                        # no other ev notices for thistime
                        del(self.events[thistime])
                        item_delete_point = bisect.bisect(self.timestamps,thistime) 
                        del self.timestamps[item_delete_point-1]	          
                                              
    def _nextev(self):
        """Retrieve next event from event list"""
        global _t, _stop
        if not self.events: raise Simerror("No more events at time %s" %now())
        earliest=self.timestamps[0]        #Added SDWF
        _t=max(earliest,_t)
        temp=self.events[earliest] #list of actions for this time _t
        tempev=temp[0] #first action        
        del(self.events[earliest][0])
        if self.events[earliest]==[]: 
            del(self.events[earliest]) #delete key for this time and empty list of actions
            del(self.timestamps[0])   #delete key in auxiliary list for this time
        if _t > _endtime:
            _t = _endtime
            _stop = True
            return (None,)
        try:
            tt=tempev.who._nextpoint.next()
        except StopIteration:
            tempev.who._nextpoint=None
            tempev.who._terminated=True 
            tempev.who._nextTime=None
            tt=None
            tempev=tempev.who
        return tt,tempev

class _Action:
    """Structure (who=process owner, generator=process)"""
    def __init__(self,who,generator):
        self.who=who
        self.process=generator

    def __str__(self):
        return "Action%s" %((self.who,self.process))

def activate(object,process,at="undefined",delay="undefined",prior=False):
    """Application function to activate passive process.""" 
    if not (type(process) == types.GeneratorType):
        raise Simerror("Fatal SimPy error: activating function which is not a generator (contains no 'yield')")
    if not object._terminated and not object._nextTime: 
        object._nextpoint=process       #store generator reference in object; needed for reactivation
        if at=="undefined":
            at=_t
        if delay=="undefined":
            zeit=max(_t,at)
        else:
            zeit=max(_t,_t+delay)
        _e._post(_Action(who=object,generator=process),at=zeit,prior=prior)
	trace.recordActivate(object,zeit,prior)


def reactivate(object,at="undefined",delay="undefined",prior=False):
    """Application function to reactivate a process which is active,
    suspended or passive.""" 
    # Object may be active, suspended or passive
    if not object._terminated: 
        Process().cancel(object)
        # object now passive
        if at=="undefined":
            at=_t
        if delay=="undefined":
            zeit=max(_t,at)
        else:
            zeit=max(_t,_t+delay)
        _e._post(_Action(who=object,generator=object._nextpoint),at=zeit,prior=prior)
        trace.recordReactivate(object,zeit,prior)

class Queue(list):
    def __init__(self,res,moni):
        if moni==[]:
            self.monit=True # True if Monitor
        else:
            self.monit=False
        self.moni=moni # The Monitor
        self.resource=res # the resource this queue belongs to

    def enter(self,object):
        pass

    def leave(self):
        pass

class FIFO(Queue):
    def __init__(self,res,moni):
        Queue.__init__(self,res,moni)

    def enter(self,object):
        self.append(object)
        if self.monit:
            self.moni.observe(len(self),t=now())        

    def leave(self):
        a= self.pop(0)
        if self.monit:
            self.moni.observe(len(self),t=now())
        return a

class PriorityQ(FIFO):
    """Queue is always ordered according to priority.
    Higher value of priority attribute == higher priority.
    """
    def __init__(self,res,moni):
        FIFO.__init__(self,res,moni)

    def enter(self,object):
        if len(self):
            if self[-1]._priority[self.resource] >= object._priority[self.resource]:
                self.append(object)
            else:
                z=0
                while self[z]._priority[self.resource] >= object._priority[self.resource]:
                    z += 1
                self.insert(z,object)
        else:
            self.append(object)
        if self.monit:
            self.moni.observe(len(self),t=now())

class Resource(Lister):
    """Models shared, limited capacity resources with queuing;
    FIFO is default queuing discipline.
    """
    
    def __init__(self,capacity=1,name="a_resource",unitName="units",
                 qType=FIFO,preemptable=0,monitored=False): 
        self.name=name          # resource name
        self.capacity=capacity  # resource units in this resource
        self.unitName=unitName  # type name of resource units
        self.n=capacity         # uncommitted resource units

        if monitored:           # Monitor waitQ, activeQ
            self.actMon=Monitor(name="Active Queue Monitor %s"%self.name,
                                 ylab="nr in queue",tlab="time")
            monact=self.actMon
            self.waitMon=Monitor(name="Wait Queue Monitor %s"%self.name,
                                 ylab="nr in queue",tlab="time")
            monwait=self.waitMon
        else:
            monwait=None
            monact=None            
        self.waitQ=qType(self,monwait)
        self.preemptable=preemptable
        self.activeQ=qType(self,monact)
        self.priority_default=0
        
    def _request(self,arg):
        """Process request event for this resource"""
        object=arg[1].who
        if len(arg[0]) == 4:        # yield request,self,resource,priority
            object._priority[self]=arg[0][3]
        else:                       # yield request,self,resource 
            object._priority[self]=self.priority_default
        if self.preemptable and self.n == 0: # No free resource
            # test for preemption condition
            preempt=object._priority[self] > self.activeQ[-1]._priority[self]
            # If yes:
            if preempt:
                z=self.activeQ[-1]
                # suspend lowest priority process being served
                suspended = z
                # record remaining service time
                z._remainService = z._nextTime - _t 
                Process().cancel(z)
                # remove from activeQ
                self.activeQ.remove(z)
                # put into front of waitQ
                self.waitQ.insert(0,z)
                # record that it has been preempted
                z._preempted = 1
                # passivate re-queued process
                z._nextTime=None 
                # assign resource unit to preemptor
                self.activeQ.enter(object)                    
                # post event notice for preempting process
                _e._post(_Action(who=object,generator=object._nextpoint),at=_t,prior=1)
            else:
                self.waitQ.enter(object)
                # passivate queuing process
                object._nextTime=None 
        else: # treat non-preemption case       
            if self.n == 0:
                self.waitQ.enter(object)
                # passivate queuing process
                object._nextTime=None
            else:
                self.n -= 1
                self.activeQ.enter(object)
                _e._post(_Action(who=object,generator=object._nextpoint),at=_t,prior=1)

    def _release(self,arg):
        """Process release request for this resource"""
        self.n += 1
        self.activeQ.remove(arg[1].who)
        #reactivate first waiting requestor if any; assign Resource to it
        if self.waitQ:
            object=self.waitQ.leave()
            self.n -= 1             #assign 1 resource unit to object
            self.activeQ.enter(object)
            # if resource preemptable:
            if self.preemptable:
                # if object had been preempted:
                if object._preempted:
                    object._preempted = 0
                    # reactivate object delay= remaining service time
                    reactivate(object,delay=object._remainService)
                # else reactivate right away
                else:
                    reactivate(object,delay=0,prior=1)
            # else:
            else:
                reactivate(object,delay=0,prior=1)
        _e._post(_Action(who=arg[1].who,generator=arg[1].who._nextpoint),at=_t,prior=1)



def scheduler(till=0):
    """Schedules Processes/semi-coroutines until time 'till'.
    Deprecated since version 0.5.
    """
    simulate(until=till)
    
def holdfunc(a):
    a[0][1]._hold(a)

def requestfunc(a):
    a[0][2]._request(a)

def releasefunc(a):
    a[0][2]._release(a)

def passivatefunc(a):
    a[0][1]._passivate(a)
    
def simulate(until=0):
    """Schedules Processes/semi-coroutines until time 'until'"""
    
    """Gets called once. Afterwards, co-routines (generators) return by 'yield' with a cargo:
    yield hold, self, <delay>: schedules the "self" process for activation after <delay> time units.
                               If <,delay> missing, same as "yield hold,self,0"
                               
    yield passivate,self    :  makes the "self" process wait to be re-activated

    yield request,self,<Resource> : request 1 unit from <Resource>.

    yield release,self,<Resource> : release 1 unit to <Resource>    

    Event notices get posted in event-list by scheduler after "yield" or by "activate"/"reactivate" functions.
    
    """
    global _endtime,_e,_stop,_t 
    _stop=False
    if _e == None:
        raise Simerror("Fatal SimPy error: Simulation not initialized")
    if _e.events == {}:
        message="SimPy: No activities scheduled"
        return message
        
    _endtime=until
    message="SimPy: Normal exit" 
    dispatch={hold:holdfunc,request:requestfunc,release:releasefunc,
              passivate:passivatefunc}
    while not _stop and _t<=_endtime:
        try:
            a=_e._nextev()        
            if not a[0]==None:
                command = a[0][0]
                dispatch[command](a)
		trace.recordEvent(command,a)
            else:
                if not a==(None,): #not at endtime!
                    trace.tterminated(a[1])
        except Simerror, error:
            message="SimPy: "+ error.value
            _stop = True 
    _e=None
    if not(trace.outfile is sys.stdout):
        trace.outfile.close()
    return message
    
class Trace(Lister):
    commands={1234:"hold",5678:"passivate",135:"request",246:"release"}

    def __init__(self,start=0,end=10000000000L,toTrace=\
                 ["hold","activate","cancel","reactivate","passivate","request",
                  "release","interrupt","terminated"],outfile=sys.stdout):
        #global tracego
        Trace.commandsproc={1234:Trace.thold,5678:Trace.tpassivate,135:Trace.trequest,246:Trace.trelease}
        self.start=start
        self.end=end
        self.toTrace=toTrace
        self.tracego=True
        self.outfile=outfile
        self._comment=None

    def treset(self):
        Trace.commandsproc={1234:Trace.thold,5678:Trace.tpassivate,135:Trace.trequest,246:Trace.trelease}
        self.start=0
        self.end=10000000000L
        self.toTrace=["hold","activate","cancel","reactivate","passivate","request",
                  "release","interrupt","terminated"]
        self.tracego=True
        self.outfile=sys.stdout
        self._comment=None

    def tchange(self,**kmvar):
        for v in kmvar.keys():
            if v=="start":
                self.start=kmvar[v]
            elif v=="end":
                self.end=kmvar[v]
            elif v=="toTrace":
                self.toTrace=kmvar[v]
            elif v=="outfile":
                self.outfile=kmvar[v]                
                
    def tstart(self):
        self.tracego=True

    def tstop(self):
        self.tracego=False

    def ifTrace(self,cond):
        if self.tracego and (self.start <= now() <= self.end) and cond:
            return True

    def thold(self,par):
        return "delay: %s"%par[0][2]
    thold=classmethod(thold)
    
    def trequest(self,par):
        res=par[0][2]
        if len(par[0])==4:
            priority=" priority: "+str(par[0][3])
        else:
            priority=" priority: default"
        wQ=[x.name for x in res.waitQ]
        aQ=[x.name for x in res.activeQ]
        return "<%s> %s \n. . .waitQ: %s \n. . .activeQ: %s"%(res.name,priority,wQ,aQ)
    trequest=classmethod(trequest)
    
    def trelease(self,par):
        res=par[0][2]
        wQ=[x.name for x in res.waitQ]
        aQ=[x.name for x in res.activeQ]
        return "<%s> \n. . .waitQ: %s \n. . .activeQ: %s"%(res.name,wQ,aQ)
    trelease=classmethod(trelease)
    
    def tpassivate(self,par):
        return ""
    tpassivate=classmethod(tpassivate)

    def tactivate(self,par):
        pass
    tactivate=classmethod(tactivate)
    
    def recordEvent(self,command,whole):        
        if self.ifTrace(Trace.commands[command] in self.toTrace):
            print >>self.outfile, now(),Trace.commands[command],"<"+whole[0][1].name+">",\
                  Trace.commandsproc[command](whole)
            if self._comment:
                print >>self.outfile,"----",self._comment
        self._comment=None


    def recordInterrupt(self,who,victim):
        if self.ifTrace("interrupt" in self.toTrace):
            print >>self.outfile,"%s interrupt by: <%s> of: <%s>"%(now(),who.name,victim.name)
            if self._comment:
                print >>self.outfile,"----",self._comment
        self._comment=None
                
    def recordCancel(self,who,victim):
        if self.ifTrace("interrupt" in self.toTrace):
            print >>self.outfile,"%s cancel by: <%s> of: <%s>"%(now(),who.name,victim.name)
            if self._comment:
                print >>self.outfile,"----",self._comment
        self._comment=None
        
    def recordActivate(self,who,when,prior):
        if self.ifTrace("activate" in self.toTrace):
            print >>self.outfile,"%s activate <%s> at time: %s prior: %s"%(now(),who.name,\
                                                                       when, prior)
            if self._comment:
                print >>self.outfile,"----",self._comment
        self._comment=None                                                                       

    def recordReactivate(self,who,when,prior):
        if self.ifTrace("reactivate" in self.toTrace):
            print >>self.outfile,"%s reactivate <%s> time: %s prior: %s"%(now(),who.name,\
                                                                        when, prior)
            if self._comment:
                print >>self.outfile,"----",self._comment
        self._comment=None

    def tterminated(self,who):
        if self.ifTrace("terminated" in self.toTrace):
            print >>self.outfile,"%s <%s> terminated"%(now(),who.name)
            if self._comment:
                print >>self.outfile,"----",self._comment
        self._comment=None        

    def ttext(self,par):
        self._comment=par
  
class Monitor(list):
    """ Monitored variables

    A Class for monitored variables, that is, variables that allow one
    to gather simple statistics.  A Monitor is a subclass of list and
    list operations can be performed on it. An object is established
    using m= Monitor(name = '..'). It can be given a
    unique name for use in debugging and in tracing and ylab and tlab
    strings for labelling graphs.
    """
    def __init__(self,name='',ylab='y',tlab='t'):
        list.__init__(self)
        self.startTime = 0.0
        self.name = name
        self.ylab = ylab
        self.tlab = tlab

    def observe(self,y,t=None):
        """record y and t"""
        if t is  None: t = now()
        self.append([t,y])

    def tally(self,y):
        """ deprecated: tally for backward compatibility"""
        self.observe(y,0)
                   
    def accum(self,y,t=None):
        """ deprecated:  accum for backward compatibility"""
        self.observe(y,t)

    def reset(self,t=None):
        """reset the sums and counts for the monitored variable """
        self[:]=[]
        if t is None: t = now()
        self.startTime = t
        
    def tseries(self):
        """ the series of measured times"""
        return list(zip(*self)[0])

    def yseries(self):
        """ the series of measured values"""
        return list(zip(*self)[1])

    def count(self):
        """ deprecated: the number of observations made """
        return self.__len__()
        
    def total(self):
        """ the sum of the y"""
        if self.__len__()==0:  return 0
        else:
            sum = 0.0
            for i in range(self.__len__()):
                sum += self[i][1]
            return sum # replace by sum() later

    def mean(self):
        """ the simple average of the monitored variable"""
        try: return 1.0*self.total()/self.__len__()
        except:  print 'SimPy: No observations  for mean'

    def var(self):
        """ the sample variance of the monitored variable """
        n = len(self)
        tot = self.total()
        ssq=0.0
        yy = self.yseries()
        for i in range(self.__len__()):
            ssq += self[i][1]**2 # replace by sum() eventually
        try: return (ssq - float(tot*tot)/n)/n
        except: print 'SimPy: No observations for sample variance'
        
    def timeAverage(self,t=None):
        """ the time-average of the monitored variable.

            If t is used it is assumed to be the current time,
            otherwise t =  now()
        """
        N = self.__len__()
        if N  == 0:
            print 'SimPy: No observations for timeAverage'
            return None

        if t == None: t = now()
        sum = 0.0
        tlast = self.startTime
        #print 'DEBUG: timave ',t,tlast
        ylast = 0.0
        for i in range(N):
            ti,yi = self[i]
            sum += ylast*(ti-tlast)
            tlast = ti
            ylast = yi
        sum += ylast*(t-tlast)
        T = t - self.startTime
        if T == 0:
             print 'SimPy: No elapsed time for timeAverage'
             return None
        #print 'DEBUG: timave ',sum,t,T
        return sum/float(T)

    def histogram(self,low=0.0,high=100.0,nbins=10):
        """ A histogram of the monitored y data values.
        """
        h = Histogram(name=self.name,low=low,high=high,nbins=nbins)
        ys = self.yseries()
        for y in ys: h.addIn(y)
        return h


def test_demo():
    class Aa(Process):
        sequIn=[]
        sequOut=[]
        def __init__(self,holdtime,name):
            Process.__init__(self,name)
            self.holdtime=holdtime

        def life(self,priority):
            for i in range(1):
                Aa.sequIn.append(self.name)
                print now(),rrr,"waitQ:",len(rrr.waitQ),"activeQ:",\
                      len(rrr.activeQ)
                print "waitQ: ",[(k.name,k._priority[rrr]) for k in rrr.waitQ]
                print "activeQ: ",[(k.name,k._priority[rrr]) \
                           for k in rrr.activeQ]
                assert rrr.n+len(rrr.activeQ)==rrr.capacity, \
               "Inconsistent resource unit numbers"
                print now(),self.name,"requests 1 ", rrr.unitName
                yield request,self,rrr,priority
                print now(),self.name,"has 1 ",rrr.unitName
                print now(),rrr,"waitQ:",len(rrr.waitQ),"activeQ:",\
                      len(rrr.activeQ)
                print now(),rrr,"waitQ:",len(rrr.waitQ),"activeQ:",\
                      len(rrr.activeQ)
                assert rrr.n+len(rrr.activeQ)==rrr.capacity, \
               "Inconsistent resource unit numbers"
                yield hold,self,self.holdtime
                print now(),self.name,"gives up 1",rrr.unitName
                yield release,self,rrr
                Aa.sequOut.append(self.name)
                print now(),self.name,"has released 1 ",rrr.unitName
                print "waitQ: ",[(k.name,k._priority[rrr]) for k in rrr.waitQ]
                print now(),rrr,"waitQ:",len(rrr.waitQ),"activeQ:",\
                      len(rrr.activeQ)
                assert rrr.n+len(rrr.activeQ)==rrr.capacity, \
                       "Inconsistent resource unit numbers"

    class Destroyer(Process):
        def __init__(self):
            Process.__init__(self)

        def destroy(self,whichProcesses):
            for i in whichProcesses:
                Process().cancel(i)
            yield hold,self,0

    class Observer(Process):
        def __init__(self):
            Process.__init__(self)

        def observe(self,step,processes,res):
            while now()<11:
                for i in processes:
                    print "++ %s process: %s: active:%s, passive:%s, terminated: %s,interrupted:%s, queuing:%s"\
                          %(now(),i.name,i.active(),i.passive(),i.terminated(),i.interrupted(),i.queuing(res))
                print
                yield hold,self,step
            
            
    print "****First case == priority queue, resource service not preemptable"
    initialize()
    rrr=Resource(5,name="Parking",unitName="space(s)", qType=PriorityQ,
                 preemptable=0)
    procs=[]
    for i in range(10):
        z=Aa(holdtime=i,name="Car "+str(i))
        procs.append(z)
        activate(z,z.life(priority=i))
    o=Observer()
    activate(o,o.observe(1,procs,rrr))
    a=simulate(until=10000)
    print a
    print "Input sequence: ",Aa.sequIn
    print "Output sequence: ",Aa.sequOut

    print "\n****Second case == priority queue, resource service preemptable"
    initialize()
    rrr=Resource(5,name="Parking",unitName="space(s)", qType=PriorityQ,
                 preemptable=1)
    procs=[]
    for i in range(10):
        z=Aa(holdtime=i,name="Car "+str(i))
        procs.append(z)
        activate(z,z.life(priority=i))
    o=Observer()
    activate(o,o.observe(1,procs,rrr))
    Aa.sequIn=[]
    Aa.sequOut=[]
    a=simulate(until=10000)
    print a
    print "Input sequence: ",Aa.sequIn
    print "Output sequence: ",Aa.sequOut   

def test_interrupt():
    class Bus(Process):
        def __init__(self,name):
            Process.__init__(self,name)

        def operate(self,repairduration=0):
            print now(),">> %s starts" %(self.name)
            tripleft = 1000
            while tripleft > 0:
                yield hold,self,tripleft
                if self.interrupted():
                    print "interrupted by %s" %self.interruptCause.name
                    print "%s: %s breaks down " %(now(),self.name)
                    tripleft=self.interruptLeft
                    self.interruptReset()
                    print "tripleft ",tripleft
                    reactivate(br,delay=repairduration) # breakdowns only during operation
                    yield hold,self,repairduration
                    print now()," repaired"
                else:
                    break # no breakdown, ergo bus arrived
            print now(),"<< %s done" %(self.name)

    class Breakdown(Process):
        def __init__(self,myBus):
            Process.__init__(self,name="Breakdown "+myBus.name)
            self.bus=myBus

        def breakBus(self,interval):

            while True:
                yield hold,self,interval
                if self.bus.terminated(): break
                self.interrupt(self.bus)
                
    print"\n\n+++test_interrupt"
    initialize()
    b=Bus("Bus 1")
    activate(b,b.operate(repairduration=20))
    br=Breakdown(b)
    activate(br,br.breakBus(200))
    print simulate(until=4000)

def testMonitoredResource():
    class A(Process):
        def __init__(self,name):
            Process.__init__(self,name=name)
        def life(self):
            yield request,self,res
            yield hold,self,5
            print now(), res
            yield release,self,res
    initialize()
    res=Resource(name="'The Resource'",monitored=True)
    for i in range(10):
        p=A(name="User%s"%(i+1))
        activate(p,p.life())
    excond=simulate(until=1000)
    print
    print excond
    print "%s:\n%s\n%s:\n%s"%(res.waitMon.name,res.waitMon,
                             res.actMon.name,res.waitMon)
    
if __name__ == "__main__":
    trace=Trace(end=4000)
    test_demo()
    trace=Trace(end=2000)
    test_interrupt()
    testMonitoredResource()
else:
    trace=Trace()
    
