"""
oofun (OpenOpt Function) usage for to skip calculating same parts of code
(that can be present in non-lin funcs f, df, c, dc, h, dh, d2f etc)

see also examples/oofun/*.py files for other oofun benefits

Requires openopt v > 0.18 (latest tarball or svn is ok).

Let's consider the problem
(x0-0)^2 + (x1-1)^2 + (x2-2)^2 +... +(x[n-1]-(n-1))^2 -> min
n=15
subjected to
c0(x) = x[n-1] ** 2 + x[n-2] ** 2 - 1 <= 0
c1(x) = c0(x)^2-1 <=0
c2(x) = c1(x)^2-1.5 <=0
c3(x) = c0(x) + 2*c2(x) + 1.5 <= 0
"""

from numpy import arange, ones, asarray, zeros, array, hstack
from scikits.openopt import NLP, oofun

n= 15

x0 = ones(n)
f = lambda x: ((x - arange(n)) ** 2).sum()
df = lambda x: 2 * (x-arange(n))

c0 = oofun(lambda x: x[-1] ** 2 + x[-2] ** 2 - 1, name = 'constraint 0')
c1 = oofun(lambda c: c**2 - 1, input = c0, name = 'constraint 1')
c2 = oofun(lambda c: c**2 - 1.5, input = c1) # parameter "name" is optional
c3 = oofun(lambda y, z: y + 2*z + 1.5, input = [c0, c2], name = 'constraint 3')

c = [c2, c0, c1, c3] # ORDER: any

p = NLP(f,  x0,  df=df, c=c)
r = p.solve('ipopt')
#Solver:   Time Elapsed = 1.19 	CPU Time Elapsed = 1.07
#objFunValue: 332.71633 (feasible, max constraint =  1.00002e-08)
