#include <windows.h>
#include "stubmsg.h"

#if _DEBUG
#  define DPRINTF(msg) MessageBox(NULL, msg, "Debugging Print", MB_OK)
#else
#  define DPRINTF(msg)
#endif

/* Subsystem */
#define SCRIPT_SUBSYSTEM_UNKNOWN        0
#define SCRIPT_SUBSYSTEM_NATIVE         1
#define SCRIPT_SUBSYSTEM_WINDOWS_GUI    2
#define SCRIPT_SUBSYSTEM_WINDOWS_CUI    3

/* Characteristics */
#define SCRIPT_LOADER_DEBUG     0x0001
#define SCRIPT_LOADER_VERBOSE   0x0002
#define SCRIPT_LOADER_OPTIMIZE  0x0004

typedef struct _SCRIPT_INFO {
  DWORD Signature;              /* Identifying signature ("FT\0\0") */
  WORD MajorPythonVersion;      /* Major version of the required Python */
  WORD MinorPythonVersion;      /* Minor version of the required Python */
  WORD Subsystem;               /* Subsystem required to run the script */
  WORD Characteristics;         /* Characteristics of the script loader. */
  WORD ScriptAddress;           /* Offset of the script data. */
  WORD ScriptSize;              /* Size of the script data, in bytes. */
} SCRIPT_INFO, *PSCRIPT_INFO;

typedef void PyObject;
static void (*Py_SetProgramName)(char *);
static char *(*Py_GetProgramFullPath)(void);
static char *(*Py_GetPath)(void);
static void (*Py_Initialize)(void);
static void (*PySys_SetArgv)(int, char **);
static int (*PyRun_SimpleString)(char *);
static void (*Py_Finalize)(void);

static LPVOID GetResource(UINT uID, LPCTSTR lpType)
{
  HRSRC hResourceInfo;
  HGLOBAL hResource;

  /* Locate the resource in the .EXE file. */
  hResourceInfo = FindResource(NULL, MAKEINTRESOURCE(uID), lpType);
  if (hResourceInfo == NULL)
    return NULL;
  /* Load the resource into global memory. */
  hResource = LoadResource(NULL, hResourceInfo);
  if (hResource == NULL)
    return NULL;
  /* Lock the resource into global memory. */
  return LockResource(hResource);
}


static void ErrorExit(DWORD dwMessageId)
{
  LPTSTR lpMessage;
  DWORD dwErrorCode = GetLastError();
  if (FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
                    FORMAT_MESSAGE_FROM_HMODULE,
                    NULL, dwMessageId, 0,
                    (LPVOID) &lpMessage, 0,
                    NULL)) {
    MessageBox(NULL, lpMessage, "Runtime Error", MB_OK | MB_ICONSTOP);
    LocalFree(lpMessage);
  } else {
    dwErrorCode = GetLastError();
  }
  ExitProcess(dwErrorCode);
}


static HMODULE LoadPython(PSCRIPT_INFO pScriptInfo)
{
  TCHAR szDllName[] = {
    'p', 'y', 't', 'h', 'o', 'n', '0', '0', '.', 'd', 'l', 'l', '\0'
  };
  TCHAR szDllDebugName[] = {
    'p', 'y', 't', 'h', 'o', 'n', '0', '0', '_', 'd', '.', 'd', 'l', 'l', '\0'
  };
  PTSTR szFileName;
  HMODULE hPython;

  szFileName = ((pScriptInfo->Characteristics & SCRIPT_LOADER_DEBUG)
                ? (PTSTR) szDllDebugName : (PTSTR) szDllName);
  szFileName[6] += (pScriptInfo->MajorPythonVersion & 0xFF);
  szFileName[7] += (pScriptInfo->MinorPythonVersion & 0xFF);
  hPython = LoadLibraryEx(szFileName, NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
  if (hPython == NULL) {
    return NULL;
  }

  if (pScriptInfo->Characteristics & SCRIPT_LOADER_VERBOSE) {
    int *Py_Flag = (int *) GetProcAddress(hPython, "Py_VerboseFlag");
    if (Py_Flag == NULL) {
      CloseHandle(hPython);
      return NULL;
    }
    *Py_Flag = 1;
  }
  if (pScriptInfo->Characteristics & SCRIPT_LOADER_OPTIMIZE) {
    int *Py_Flag = (int *) GetProcAddress(hPython, "Py_OptimizeFlag");
    if (Py_Flag == NULL) {
      CloseHandle(hPython);
      return NULL;
    }
    *Py_Flag = 1;
  }

#define LOADPROC(TYPE, NAME, ARGS) do {                   \
    NAME = (TYPE(*)ARGS) GetProcAddress(hPython, #NAME);  \
    if (NAME == NULL) {                                   \
      CloseHandle(hPython);                               \
      return NULL;                                        \
    }                                                     \
  } while (0)
  
  LOADPROC(void, Py_SetProgramName, (char *));
  LOADPROC(char *, Py_GetProgramFullPath, (void));
  LOADPROC(char *, Py_GetPath, (void));
  LOADPROC(void, Py_Initialize, (void));
  LOADPROC(int, PySys_SetArgv, (int, char **));
  LOADPROC(int, PyRun_SimpleString, (char *));
  LOADPROC(void, Py_Finalize, (void));
#undef LOADPROC

  return hPython;
}


static LPBYTE GetScript(PSCRIPT_INFO pScriptInfo)
{
  TCHAR szPath[MAX_PATH+1];
  HANDLE hFile;
  DWORD dwBytesRead;
  LPBYTE lpScript;

  if (!GetModuleFileName(NULL,  szPath, MAX_PATH)) {
    return NULL;
  }

  hFile = CreateFile(szPath, GENERIC_READ, FILE_SHARE_READ, NULL,
                     OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
  if (hFile == INVALID_HANDLE_VALUE) {
    return NULL;
  }
  if (SetFilePointer(hFile, pScriptInfo->ScriptAddress, NULL, FILE_BEGIN) ==
      INVALID_SET_FILE_POINTER) {
    CloseHandle(hFile);
    return NULL;
  }
  lpScript = LocalAlloc(LMEM_FIXED, pScriptInfo->ScriptSize);
  if (lpScript == NULL) {
    CloseHandle(hFile);
    return NULL;
  }
  if (!ReadFile(hFile, lpScript, pScriptInfo->ScriptSize, &dwBytesRead, NULL)) {
    LocalFree(lpScript);
    CloseHandle(hFile);
    return NULL;
  }
  CloseHandle(hFile);
  if (pScriptInfo->ScriptSize != dwBytesRead) {
    LocalFree(lpScript);
    return NULL;
  }
  return lpScript;
}


/* Undo the addition of argv0_path (in getpathp.c) to module_search_path. */
static int path_compare(register const char *s1, register const char *s2,
                        register size_t n)
{
  char c1 = '\0';
  char c2 = '\0';

  while (n > 0) {
    c1 = *s1++;
    c2 = *s2++;
    if (c1 == '\0' || c1 != c2)
      return c1 - c2;
    n--;
  }
  return c1 == ';' || c1 == '\0';
}

static void FixupSysPath(void)
{
  char *progpath = Py_GetProgramFullPath();
  char *path = Py_GetPath();
  char argv0_path[MAX_PATH+3];
  size_t len;

  /* argv0_path = ";;" + os.path.dirname(progpath) */
  len = 0;
  argv0_path[len++] = ';';
  argv0_path[len++] = ';';
  while (*progpath) argv0_path[len++] = *progpath++;
  while (argv0_path[--len] != '\\');
  argv0_path[len] = '\0';
  DPRINTF(argv0_path);

  for (path = Py_GetPath(); *path;) {
    DPRINTF(path);
    /* scan for starting delimiter */
    while (*path != ';' && *path) *path++;
    /* compare path with argv0_path */
    /* if the two paths compare equal, remove `argv0_path` from `path` */
    if (path_compare(path, argv0_path, len) == 0) {
      char *dest = path;
      int i = 0;
      path += len;
      while ((dest[i] = path[i])) i++;
    } else {
      path++;
    }
  }
  DPRINTF(Py_GetPath());
}


int WINAPI WinMain(HINSTANCE hInstance,
                   HINSTANCE hPrevInstance,
                   LPSTR lpCmdLine,
                   int nCmdShow)
{
  PSCRIPT_INFO pScriptInfo;
  HMODULE hPython;
  PBYTE pScript;
  int exitcode;

  /* Locate the script info resource in the .EXE file. */
  /* Note, no action is required to cleanup the resource */
  pScriptInfo = GetResource(1, RT_RCDATA);
  if (pScriptInfo == NULL) {
    ErrorExit(IDS_EXECUTABLE_INVALID);
  }
  DPRINTF("Loaded the ScriptInfo resource.");

  /* Load the Python DLL */
  hPython = LoadPython(pScriptInfo);
  if (hPython == NULL) {
    ErrorExit(IDS_PYTHON_ERROR);
  }
  DPRINTF("Loaded the Python DLL.");

  /* Load the script itself */
  pScript = GetScript(pScriptInfo);
  if (pScript == NULL) {
    CloseHandle(hPython);
    ErrorExit(IDS_EXECUTABLE_INVALID);
  }
  DPRINTF("Loaded script bytes.");

  /* Initialize Python */
  Py_SetProgramName(__argv[0]);
  FixupSysPath();
  Py_Initialize();
  DPRINTF("Initialized Python.");

  /* Run the script itself. */
  PySys_SetArgv(__argc, __argv);
  exitcode = PyRun_SimpleString(pScript);
  DPRINTF("Script completed.");

  /* Cleanup */
  Py_Finalize();
  LocalFree(pScript);
  CloseHandle(hPython);

  return exitcode;
}
