#Copyright (c) 2006 Erich Schubert erich@debian.org

#Permission is hereby granted, free of charge, to any person obtaining a copy
#of this software and associated documentation files (the "Software"), to deal
#in the Software without restriction, including without limitation the rights
#to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
#copies of the Software, and to permit persons to whom the Software is
#furnished to do so, subject to the following conditions:

#The above copyright notice and this permission notice shall be included in
#all copies or substantial portions of the Software.

#THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
#AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
#LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
#OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
#SOFTWARE.
import sys, re
from popen2 import popen3, Popen4
from util import Util

class Iptables:
	"""
	Interface to controlling iptables
	"""
	# iptables commands to use
	iptablessave = "/sbin/iptables-save -c"
	iptablesrestore = "/sbin/iptables-restore -c"
	iptablesset = "/sbin/iptables-restore"
	iptablesversion = "/sbin/iptables --version"

	# match for debugging errors
	match_errorline = re.compile(r"^Error occurred at line: ([0-9]+)$")
	match_errormsg  = re.compile(r"^iptables-restore(?: v[0-9.]+)?: (?:iptables-restore: )?(.+)$")
	match_hidemsg   = re.compile(r"^Try `iptables-restore -h' or 'iptables-restore --help' for more information.$")
	match_version   = re.compile(r"^iptables v([0-9]+\.[0-9.]+)$")

	# version number cache
	_version = None

	# Handled error class
	class Error(Exception):
		"""
		Basic exception class
		"""
		pass

	def version(min=None, max=None):
		"""
		Return iptables version or test for a minimum and/or maximum version

		min -- minimal iptables version required
		max -- maximum iptables version required
		"""
		if not Iptables._version:
			# query iptables version
			ir, iw, ie = popen3(Iptables.iptablesversion)
			iw.close()
			result = ir.readlines()
			ir.close()
			ie.close()
			for line in result:
				m = Iptables.match_version.match(line)
				if m and m.group(1):
					Iptables._version = m.group(1)
					break
			# still no version number? - raise an exception
			if not Iptables._version:
				raise Error("Couldn't get iptables version!")
		if not min and not max:
			return Iptables._version
		if min:
			if Util.compare_versions(Iptables._version, min) < 0: return False
		if max:
			if Util.compare_versions(Iptables._version, max) > 0: return False
		return True

	version = staticmethod(version)

	def save():
		"""
		Dump current iptables ruleset into an array of strings
		"""
		# save old iptables status
		sr, sw, se = popen3(Iptables.iptablessave)
		sw.close()
		savedlines = sr.readlines()
		sr.close()
		for line in se.readlines():
			# skip empty lines
			if line=="\n":
				continue
			sys.stderr.write(line)
		se.close()
		return savedlines
	save = staticmethod(save)

	def restore(savedlines):
		"""
		Restore iptables rules from a list of strings
		(generated by iptables_save)
		"""
		# restore old iptables rules
		ipr = Popen4(Iptables.iptablesrestore)
		rr, rw = ipr.fromchild, ipr.tochild
		for line in savedlines:
			rw.write(line)
		rw.close()
		# output any error
		for line in rr.readlines():
			# skip empty lines
			if line=="\n":
				continue
			sys.stderr.write(line)
		rr.close()
		return ipr.wait()
	restore = staticmethod(restore)

	def commit(lines):
		"""
		Commit iptables rules from a list of (annotated!) commands
		"""
		# TODO: verify that the lines don't contain linewraps
		# and have logging info!
		# restore old iptables rules
		ipr = Popen4(Iptables.iptablesset)
		rr, rw = ipr.fromchild, ipr.tochild
		for line in lines:
			rw.write(line[0])
			rw.write("\n")
		rw.close()
		# output any error
		errormsg = [ None, None ]
		for line in rr.readlines():
			# skip empty lines
			if line=="\n":
				continue
			# try to grab the error message with line number
			if not errormsg[1]:
				m = Iptables.match_errorline.match(line)
				if m:
					errormsg[1] = lines[int(m.group(1))-1][1]
					continue
			# try to grab a detailed error description
			if not errormsg[0]:
				m = Iptables.match_errormsg.match(line)
				if m:
					errormsg[0] = m.group(1)
					continue
			# ignore the default "info" message
			m = Iptables.match_hidemsg.match(line)
			if m:
				continue
			# print remaining lines
			sys.stderr.write(line)
		rr.close()
		success = (ipr.wait() == 0)
		if not success:
			if errormsg:
				raise Iptables.Error("Firewall commit failed: %s, caused by %s" % (errormsg[0], errormsg[1]))
			else:
				raise Iptables.Error("Firewall commit failed due to unknown error.")
		return success
	commit = staticmethod(commit)
