#!/usr/bin/python -t
"""$Id: pydb.py.in,v 1.105 2006/07/29 09:00:03 rockyb Exp $
A Python debugger. Contains user-callable routines, e.g.
run, set_trace, runevel.

(See also pydb.doc for documentation.)"""

# This part contains user entry subroutines.

_debugger_name = 'pydb'

# The name of the debugger we are currently going by.
_version = '1.17'

import inspect, os, re, sys, traceback

from optparse import OptionParser

from gdb import Gdb
from gdb import Restart

__all__ = ["Pdb",     "exception_hook", "help",
           "pm",      "post_mortem", "run",
           "runcall", "runeval",
           "set_trace"]

def init():
    global program
    global _re_linetrace_delay

    program = os.path.basename(sys.argv[0])

def process_options(pydb, debugger_name, prog, pkg_version,
                    option_list=None):
    """Handle debugger options. Use option_list if you want are writing
    another main program and want to extend the existing set of debugger
    options.

    The options dicionary from opt_parser is return. Global sys.argv is
    also updated."""
    usage_str="""%s [debugger-options] [python-script [script-options...]]

       Runs the extended python debugger""" % (prog)

    optparser = OptionParser(usage=usage_str, option_list=option_list,
                             version="%%prog version %s" % pkg_version)

    optparser.add_option("-X", "--trace", dest="linetrace",
                         action="store_true", default=False, 
                         help="Show lines before executing them. " +
                         "This option also sets --batch")
    optparser.add_option("--batch", dest="noninteractive",
                         action="store_true", default=False, 
                         help="Don't run interactive commands shell on "+
                         "stops.")
    optparser.add_option("--basename", dest="basename",
                         action="store_true", default=False, 
                         help="Filenames strip off basename, (e.g.For testing)"
                         )
    optparser.add_option("-x", "--command", dest="command",
                         action="store", type='string', metavar='FILE',
                         help="Execute commands from FILE.")
    optparser.add_option("--cd", dest="cd",
                         action="store", type='string', metavar='DIR',
                         help="Change current directory to DIR.")
    optparser.add_option("-n", "--nx", dest="noexecute",
                         action="store_true", default=False, 
                         help="Don't execute commands found in any " +
                         "initialization files")
    optparser.add_option("-o", "--output", dest="output", metavar='FILE',
                         action="store", type='string',
                         help="Write debugger's output (stdout) "
                         + "to FILE")
    optparser.add_option("--error", dest="errors", metavar='FILE',
                         action="store", type='string',
                         help="Write debugger's error output "
                         + "(stderr) to FILE")
    optparser.add_option("-e", "--exec", dest="execute", type="string",
                        help="list of debugger commands to " +
                        "execute. Separate the commands with ;;")

    # Set up to stop on the first non-option because that's the name
    # of the script to be debugged on arguments following that are
    # that scripts options that should be left untouched.  We would
    # not want to interpret and option for the script, e.g. --help, as
    # one one of our own, e.g. --help.

    optparser.disable_interspersed_args()

    # execfile() run out of Bdb.py uses sys.argv, so we have to
    # clobber it and make it what the debugged script expects. Also
    # the debugged script probably wants to look at sys.argv.
    # So we'll change sys.argv to look like the program were invoked
    # directly

    # Save the original just for use in restart (via exec)
    pydb._sys_argv = list(sys.argv)   
    (opts, sys.argv) = optparser.parse_args()

    if opts.linetrace: opts.noninteractive = True
    pydb.linetrace = opts.linetrace
    pydb.noninteractive = opts.noninteractive

    # --nx or -n
    if not opts.noexecute:
        # Read $HOME/.pydbrc and ./.pydbrc
        if 'HOME' in os.environ:
            envHome = os.environ['HOME']
            pydb.setup_source("%s.%src" % (envHome, debugger_name))
        pydb.setup_source(".%src" % debugger_name);

    if opts.cd:
        os.chdir(opts.cd)

    if opts.basename: pydb.basename = True

    # As per gdb, first we execute user initialization files and then
    # we execute any file specified via --command.
    if opts.command:
        pydb.setup_source(os.path.expanduser(opts.command), True);

    if opts.execute:
        pydb.cmdqueue = list(opts.execute.split(';;'))

    if opts.output:
        try: 
            pydb.stdout = open(opts.output, 'w')

        except IOError, (errno, strerror):
            print "I/O in opening debugger output file %s" % opts.output
            print "error(%s): %s" % (errno, strerror)
        except ValueError:
            print "Could not convert data to an integer."
        except:
            print "Unexpected error in opening debugger output file %s" % \
                  opts.output
            print sys.exc_info()[0]
            sys.exit(2)

    if opts.errors:
        try: 
            pydb.stderr = open(opts.errors, 'w')
        except IOError, (errno, strerror):
            print "I/O in opening debugger output file %s" % opts.errors
            print "error(%s): %s" % (errno, strerror)
        except ValueError:
            print "Could not convert data to an integer."
        except:
            print "Unexpected error in opening debugger output file %s" % \
                  opts.errors
            print sys.exc_info()[0]
            sys.exit(2)
    return opts

class Pdb(Gdb):

    """A debugger class for Python that resembles the gdb (GNU
    debugger) command set. Really we just inherit everything from the
    Gdb class, but we keep the name Pdb to be compatible with the stock
    Python debugger's (pdb's) Pdb class."""

    def __init__(self, completekey='tab', stdin=None, stdout=None):
        Gdb.__init__(self, completekey, stdin, stdout)

#########################################################
# Alternative top-level calls
#########################################################

def run(statement, globals=None, locals=None):

    """Execute the statement (given as a string) under debugger
    control starting with the statement subsequent to the place that
    this call appears in your program.

    The debugger prompt appears before any code is executed;
    you can set breakpoints and type 'continue', or you can step
    through the statement using 'step' or 'next'

    The optional globals and locals arguments specify the environment
    in which the code is executed; by default the dictionary of the
    module __main__ is used."""

    p = Pdb()
    p.running=True
    p.run(statement, globals, locals)

def runeval(expression, globals=None, locals=None):

    """Evaluate the expression (given as a string) under debugger
    control starting with the statement subsequent to the place that
    this appears in your program.

    When runeval() returns, it returns the value of the expression.
    Otherwise this function is similar to run()."""

    p = Pdb()
    p.running=True
    return p.runeval(expression, globals, locals)

def runcall(*args, **kwds):

    """ Call the function (a function or method object, not a string)
    with the given arguments starting with the statement subsequent to
    the place that this appears in your program.

    When runcall() returns, it returns whatever the function call
    returned.  The debugger prompt appears as soon as the function is
    entered."""

    p = Pdb()
    p.running=True
    return p.runcall(*args, **kwds)

def set_trace(dbg_cmds=None, add_exception_hook=True):
    """Enter the debugger at the calling stack frame.  This is useful to
    hard-code a breakpoint at a given point in a program, even if the code
    is not otherwise being debugged (e.g., when an assertion fails).
    Leaving this the debugger terminates the program.

    dbg_cmds is an optional list of debugger commands you want to run.

    If add_exception is True (the default), we will install an exception hook
    to call the post-mortem debugger on an unhandled exception.
    """
    global _pydb_trace
    try:
        if not isinstance(_pydb_trace, Pdb):
            print "Your program should not use _pydb_trace"
            return
    except NameError:
        _pydb_trace = Pdb()
        _pydb_trace.reset()
        _pydb_trace._program_sys_argv = list(sys.argv)
        _pydb_trace._sys_argv = list(_pydb_trace._program_sys_argv)
        _pydb_trace._sys_argv[:0] = [_debugger_name]
        _pydb_trace.main_dirname = os.path.dirname(sys.argv[0])
    except:
        print "Unknown error"
        return
    # We don't support "run" so we'll make "run" and "R" be "restart"
    _pydb_trace.do_R = _pydb_trace.do_run = _pydb_trace.do_restart
    _pydb_trace.curframe=inspect.currentframe()
    _pydb_trace.running = True

    if dbg_cmds != None:
        _pydb_trace.cmdqueue = list(dbg_cmds)

    if add_exception_hook:
        sys.excepthook = exception_hook

    # There can be no other commands after the following one:
    if sys.version_info[0] == 2 and sys.version_info[1] >= 4:
        _pydb_trace.set_trace(_pydb_trace.curframe)
    else:
        # older versions
        _pydb_trace.set_trace()

# Post-Mortem interface

def post_mortem(t=None, dbg_cmds=None, cmdfile=None, frameno=1):
    """Enter debugger read loop after your program has crashed.

    If no traceback parameter, t, is supplied, the last traceback and
    if that doesn't exist either we'll assume that sys.exec_info()
    contains what we want and frameno is the index location of where
    we want to start.

    dbg_cmds is a list of debugger commands you want to run.

    cmdfile is an optional debugger command file you want to run
    "source" on.

    frameno specifies how many frames to ignore in the traceback.  The
    default is 1 - we don't need the call to post_mortem. If you have
    wrapper functions that call this one, you may want to increase
    frameno.
    """

    p = Pdb()
    p.reset()
    p.running = False
    re_bogus_file = re.compile("^<.+>$")

    if t is None:
        # frameno+1 because we are about to add one more level of call
        # in get_last_tb_or_frame_tb
        t = get_last_tb_or_frame_tb(frameno+1)

    # t has least-recent traceback entry first. We want the most-recent
    # entry. Also we'll pick out a mainpyfile name if it hasn't previously
    # been set.
    while t.tb_next is not None:
        filename = t.tb_frame.f_code.co_filename
        if 0 == len(p.mainpyfile) and not re_bogus_file.match(filename):
            p.mainpyfile = filename
        t = t.tb_next
    p.curframe = t.tb_frame

    if dbg_cmds != None:
        p.cmdqueue = list(dbg_cmds)

    if 0 == len(p._program_sys_argv):
        # Fake program (run command) args since we weren't called with any
        p._program_sys_argv = list(sys.argv[1:])
        p._program_sys_argv[:0] = [p.mainpyfile]

    if 0 == len(p._sys_argv):
        # Fake script invocation (restart) args since we don don't have any
        p._sys_argv = list(p._program_sys_argv)
        p._sys_argv[:0] = [_debugger_name]

    if cmdfile is not None:
        p.do_source(cmdfile)

    try:

        # FIXME: This can be called from except hook in which case we
        # need this. Dunno why though.
        try:
            _pydb_trace.set_trace(t.tb_frame)
        except:
            pass

        p.interaction(t.tb_frame, t)
    except Restart:
        while True:
            sys.argv = list(p._program_sys_argv)
            p.msg("Restarting %s with arguments:\n\t%s"
                  % (p.filename(p.mainpyfile),
                     " ".join(p._program_sys_argv[1:])))
            try:
                p._runscript(p.mainpyfile)
                if p._user_requested_quit:
                    break
                if p.noninteractive: break
            except Restart:
                pass

def pm(debg_cmds=None, frameno=1):

    """Set up post-mortem debugging using the last traceback.  But if
    there is no traceback, we'll assume that sys.exec_info() contains
    what we want and frameno is the index location of where we want
    to start.

    dbg_cmds is an optional list of debugger commands you want to run.
    """

    # frameno+1 because we are about to add one more level of call
    # in get_last_tb_or_frame_tb
    tb = get_last_tb_or_frame_tb(frameno+1)
    post_mortem(tb, dbg_cmds)

def exception_hook(type, value, tb, opts=None, cmdfile=None):
    """An exception hook to call pydb's post-mortem debugger.

    cmdfile is an optional debugger command file you want to run
    "source" on.

    Opts is an optional dictionary of debugger variables you want set;
    the key is the variable name and the value is the value to give
    the variable in the pydb instance.

    To use add this to your Python program
       import sys
       sys.excepthook = pydb.exception_hook
    """
    traceback.print_exception(type, value, tb)
    post_mortem(tb, opts, cmdfile, 0)


# print help
def help():
    doc_file='%s.doc' % _debugger_name
    for dirname in sys.path:
        fullname = os.path.join(dirname, _debugger_name, doc_file)
        if os.path.exists(fullname):
            sts = os.system('${PAGER-more} '+fullname)
            if sts: print 'Pager exit status: %s' % str(sts)
            break
    else:
        print ("Sorry, can't find the help file '%s' along the "
                  + "Python search path") % doc_file

#########################################################
# Main program
#########################################################

def main():
    """Routine which gets run if we were invoked directly"""

    pydb = Pdb()
    process_options(pydb, _debugger_name, program, '1.17')

    # process_options has munged sys.argv to remove any options that
    # options that belong to this debugger. The original options to
    # invoke the debugger and script are in global sys_argv

    if len(sys.argv) == 0:
        # No program given to debug. Set to go into a command loop
        # anyway
        mainpyfile = None

    else:
        # Save the DEBUGGED programs arguments. This is in case
        # the debugged script munges these, we have a good copy to use
        # for restart
        pydb._program_sys_argv = list(sys.argv)

        mainpyfile = pydb._program_sys_argv[0] # Get script filename.
        if not os.path.exists(mainpyfile):
            print "pydb: Python script file '%s' does not exist" \
                  % mainpyfile
            sys.exit(1)

        # Replace pydb's dir with script's dir in front of
        # module search path.
        sys.path[0] = pydb.main_dirname = os.path.dirname(mainpyfile)

    while True:

        # Run the debugged script over and over again until we get it
        # right.

        try:
            if mainpyfile:
                pydb._runscript(mainpyfile)
            else:
                pydb._wait_for_mainpyfile = True
                pydb.interaction(None, None)

            if pydb._user_requested_quit: break
            if pydb.noninteractive: break
            pydb.msg("The program finished and will be restarted")
        except Restart:
            if pydb._program_sys_argv:
                sys.argv = list(pydb._program_sys_argv)
                pydb.msg("Restarting %s with arguments:\n\t%s"
                         % (pydb.filename(mainpyfile),
                            " ".join(pydb._program_sys_argv[1:])))
            else: break
        except SystemExit:
            # In most cases SystemExit does not warrant a post-mortem session.
            pydb.msg("The program exited via sys.exit(). Exit status: %s" %
                     str(sys.exc_info()[1]))
            pydb.running = False;
            if pydb.noninteractive: break
        except:
            traceback.print_exc()
            if pydb.noninteractive:
                pydb.errmsg("Uncaught exception.")
                break
            pydb.errmsg("Uncaught exception. Entering post mortem debugging")
            pydb.errmsg( "Running 'c' or 'step' will restart the program")
            pydb.running = False;
            t = sys.exc_info()[2]
            while t.tb_next is not None:
                t = t.tb_next
            try:
                pydb.interaction(t.tb_frame,t)
            except Restart:
                sys.argv = list(pydb._program_sys_argv)
                pydb.msg("Restarting %s with arguments:\n\t%s"
                         % (pydb.filename(mainpyfile),
                            " ".join(pydb._program_sys_argv[1:])))
            else:
                pydb.msg("Post mortem debugger finished.")
                pydb.msg(mainpyfile + " will be restarted")
        pydb.step_ignore = 0

init()

# When invoked as main program, invoke the debugger on a script
if __name__=='__main__':
    main()

#
# Local variables:
#  mode: Python
# End:
