#!/usr/bin/env python
# arch-tag: a6caa465-e8f8-4cb9-a702-bec62c1ec2dc
# Copyright (C) 2004 David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test suite for NameParser."""

import sys

import pybaz as arch

import framework
import fixtures


# Should NameParser be tested for consistency with ForkNameParser?
use_forknameparser = False


class NameComponents(framework.NewTestCase):
    tests = []

    def is_archive_name(self):
        """NameParser.is_archive_name works."""
        is_archive_name = arch.NameParser.is_archive_name
        self.failUnless(is_archive_name('arch@arch.thinkmo.de--2003-archives'))
        self.failIf(is_archive_name('archo--dummy'))
        self.failUnless(is_archive_name('arch@arch.org--archie--dada--dudu'))
        self.failUnless(is_archive_name('1@'))
    tests.append('is_archive_name')

    def _is_category_or_branch_name(self, predicate):
        self.failUnless(predicate('moin'))
        self.failUnless(predicate('test-test'))
        self.failIf(predicate(''))
        self.failIf(predicate('test--test'))
        self.failIf(predicate('test-'))
        self.failIf(predicate('-test'))
        self.failIf(predicate('10'))
        self.failIf(predicate('emacs+junk'))

    def is_category_name(self):
        """NameParser.is_category_name works."""
        is_category_name = arch.NameParser.is_category_name
        self._is_category_or_branch_name(is_category_name)
    tests.append('is_category_name')

    def is_branch_name(self):
        """NameParser.is_branch_name works."""
        is_branch_name = arch.NameParser.is_branch_name
        self._is_category_or_branch_name(is_branch_name)
    tests.append('is_branch_name')

    def is_version_id(self):
        """NameParser.is_version_id works."""
        is_version_id = arch.NameParser.is_version_id
        self.failIf(is_version_id(''))
        self.failUnless(is_version_id('0'))
        self.failUnless(is_version_id('1.1'))
        self.failUnless(is_version_id('1.2.3.4.5.6.7'))
        self.failUnless(is_version_id('1.1a'))
        self.failUnless(is_version_id('1.1-3'))
        self.failIf(is_version_id('.0'))
        self.failIf(is_version_id('0*'))
        self.failIf(is_version_id('a'))
        self.failIf(is_version_id('a.b.c.d'))
    tests.append('is_version_id')

# levels = ['patch-1321', 'base-0', 'version-0', 'versionfix-209', 'vsn-12',
#           'a', 'version-1', 'version-1.2.3',]

    def is_patchlevel(self):
        """NameParser.is_patchlevel works."""
        is_patchlevel = arch.NameParser.is_patchlevel
        self.failIf(is_patchlevel(''))
        self.failIf(is_patchlevel('0'))
        self.failIf(is_patchlevel('base-'))
        self.failUnless(is_patchlevel('base-0'))
        self.failIf(is_patchlevel('base-1'))
        self.failIf(is_patchlevel('patch-'))
        self.failUnless(is_patchlevel('patch-1'))
        self.failUnless(is_patchlevel('patch-120'))
        self.failIf(is_patchlevel('version-'))
        self.failUnless(is_patchlevel('version-0'))
        self.failIf(is_patchlevel('versionfix-'))
        self.failUnless(is_patchlevel('versionfix-1'))
        self.failUnless(is_patchlevel('versionfix-99'))
        self.failIf(is_patchlevel('base-0.0'))
        self.failIf(is_patchlevel('patch-1.2'))
        self.failIf(is_patchlevel('version-1.'))
        self.failIf(is_patchlevel('versionfix-1-0'))
    tests.append('is_patchlevel')

    def is_patchlevel_bugs(self):
        """NameParser.is_patchlevel accepts some impossible values."""
        is_patchlevel = arch.NameParser.is_patchlevel
        self.failUnless(is_patchlevel('patch-0'))
        self.failUnless(is_patchlevel('version-1'))
        self.failUnless(is_patchlevel('versionfix-0'))
    tests.append('is_patchlevel_bugs')


class DefaultArchiveFixture(fixtures.EmptySandboxFixture):
    """A fixture that sets my-default-archive in a sandbox."""

    def __init__(self):
        super(DefaultArchiveFixture, self).__init__()
        self.default_archive = "default@archive"

    def setUp(self):
        """Setup and sandbox and sets my-id."""
        super(DefaultArchiveFixture, self).setUp()
        import pybaz._builtin
        pybaz._builtin._arch.set_default_archive(self.default_archive)


class InvalidNames(framework.NewTestCase):
    fixture = DefaultArchiveFixture()
    tests = []

    def _invalid(self, name_str):
        if use_forknameparser:
            self._invalid_helper(arch._nameparser.ForkNameParser(name_str))
        self._invalid_helper(arch.NameParser(name_str))

    def _invalid_helper(self, name):
        self.failIf(name.has_archive())
        self.assertEqual(name.get_archive(), None)
        self.assertEqual(name.get_nonarch(), None)
        self.failIf(name.has_category())
        self.assertEqual(name.get_category(), None)
        self.failIf(name.is_category())
        self.assertEqual(name.get_branch(), None)
        self.failIf(name.has_package())
        self.assertEqual(name.get_package(), None)
        self.failIf(name.is_package())
        self.failIf(name.has_version())
        self.assertEqual(name.get_version(), None)
        self.assertEqual(name.get_package_version(), None)
        self.failIf(name.is_version())
        self.failIf(name.has_patchlevel())
        self.assertEqual(name.get_patchlevel(), None)

    def invalid_category(self):
        """invalid category names are handled properly."""
        self._invalid('good@archive')
        self._invalid('good@archive/')
        self._invalid('bad-archive/category')
        self._invalid('bad.@archive/category')
        self._invalid('-bad@archive/category')
        self._invalid('good@archive/bad+category')
        self._invalid('bad+category')
        self._invalid('/category')
        self._invalid('1-cat')
    tests.append('invalid_category')

    def invalid_branch(self):
        """invalid branch names are handled properly."""
        self._invalid('bad-archive/category--branch')
        self._invalid('good@archive/bad+category--branch')
        self._invalid('good@archive/category--bad+branch')
        self._invalid('good@archive/category---branch')
        self._invalid('/category--branch')
        self._invalid('bad+category--branch')
        self._invalid('category--bad+branch')
        self._invalid('category---branch')
        self._invalid('--branch')
    tests.append('invalid_branch')

    def invalid_version(self):
        """invalid version names are handled properly."""
        # all components
        self._invalid('bad-archive/category--branch--0')
        self._invalid('good@archive/bad+category--branch--0')
        self._invalid('good@archive/category--bad+branch--0')
        self._invalid('good@archive/category---branch--0')
        self._invalid('good@archive/category--branch---0')
        self._invalid('good@archive/category--branch--0--')
        self._invalid('/category--branch--0')
        self._invalid('good@archive/--branch--0')
        self._invalid('good@archive/category----0')
        # no branch
        self._invalid('bad-archive/category--0')
        self._invalid('good@archive/bad+category--0')
        self._invalid('good@archive/category---0')
        self._invalid('good@archive/categor--0--')
        self._invalid('/category--0')
        self._invalid('good@archive/0')
        self._invalid('good@archive/1.0')
        # no archive
        self._invalid('bad+category--branch--0')
        self._invalid('category--bad+branch--0')
        self._invalid('--branch--0')
        self._invalid('category----0')
        self._invalid('category---0')
        self._invalid('category--0--')
        # no archive and no branch
        self._invalid('bad+category--0')
        self._invalid('--0')
        self._invalid('category---0')
        self._invalid('category--0--')
        # just version
        self._invalid('0')
        self._invalid('1.0')
    tests.append('invalid_version')

    def invalid_revision(self):
        """invalid revision names are handled properly."""
        # all components
        self._invalid('bad-archive/category--branch--0--patch-1')
        self._invalid('good@archive/bad+category--branch--0--patch-1')
        self._invalid('good@archive/category--bad+branch--0--patch-1')
        self._invalid('good@archive/category--branch--0^bad--patch-1')
        self._invalid('good@archive/category--branch--0--bad-1')
        self._invalid('good@archive/category--branch--0--base-1.0')
        self._invalid('good@archive/category--branch--0--patch-1.0')
        self._invalid('good@archive/category--branch--0--version-1.0')
        self._invalid('good@archive/category--branch--0--versionfix-1.0')
        self._invalid('good@archive/category---branch--0--patch-1')
        self._invalid('good@archive/category--branch---0--patch-1')
        self._invalid('good@archive/category--branch--0---patch-1')
        self._invalid('/category--branch--0--patch-1')
        self._invalid('good@archive/--branch--0--patch-1')
        self._invalid('good@archive/category----0--patch-1')
        self._invalid('good@archive/category--branch----patch-1')
        self._invalid('good@archive/category--branch--0--')
        # no branch
        self._invalid('bad-archive/category--0--patch-1')
        self._invalid('good@archive/bad+category--0--patch-1')
        self._invalid('good@archive/category--0--bad-1')
        self._invalid('good@archive/category---0--patch-1')
        self._invalid('good@archive/category--0---patch-1')
        self._invalid('/category--0--patch-1')
        self._invalid('good@archive/--0--patch-1')
        self._invalid('good@archive/category----patch-1')
        self._invalid('good@archive/category--0--')
        # no archive
        self._invalid('bad+category--branch--0--patch-1')
        self._invalid('category--bad+branch--0--patch-1')
        self._invalid('category--branch--0--bad-1')
        self._invalid('category---branch--0--patch-1')
        self._invalid('category--branch---0--patch-1')
        self._invalid('category--branch--0---patch-1')
        self._invalid('--branch--0--patch-1')
        self._invalid('category----0--patch-1')
        self._invalid('category--branch----patch-1')
        self._invalid('category--branch--0--')
        # no archive and no branch
        self._invalid('bad+category--0--patch-1')
        self._invalid('category--0--bad-1')
        self._invalid('category---0--patch-1')
        self._invalid('category--0---patch-1')
        self._invalid('--0--patch-1')
        self._invalid('category----patch-1')
        self._invalid('category--0--')
        # patchlevel-only names are valid category names
    tests.append('invalid_revision')


class CategoryNames(framework.NewTestCase):
    fixture = DefaultArchiveFixture()
    tests = []

    def _category(self, name_str, *args):
        if use_forknameparser:
            forknameparser = arch._nameparser.ForkNameParser(name_str)
            self._category_helper(forknameparser, *args)
        self._category_helper(arch.NameParser(name_str), *args)

    def _category_helper(self, name, archive, category, nonarch=None):
        if nonarch is None:
            nonarch = category
        if archive is None:
            self.failIf(name.has_archive())
            self.assertEqual(name.get_archive(), self.fixture.default_archive)
        else:
            self.failUnless(name.has_archive())
            self.assertEqual(name.get_archive(), archive)
        self.assertEqual(name.get_nonarch(), nonarch)
        self.failUnless(name.has_category())
        self.assertEqual(name.get_category(), category)
        self.failUnless(name.is_category())
        self.assertEqual(name.get_branch(), '')
        self.failUnless(name.has_package())
        self.assertEqual(name.get_package(), category)
        self.failUnless(name.is_package())
        self.failIf(name.has_version())
        self.assertEqual(name.get_version(), None)
        self.assertEqual(name.get_package_version(), None)
        self.failIf(name.is_version())
        self.failIf(name.has_patchlevel())
        self.assertEqual(name.get_patchlevel(), None)

    def category(self):
        """NameParser handles categories correctly."""
        self._category('good@archive/category', 'good@archive', 'category')
        self._category('1@/category', '1@', 'category')
        self._category('category', None, 'category')
        self._category('1@/cat-1', '1@', 'cat-1')
        self._category('1@/cat-1-egory', '1@', 'cat-1-egory')
    tests.append('category')

    def category_bugs(self):
        """NameParser handles some invalid categories like tla does."""
        self._category('good@archive/category--', 'good@archive', 'category',
                       'category--')
        self._category('category--', None, 'category', 'category--')
        self._category('good@archive/category-', 'good@archive', 'category-')
        self._category('category-', None, 'category-')
    tests.append('category_bugs')


class BranchNames(framework.NewTestCase):
    fixture = DefaultArchiveFixture()
    tests = []

    def _branch(self, name_str, *args):
        if use_forknameparser:
            forknameparser = arch._nameparser.ForkNameParser(name_str)
            self._branch_helper(forknameparser, *args)
        self._branch_helper(arch.NameParser(name_str), *args)

    def _branch_helper(self, name, archive, category, branch, nonarch=None):
        if nonarch is None:
            nonarch = category + '--' + branch
        if archive is None:
            self.failIf(name.has_archive())
            self.assertEqual(name.get_archive(), self.fixture.default_archive)
        else:
            self.failUnless(name.has_archive())
            self.assertEqual(name.get_archive(), archive)
        self.assertEqual(name.get_nonarch(), nonarch)
        self.failUnless(name.has_category())
        self.assertEqual(name.get_category(), category)
        self.failIf(name.is_category())
        self.assertEqual(name.get_branch(), branch)
        self.failUnless(name.has_package())
        self.assertEqual(name.get_package(), category + '--' + branch)
        self.failUnless(name.is_package())
        self.failIf(name.has_version())
        self.assertEqual(name.get_version(), None)
        self.assertEqual(name.get_package_version(), None)
        self.failIf(name.is_version())
        self.failIf(name.has_patchlevel())
        self.assertEqual(name.get_patchlevel(), None)

    def branch(self):
        """NameParser handles branches correctly."""
        A, C, B = 'good@archive', 'category', 'branch'
        self._branch('good@archive/category--branch', A, C, B)
        self._branch('1@/cat-1--br-2', '1@', 'cat-1', 'br-2')
        self._branch('good@archive/cat--br-2-anch', A, 'cat', 'br-2-anch')
        self._branch('category--branch', None, C, B)
        self._branch('cat-1--br-2', None, 'cat-1', 'br-2')
        self._branch('cat--br-2-anch', None, 'cat', 'br-2-anch')
    tests.append('branch')

    def branch_bugs(self):
        """NameParser handles some invalid branches like tla does."""
        A, C, B = 'good@archive', 'category', 'branch'
        self._branch('good@archive/category--branch--', A, C, B,
                       'category--branch--')
        self._branch('category--branch--', None, C, B,
                       'category--branch--')
        self._branch('good@archive/category--branch-', A, C, 'branch-')
        self._branch('category--branch-', None, C, 'branch-')
    tests.append('branch_bugs')


class VersionNames(framework.NewTestCase):
    fixture = DefaultArchiveFixture()
    tests = []

    def _version(self, name_str, *args):
        if use_forknameparser:
            forknameparser = arch._nameparser.ForkNameParser(name_str)
            self._version_helper(forknameparser, *args)
        self._version_helper(arch.NameParser(name_str), *args)

    def _version_helper(self, name, archive, category, branch, version):
        if branch == '':
            nonarch = category + '--' + version
            package = category
        else:
            nonarch = category + '--' + branch + '--' + version
            package = category + '--' + branch
        if archive is None:
            self.failIf(name.has_archive())
            self.assertEqual(name.get_archive(), self.fixture.default_archive)
        else:
            self.failUnless(name.has_archive())
            self.assertEqual(name.get_archive(), archive)
        self.assertEqual(name.get_nonarch(), nonarch)
        self.failUnless(name.has_category())
        self.assertEqual(name.get_category(), category)
        self.failIf(name.is_category())
        self.assertEqual(name.get_branch(), branch)
        self.failUnless(name.has_package())
        self.assertEqual(name.get_package(), package)
        self.failIf(name.is_package())
        self.failUnless(name.has_version())
        self.assertEqual(name.get_version(), version)
        self.assertEqual(name.get_package_version(), package + '--' + version)
        self.failUnless(name.is_version())
        self.failIf(name.has_patchlevel())
        self.assertEqual(name.get_patchlevel(), None)

    def version(self):
        """NameParser handles versions correctly."""
        A, C, B, V = 'good@arch', 'cat', 'brn', '0'
        self._version('good@arch/cat--brn--0', A, C, B, V)
        self._version('good@arch/cat--brn--99.99', A, C, B, '99.99')
        self._version('good@arch/cat--0', A, C, '', V)
        self._version('good@arch/cat--99.99', A, C, '', '99.99')
        self._version('cat--brn--0', None, C, B, V)
        self._version('cat--brn--99.99', None, C, B, '99.99')
        self._version('cat--0', None, C, '', V)
        self._version('cat--99.99', None, C, '', '99.99')
    tests.append('version')

    def version_bugs(self):
        """NameParser handles version with leading zeros."""
        A, C, B = 'good@arch', 'cat', 'brn'
        self._version('good@arch/cat--brn--00', A, C, B, '00')
        self._version('good@arch/cat--brn--01', A, C, B, '01')
        self._version('good@arch/cat--brn--01.002', A, C, B, '01.002')
    tests.append('version_bugs')


class RevisionNames(framework.NewTestCase):
    fixture = DefaultArchiveFixture()
    tests = []

    def _revision(self, name_str, *args):
        if use_forknameparser:
            forknameparser = arch._nameparser.ForkNameParser(name_str)
            self._revision_helper(forknameparser, *args)
        self._revision_helper(arch.NameParser(name_str), *args)

    def _revision_helper(self, name, archive, category, branch, version, level):
        if branch == '':
            nonarch = category + '--' + version + '--' + level
            package = category
        else:
            nonarch = category + '--' + branch + '--' + version + '--' + level
            package = category + '--' + branch
        if archive is None:
            self.failIf(name.has_archive())
            self.assertEqual(name.get_archive(), self.fixture.default_archive)
        else:
            self.failUnless(name.has_archive())
            self.assertEqual(name.get_archive(), archive)
        self.assertEqual(name.get_nonarch(), nonarch)
        self.failUnless(name.has_category())
        self.assertEqual(name.get_category(), category)
        self.failIf(name.is_category())
        self.assertEqual(name.get_branch(), branch)
        self.failUnless(name.has_package())
        self.assertEqual(name.get_package(), package)
        self.failIf(name.is_package())
        self.failUnless(name.has_version())
        self.assertEqual(name.get_version(), version)
        self.assertEqual(name.get_package_version(), package + '--' + version)
        self.failIf(name.is_version())
        self.failUnless(name.has_patchlevel())
        self.assertEqual(name.get_patchlevel(), level)

    def revision(self):
        """NameParser handles revisions correctly."""
        A, C, B, V = 'good@arch', 'cat', 'brn', '0'
        ACBV = 'good@arch/cat--brn--0'
        self._revision(ACBV + '--base-0', A, C, B, V, 'base-0')
        self._revision(ACBV + '--patch-1', A, C, B, V, 'patch-1')
        self._revision(ACBV + '--patch-99', A, C, B, V, 'patch-99')
        self._revision(ACBV + '--version-0', A, C, B, V, 'version-0')
        self._revision(ACBV + '--versionfix-1', A, C, B, V, 'versionfix-1')
        self._revision(ACBV + '--versionfix-99', A, C, B, V, 'versionfix-99')
        CBV = 'cat--brn--0'
        self._revision(CBV + '--base-0', None, C, B, V, 'base-0')
        self._revision(CBV + '--patch-1', None, C, B, V, 'patch-1')
        self._revision(CBV + '--patch-99', None, C, B, V, 'patch-99')
        self._revision(CBV + '--version-0', None, C, B, V, 'version-0')
        self._revision(CBV + '--versionfix-1', None, C, B, V, 'versionfix-1')
        self._revision(CBV + '--versionfix-99', None, C, B, V, 'versionfix-99')
        ACV = 'good@arch/cat--0'
        self._revision(ACV + '--base-0', A, C, '', V, 'base-0')
        self._revision(ACV + '--patch-1', A, C, '', V, 'patch-1')
        self._revision(ACV + '--patch-99', A, C, '', V, 'patch-99')
        self._revision(ACV + '--version-0', A, C, '', V, 'version-0')
        self._revision(ACV + '--versionfix-1', A, C, '', V, 'versionfix-1')
        self._revision(ACV + '--versionfix-99', A, C, '', V, 'versionfix-99')
        CV = 'cat--0'
        self._revision(CV + '--base-0', None, C, '', V, 'base-0')
        self._revision(CV + '--patch-1', None, C, '', V, 'patch-1')
        self._revision(CV + '--patch-99', None, C, '', V, 'patch-99')
        self._revision(CV + '--version-0', None, C, '', V, 'version-0')
        self._revision(CV + '--versionfix-1', None, C, '', V, 'versionfix-1')
        self._revision(CV + '--versionfix-99', None, C, '', V, 'versionfix-99')
    tests.append('revision')

    def revision_bugs(self):
        """NameParser handles some impossible revisions."""
        A, C, B, V = 'good@arch', 'cat', 'brn', '0'
        ACBV = 'good@arch/cat--brn--0'
        self._revision(ACBV + '--patch-01', A, C, B, V, 'patch-01')
        self._revision(ACBV + '--version-1', A, C, B, V, 'version-1')
        self._revision(ACBV + '--version-01', A, C, B, V, 'version-01')
        self._revision(ACBV + '--version-99', A, C, B, V, 'version-99')
        self._revision(ACBV + '--versionfix-01', A, C, B, V, 'versionfix-01')
    tests.append('revision_bugs')


class NameParserObject(framework.NewTestCase):
    fixture = DefaultArchiveFixture()
    tests = []

    def _helper(self, name, expected):
        obj = arch.NameParser(name).object()
        self.assertEqual(type(obj), type(expected))
        if expected is not None:
            self.assertEqual(obj.fullname, expected.fullname)
        self.assertEqual(obj, expected)

    def empty(self):
        """NameParser.object fails on empty string."""
        self._helper('', None)
    tests.append('empty')

    def archive(self):
        """NameParser.object cannot create Archive."""
        self._helper('good@archive', None)
    tests.append('archive')

    def category(self):
        """NameParser.object can create Category."""
        self._helper('good@archive/cat', arch.Category('good@archive/cat'))
        self._helper('good@archive/cat--', arch.Category('good@archive/cat'))
        self._helper('good@archive/cat-', None)
        default = self.fixture.default_archive
        self._helper('category', arch.Category(default + '/category'))
        self._helper('category--', arch.Category(default + '/category'))
        self._helper('category-', None)
    tests.append('category')

    def branch(self):
        """NameParser.object can create Branch."""
        self._helper('good@arch/cat--brn', arch.Branch('good@arch/cat--brn'))
        self._helper('good@arch/cat--brn--', arch.Branch('good@arch/cat--brn'))
        self._helper('good@arch/cat--brn-', None)
        default = self.fixture.default_archive
        self._helper('cat--brn', arch.Branch(default + '/cat--brn'))
        self._helper('cat--brn--', arch.Branch(default + '/cat--brn'))
        self._helper('cat--brn-', None)
    tests.append('branch')

    def version(self):
        """NameParser.object can create Version."""
        self._helper('good@/cat--brn--0', arch.Version('good@/cat--brn--0'))
        self._helper('good@/cat--0', arch.Version('good@/cat--0'))
        default = self.fixture.default_archive
        self._helper('cat--brn--0', arch.Version(default + '/cat--brn--0'))
        self._helper('cat--0', arch.Version(default + '/cat--0'))
    tests.append('version')

    def revision(self):
        """NameParser.object can create Revision."""
        self._helper('good@arch/category--branch--0--patch-42',
                     arch.Revision('good@arch/category--branch--0--patch-42'))
        self._helper('good@arch/cat--0--patch-42',
                     arch.Revision('good@arch/cat--0--patch-42'))
        default = self.fixture.default_archive
        self._helper('cat--brn--0--patch-42',
                     arch.Revision(default + '/cat--brn--0--patch-42'))
        self._helper('cat--0--patch-42',
                     arch.Revision(default + '/cat--0--patch-42'))
    tests.append('revision')


framework.register(__name__)
